/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.storage;

import java.io.IOException;
import java.io.InputStream;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.dataobject.EncryptedObject;

/**
 * Interface handler for the file storage.
 *
 * @author benjamin.otterbach@open-xchange.com
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
/*
 * TODO: Switch to OXException once we refactored the exception handling from the
 * invocations from this implementations
 */
public interface Storage {

    /**
     * The delimiter character to separate the prefix from the keys
     */
    public static final String DELIMITER = "/";

    /**
     * Read encrypted object from file
     *
     * @param userid User Id
     * @param contextid Users context id
     * @param ObjId Unique object id
     * @return
     * @throws OXException
     */
    InputStream readObj(int userId, int contextId, String objectId) throws OXException;

    EncryptedObject readEncrObjOld(int userId, int contextId, String objectId) throws OXException;

    /**
     * Write encrypted object to file
     *
     * @param userid User Id
     * @param contextid Users context id
     * @param ObjId Unique object id
     * @param data byte array of data to write
     * @return
     * @throws OXException
     */
    Boolean saveEncrObj(int userId, int contextId, String objectId, byte[] data) throws OXException;

    void deleteEncrObj(String location) throws OXException;

    /**
     * Get directory prefix for the user account
     *
     * @param userid
     * @param contextid
     * @param shardid
     * @return
     */
    String getDirectoryPrefix(int userId, int contextId, int shardId) throws OXException;

    /**
     * This is to get directory prefix for guard proprietary, which had an error. Remove in near future
     *
     * @param userid
     * @param contextid
     * @param shardid
     * @return
     */
    String getDirectoryPrefix2(int userId, int contextId, int shardId) throws OXException;

    InputStream getObjectStream(String location) throws OXException;

    InputStream getObjectStream(String directoryPrefix, String ObjId) throws OXException;

    /**
     * Delete a file from the cache store. Usually used by the Cache Cleaner.
     *
     * @param location File location including directory prefix. Example: ext_1_25/0ff183ed-097c-4a5a-b534-0198659c775a
     * @throws IOException
     * @throws OXException if the GuardConfigurationService is absent
     */
    void deleteObj(String location) throws OXException;

    /**
     * Get an object from the cache store.
     *
     * @param directoryPrefix Sub-directory of the files storage location, for example: ext_1_25
     * @param ObjId ID of the file
     * @return EncryptedObj
     * @throws OXException
     */
    EncryptedObject getEncrObj(String directoryPrefix, String ObjId) throws OXException;

    /**
     * Write byte array to filename
     *
     * @param filename Full filename of object
     * @param data byte array of data
     * @return
     * @throws IOException
     * @throws OXException if the GuardConfigurationService is absent
     */
    Boolean writeObj(String directoryPrefix, String ObjId, byte[] data) throws OXException;
}
