/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.storage;

import java.io.InputStream;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.dataobject.EncryptedObject;
import com.openexchange.guard.database.GuardShardingService;
import com.openexchange.guard.storage.cache.FileCacheItem;
import com.openexchange.guard.storage.cache.FileCacheStorage;
import com.openexchange.guard.storage.exceptions.StorageExceptionCodes;
import com.openexchange.guard.storage.osgi.Services;

/**
 * {@link AbstractStorage}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public abstract class AbstractStorage implements Storage {

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.storage.Storage#readObj(int, int, java.lang.String)
     */
    @Override
    public InputStream readObj(int userId, int contextId, String objectId) throws OXException {
        filterObjId(objectId);
        // First try to pull from cache
        FileCacheStorage storage = Services.getService(FileCacheStorage.class);
        FileCacheItem item = storage.getById(objectId);
        if (item != null) {
            return getObjectStream(item.getLocation());
        }
        // default search, but probably removed
        GuardShardingService sharding = Services.getService(GuardShardingService.class);
        return getObjectStream(getDirectoryPrefix(userId, contextId, sharding.getShard(userId, contextId)), objectId);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.storage.Storage#readEncrObjOld(int, int, java.lang.String)
     */
    @Override
    public EncryptedObject readEncrObjOld(int userId, int contextId, String objectId) throws OXException {
        filterObjId(objectId);
        GuardShardingService sharding = Services.getService(GuardShardingService.class);
        return getEncrObj(getDirectoryPrefix2(userId, contextId, sharding.getShard(userId, contextId)), objectId);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.storage.Storage#saveEncrObj(int, int, java.lang.String, byte[])
     */
    @Override
    public Boolean saveEncrObj(int userId, int contextId, String objectId, byte[] data) throws OXException {
        boolean saveSuccess = false;
        filterObjId(objectId);
        GuardShardingService sharding = Services.getService(GuardShardingService.class);
        String directoryPrefix = getDirectoryPrefix(userId, contextId, sharding.getShard(userId, contextId));

        saveSuccess = writeObj(directoryPrefix, objectId, data);

        if (saveSuccess) {
            FileCacheStorage fileCacheStorage = Services.getService(FileCacheStorage.class);
            //Only inserting the file cache item if it does not exist yet
            //(It could exists when storing an attachment for multiple recipients)
            if (fileCacheStorage.getById(objectId) == null) {
                fileCacheStorage.insert(objectId, userId, directoryPrefix + DELIMITER + objectId);
            }
        }

        return saveSuccess;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.storage.Storage#getDirectoryPrefix(int, int, int)
     */
    @Override
    public String getDirectoryPrefix(int userId, int contextId, int shardId) throws OXException {
        String directory = "";
        if (contextId > 0) {
            directory = "ctx_" + contextId;
        } else {
            directory = "ext_" + shardId + "_" + userId;
        }
        return (directory);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.storage.Storage#getDirectoryPrefix2(int, int, int)
     */
    @Override
    public String getDirectoryPrefix2(int userId, int contextId, int shardId) throws OXException {
        String directory = "";
        if (contextId < 0) {
            directory = "ctx_" + contextId;
        } else {
            directory = "ext_" + shardId + "_" + userId;
        }
        return (directory);
    }

    /**
     * Filter for .. attack
     * 
     * @param objectId
     * @throws OXException
     */
    private void filterObjId(String objectId) throws OXException {
        if (objectId.contains("..")) {
            throw StorageExceptionCodes.BACK_OBJECT_ID.create(objectId);
        }
    }
}
