/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.oxapi;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.HttpClientUtils;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.oxapi.context.CustomHttpClientContext;
import com.openexchange.guard.oxapi.exceptions.OXApiExceptionCodes;
import com.openexchange.guard.oxapi.osgi.Services;
import com.openexchange.guard.oxapi.pooling.HttpConnectionPoolService;

public class Utilities extends AbstractOXCalls {

    private static final Logger LOG = LoggerFactory.getLogger(Utilities.class);

    private JsonObject getOgRequest(String unit, String action, String parameters, GuardConfigurationService guardConfigurationService) throws OXException {
        String url = getMainURI() + "/preliminary/utilities/" + unit + "/v1/" + action + "/" + parameters;
        return getOgRequest(url, guardConfigurationService);
    }

    private JsonObject getOgRequest(String url, GuardConfigurationService guardConfigurationService) throws OXException {
        JsonObject result = null;
        HttpConnectionPoolService poolService = Services.getService(HttpConnectionPoolService.class);
        HttpClient httpClient = poolService.getClient();

        HttpGet getRequest = new HttpGet(url);
        getRequest.addHeader("accept", "application/json");
        getRequest.addHeader("Authorization", "Basic " + getBasicAuthenticationEncoding());
        getRequest.setHeader("User-Agent", guardConfigurationService.getProperty(GuardProperty.apiUserAgent));

        HttpResponse response = null;
        try {
            HttpContext context = CustomHttpClientContext.create();
            response = httpClient.execute(getRequest, context);
            if (response.getStatusLine().getStatusCode() != 200) {
                EntityUtils.consume(response.getEntity());
                getRequest.releaseConnection();

                String resp = response.getStatusLine().getReasonPhrase();
                if (resp != null) {
                    if (resp.contains("Cannot find user")) {
                        result = new JsonObject();
                        result.addProperty("user", "not found");
                        return (result);
                    }
                }
                LOG.error("Error getting user data from backend " + response.getStatusLine().getReasonPhrase());
                throw OXApiExceptionCodes.WRONG_ERROR_CODE.create(response.getStatusLine().getStatusCode());
            }
            result = getJson(response);
            EntityUtils.consume(response.getEntity());
            getRequest.releaseConnection();
        } catch (Exception e2) {
            LOG.error("unable to close http stream after error", e2);
        } finally {
            HttpClientUtils.closeQuietly(response);
        }
        return result;
    }

    /**
     * Format response into a JSON object
     *
     * @param response
     * @return
     * @throws UnsupportedEncodingException
     * @throws IllegalStateException
     * @throws IOException
     */
    private JsonObject getJson(HttpResponse response) throws OXException {
        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"));
            JsonParser parser = new JsonParser();
            JsonObject result = (JsonObject) parser.parse(reader);
            return result;
        } catch (IllegalStateException | IOException e) {
            throw OXApiExceptionCodes.HANDLE_SERVER_RESPONSE_ERROR.create(e);
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                    throw OXApiExceptionCodes.HANDLE_SERVER_RESPONSE_ERROR.create(e);
                }
            }
        }
    }

    public JsonObject getUser(String email) throws OXException {
        GuardConfigurationService guardConfigurationService = Services.getService(GuardConfigurationService.class);

        try {
            email = URLEncoder.encode(email, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw OXApiExceptionCodes.UNEXPECTED_ERROR.create(e);
        }
        // If mailReolverUrl is not defined in the configuration file, use default utilities
        String property = guardConfigurationService.getProperty(GuardProperty.mailResolverUrl);
        if (Strings.isEmpty(property) || property.equals("")) {
            return (getOgRequest("mailResolver", "resolve", email, guardConfigurationService));
        } // Otherwise, use the assigned URL, adding the email address to the end
        else {
            return (getOgRequest(property + email, guardConfigurationService));
        }
    }

    /**
     * Checks for error in reply string
     *
     * @param response
     * @return true if contains error
     */
    public static boolean checkFail(String response) {
        if (response == null) {
            return (true);
        }
        if (response.contains("error")) {
            return (true);
        }
        return (false);
    }

    public static boolean checkFail(JsonObject resp) {
        if (resp == null) {
            return (false);
        }
        Gson gson = new Gson();
        return (checkFail(gson.toJson(resp)));
    }
}
