/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.oxapi;

import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.utils.HttpClientUtils;
import org.apache.http.entity.StringEntity;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.openexchange.guard.oxapi.context.CustomHttpClientContext;
import com.openexchange.guard.oxapi.osgi.Services;
import com.openexchange.guard.oxapi.pooling.HttpConnectionPoolService;
import com.openexchange.guard.oxapi.utils.ResponseHelper;

public class Normalizer extends AbstractOXCalls {

    private static Logger LOG = LoggerFactory.getLogger(Normalizer.class);

    public static String normalize(String data) {
        try {
            JsonParser parser = new JsonParser();
            JsonObject json = (JsonObject) parser.parse(data);
            Gson gson = new Gson();
            return gson.toJson(normalize(json));
        } catch (Exception ex) {
            LOG.error("Error normalizing data", ex);
            return data;
        }
    }

    /**
     * Send string to backend for normalization. Primarily for PGP Inline
     * 
     * @param data
     * @return
     * @throws Exception
     */
    public static String normalizeString(String data) {
        try {
            JsonObject content = new JsonObject();
            content.addProperty("content", data);
            JsonObject newcontent = putRequest(content);
            String emailContent = newcontent.get("content").getAsString();
            return emailContent;
        } catch (Exception e) {
            LOG.error("Problem normalizing string ", e);
            return data;
        }
    }

    /**
     * Send email JSON to OX backend for normalization (strip scripts, etc)
     * 
     * @param json
     * @return
     */
    public static JsonObject normalize(JsonObject json) {
        if (json.has("attachments")) {
            JsonArray attachments = json.get("attachments").getAsJsonArray();
            JsonArray newattach = new JsonArray();
            for (int i = 0; i < attachments.size(); i++) {
                JsonObject attach = attachments.get(i).getAsJsonObject();
                if (attach.has("content")) {
                    if (!attach.get("content").isJsonNull()) {
                        try {
                            String data = attach.get("content").getAsString();
                            JsonObject content = new JsonObject();
                            content.addProperty("content", data);
                            JsonObject newcontent = putRequest(content);
                            String emailContent = newcontent.get("content").getAsString();
                            attach.remove("content");
                            attach.addProperty("content", emailContent);
                        } catch (Exception ex) {
                            LOG.error("Error normalizing data", ex);
                        }
                    }
                    newattach.add(attach);
                }
            }
        }
        return json;
    }

    /**
     * Actual send of data to OX Backend
     *
     * @param data String of JSON data to send
     * @param database The ID of the database
     * @param cid ContextId if pertinent. 0 if not
     * @param write Write or read ony
     * @return
     * @throws Exception
     */
    private static JsonObject putRequest(JsonObject content) throws Exception {
        HttpConnectionPoolService poolService = Services.getService(HttpConnectionPoolService.class);
        HttpClient httpClient = poolService.getClient();
        HttpContext context = CustomHttpClientContext.create();

        String url = getMainURI() + "/preliminary/htmlproc/v1/sanitize";
        HttpPut putRequest = new HttpPut(url);

        String data = content.toString();
        StringEntity entity = new StringEntity(data, "UTF-8");
        entity.setContentType("application/json;charset=UTF-8");
        putRequest.addHeader("accept", "application/json");
        putRequest.addHeader("Authorization", "Basic " + getBasicAuthenticationEncoding());
        putRequest.setHeader("User-Agent", Api.USER_AGENT);
        putRequest.setEntity(entity);
        HttpResponse response = httpClient.execute(putRequest, context);
        JsonObject result = null;
        if (response.getStatusLine().getStatusCode() != 200) {
            try {
                result = ResponseHelper.getJson(response);
            } catch (Exception ex) {
                try {
                    EntityUtils.consume(response.getEntity());
                    putRequest.releaseConnection();
                    HttpClientUtils.closeQuietly(response);
                } catch (Exception e2) {
                    LOG.error("unable to close http stream after error", e2);
                }
                throw new Exception("Failed : HTTP error code : " + response.getStatusLine().getStatusCode());
            }
            HttpClientUtils.closeQuietly(response);
            putRequest.releaseConnection();
            return result;
        }
        result = ResponseHelper.getJson(response);
        EntityUtils.consume(response.getEntity());
        HttpClientUtils.closeQuietly(response);
        putRequest.releaseConnection();
        return result;
    }
}
