/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keys.internal;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.email.storage.ogEmail.Email;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.keys.GuardKeyService;
import com.openexchange.guard.keys.GuardMasterKeyService;
import com.openexchange.guard.keys.HKPClientService;
import com.openexchange.guard.keys.PGPPublicKeyRingFactory;
import com.openexchange.guard.keys.PublicKeyService;
import com.openexchange.guard.keys.dao.GuardKeys;
import com.openexchange.guard.keys.dao.OGPGPKey;
import com.openexchange.guard.keys.dao.RecipKey;
import com.openexchange.guard.keys.exceptions.PublicKeyExceptionCodes;
import com.openexchange.guard.keys.osgi.Services;
import com.openexchange.guard.keys.storage.KeyTableStorage;
import com.openexchange.guard.keys.storage.OGPGPKeysStorage;
import com.openexchange.guard.keys.util.PGPUtil;
import com.openexchange.guard.mailcreator.MailCreatorService;
import com.openexchange.guard.notification.GuardNotificationService;
import com.openexchange.guard.oxapi.Capabilities;
import com.openexchange.guard.ratifier.GuardRatifierService;
import com.openexchange.guard.translation.GuardTranslationService;
import com.openexchange.guard.user.GuardUser;
import com.openexchange.guard.user.GuardUserService;

/**
 * {@link PublicKeyServiceImpl}
 *
 * @author <a href="mailto:martin.schneider@open-xchange.com">Martin Schneider</a>
 * @since v2.4.0
 */
public class PublicKeyServiceImpl implements PublicKeyService {

    private static final String WEBMAIL_CAPABILITY = "webmail";
    private static final Logger LOG = LoggerFactory.getLogger(PublicKeyServiceImpl.class);

    @Override
    public RecipKey getPubKey(String email, int senderID, int senderCID, String from, boolean send, boolean remote, String host) throws OXException {
        return (getPubKey(email, senderID, senderCID, from, send, remote, "", "", null, -1, -1, host));
    }

    private GuardKeys getPGPKeysFromEmail(String email, int userid, int cid) throws OXException {
        OGPGPKeysStorage ogpgpKeysStorage = Services.getService(OGPGPKeysStorage.class);
        try {
            List<OGPGPKey> keys = ogpgpKeysStorage.getForUserByEmail(userid, cid, Arrays.asList(new String[] { email }));
            for (OGPGPKey key : keys) {
                if (key.getUserId() == userid) {
                    return this.create(key);
                }
            }
        } catch (Exception e) {
            // TODO throw oxexception
        }
        return null;
    }

    private GuardKeys create(OGPGPKey key) throws IOException {
        GuardKeys ret = new GuardKeys();
        ret.setEmail(key.getEmail());
        ret.setPublicKeyRing(PGPPublicKeyRingFactory.create(key.getPublicPGPAscData()));
        ret.setInline(key.isInline());
        ret.setContextid(key.getContextId());
        ret.setUserid(key.getUserId());
        return ret;
    }

    @Override
    public RecipKey getPubKey(String email, int senderID, int senderCID, String from, boolean send, boolean remote, String guestlang, String fromName, String pin, int ownerId, int contextId, String host) throws OXException {
        GuardRatifierService validatorService = Services.getService(GuardRatifierService.class);
        validatorService.validate(email);

        RecipKey result = null;
        // First, we'll check if there is a key uploaded by the user
        GuardKeys key = null;
        // If there is a senderID, check for upload

        if (senderID != 0) {
            key = this.getPGPKeysFromEmail(email, senderID, senderCID);
        }
        if ((key == null) && (ownerId > 0) && (contextId > 0)) {
            key = this.getPGPKeysFromEmail(email, ownerId, contextId);
        }
        if (key != null) {
            result = new RecipKey(key);
            return result;
        }

        // Next check if we have in our masterKeyTable
        result = checkMaster(email);
        if (result != null) {
            return result;
        }

        if (result == null) {
            // If not in our keytable, and not in cache, then let's search the OX user database to see if member
            try {
                GuardUserService guardUserService = Services.getService(GuardUserService.class);
                GuardUser user = guardUserService.getUser(email);

                if (user != null) {
                    // First, let's check if this is an alias for a known user
                    GuardKeyService keyService = Services.getService(GuardKeyService.class);
                    int userId = user.getId();
                    int lContextId = user.getContextId();
                    GuardKeys aliasKey = keyService.getKeys(userId, lContextId);
                    if (aliasKey != null) {
                        result = recipKeyFromGuardKey(aliasKey);
                        return result;
                    }

                    if (hasWebMail(userId, lContextId) && hasGuard(userId, lContextId)) {
                        result = new RecipKey();
                        result.setUserid(userId);
                        result.setCid(lContextId);
                        result.setLang(user.getLanguage());
                        result.setName(user.getName());
                        result.setEmail(email);
                        result.setPgp(true);
                        if (send) {
                            LOG.debug("Found id for " + email + " in main ox users, creating key");

                            GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES);
                            result.setNewGuestPass(cipherService.generateRandomPassword());

                            MailCreatorService mailCreatorService = Services.getService(MailCreatorService.class);

                            GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
                            int templId = guardConfigService.getIntProperty(GuardProperty.templateID, result.getUserid(), result.getCid());
                            List<String> fromemail = mailCreatorService.getFromAddress(result.getName(), result.getEmail(), senderID, senderCID);
                            JsonObject pwEmail = mailCreatorService.getOxPasswordEmail(result.getEmail(), fromName != null ? fromName : from, fromemail.get(0), fromemail.get(1), result.getNewGuestPass(), result.getLang(), templId, host, result.getUserid(), result.getCid());

                            GuardNotificationService guardNotificationService = Services.getService(GuardNotificationService.class);
                            guardNotificationService.send(pwEmail, null);

                            KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
                            GuardKeys currentKeyForUser = ogKeyTableStorage.getCurrentKeyForUser(result.getUserid(), result.getCid());
                            boolean markNewKeyAsCurrent = true;
                            if (currentKeyForUser != null) {
                                markNewKeyAsCurrent = false;
                            }

                            // When creating a new key for a recipient, we will always create a recovery in case of lost email, etc
                            GuardMasterKeyService masterKeyService = Services.getService(GuardMasterKeyService.class);
                            GuardKeys keys = masterKeyService.createMasterKeys(result.getUserid(), result.getCid(), email, result.getName(), result.getNewGuestPass(), result.getLang(), false, true, markNewKeyAsCurrent);
                            result.setPgpring(keys.getPublicKeyRing());
                            result.setPubkey((keys == null) ? null : keys.getPublicKey());

                        }
                        return result;
                    }
                }
            } catch (Exception ex) {
                LOG.error("Error getting public key for " + email, ex);
                //                result = null;
                //                return result;
                throw new OXException();
            }
        }

        // remote = true if from remote lookup - guest accounts wont be created
        if (!remote) {
            String hkpClientToken;
            try {
                hkpClientToken = (from == null || from.isEmpty()) ? null : CipherUtil.getMD5(from);
            } catch (Exception e) {
                throw PublicKeyExceptionCodes.CRYPTO_ERROR.create(e, e.getMessage());
            }
            HKPClientService hkpClient = Services.getService(HKPClientService.class);
            PGPPublicKeyRing remoteKey = hkpClient.findFirst(hkpClientToken, email);
            if (remoteKey != null) {
                result = new RecipKey();
                result.setPgpring(remoteKey);
                result.setEmail(email);
                result.setPgp(true);
                result.setExpired(PGPUtil.checkAllExpired(remoteKey));
            }

            if (result != null) {
                return result;
            }
        } else {
            return result; // If remote lookup, we don't create guest accounts, exit here.
        }

        // Finally, if all other options exhausted, create new guest account
        if (result == null) {
            result = createGuest(email, senderCID, send, guestlang, pin);
        }
        return result;

    }

    @Override
    public RecipKey checkMaster(String email) throws OXException {
        RecipKey result = null;
        EmailStorage restDbOGEmailStorage = Services.getService(EmailStorage.class);

        //Get the mapping for the specified email
        Email emailMapping = restDbOGEmailStorage.getByEmail(email);
        if (emailMapping != null) {

            //looking for a current key
            KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
            GuardKeys currentKey = ogKeyTableStorage.getCurrentKeyForUser(emailMapping.getUserId(), emailMapping.getContextId(), email);
            if (currentKey != null) {
                result = recipKeyFromGuardKey(currentKey);
                LOG.debug("Found key for " + email + " in keytable, cid = " + result.getCid());
            }
        }
        return result;
    }

    /**
     * Check if the user has Guard capability enabled
     *
     * @param id The user identifier
     * @param cid The context identifier
     * @return true if the specified user has the 'guard' capability; false otherwise
     */
    private static boolean hasGuard(int id, int cid) {
        try {
            Capabilities cap = new Capabilities();
            return cap.hasUserCapability("guard", id, cid);
        } catch (Exception e) {
            return true;
        }
    }

    /**
     * Checks if the user has Webmail capability enabled
     * 
     * @param id the user identifier
     * @param cid the context identifier
     * @return true if the specified user has the "webmail" capability; false otherwise
     * @throws Exception due an error
     */
    private static boolean hasWebMail(int id, int cid) throws Exception {
        return new Capabilities().hasUserCapability(WEBMAIL_CAPABILITY, id, cid);
    }

    // Create a RecipKey from GuardKey
    private RecipKey recipKeyFromGuardKey(GuardKeys key) throws OXException {
        if (key != null) {
            RecipKey result = new RecipKey();
            result.setPubkey(key.getPublicKey());
            result.setUserid(key.getUserid());
            result.setCid(key.getContextid());
            result.setGuest(key.getContextid() < 0);
            result.setSettings(key.getSettings());
            result.setPgpring(key.getPublicKeyRing());
            if (result.getPgpring() != null) {
                result.setExpired(PGPUtil.checkAllExpired(result.getPgpring()));
                result.setPgp(true);
            }
            result.setLang(key.getLanguage());
            if (result.getLang() == null) {
                GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
                result.setLang(translationService.getAvailableCode("NON_EXISTENT"));
            }
            return result;
        }
        return null;
    }

    // Create a RecipKey from GuardKey
    private RecipKey createGuest(String email, int senderCID, boolean sending, String guestlang, String pin) throws OXException {
        RecipKey result = null;
        if (senderCID < 0) {
            return null; // Guest can't create another Guest account
        }
        if (sending) {
            result = new RecipKey();
            if (guestlang == null || guestlang.equals("")) {  // Guest language will be blank if email sent before prompt displayed
                GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
                guestlang = configService.getProperty(GuardProperty.defaultLanguage);
            }
            GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES);
            result.setNewGuestPass(cipherService.generateRandomPassword());

            GuardMasterKeyService masterKeyService = Services.getService(GuardMasterKeyService.class);
            GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
            GuardKeys keys = masterKeyService.createMasterKeys(0, (senderCID * -1), email, email, result.getNewGuestPass(), translationService.getAvailableCode(guestlang), false, true, true);// Guest

            result.setPubkey((keys == null) ? null : keys.getPublicKey());
            result.setPgpring((keys == null) ? null : keys.getPublicKeyRing());
            result.setUserid((keys == null) ? -1 : keys.getUserid());
            result.setCid((senderCID * -1));
            result.setGuest(true);
            result.setLang(translationService.getAvailableCode(guestlang));
            if (pin != null) {
                try {
                    GuardKeyService keyService = Services.getService(GuardKeyService.class);
                    keyService.updatePin(result.getUserid(), result.getCid(), pin);
                } catch (Exception e) {
                    LOG.error("Unable to add PIN ", e);
                }
            }
        } else {
            result = new RecipKey();
            result.setPubkey(null);
            result.setGuest(true);
        }
        return result;
    }
}
