/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keys.dao;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.X509EncodedKeySpec;
import java.sql.Date;
import java.util.Iterator;
import org.apache.commons.codec.binary.Base64;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPKeyRingGenerator;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.operator.bc.BcKeyFingerprintCalculator;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.keys.PGPPublicKeyRingFactory;
import com.openexchange.guard.keys.exceptions.KeysExceptionCodes;

/**
 * Manage PGP and RSA keys. Call create and encrypt of keys
 *
 * @author greg
 */
public class GuardKeys {

    private PGPSecretKeyRing keyring;

    private PGPPublicKeyRing publicKeyRing;

    private PublicKey publicKey;

    private String privkey;

    private String salt;

    private boolean passwordNeeded = false;

    private int userid;

    private int contextid;

    private Date lastup;

    private String misc;

    private JsonObject settings;

    private String email;

    private String question;

    private String answer;

    private String recovery;

    private long keyid;

    private boolean current;

    private boolean inline = false;

    private int version;

    private String language;

    private boolean recoveryAvail = false;

    /**
     * Initialises a new {@link GuardKeys}.
     */
    public GuardKeys() {
        salt = CipherUtil.generateSalt();
    }

    public GuardKeys(int userId, int contextId, PublicKey publickey, String eMail) {
        userid = userId;
        contextid = contextId;
        publicKey = publickey;
        email = eMail;
    }

    /**
     * Populate the {@link GuardKeys} based on the {@link DeletedKey}
     *
     * @param userId
     * @param contextId
     * @param keyRing
     * @param salt
     */
    public GuardKeys(int userId, int contextId, PGPSecretKeyRing keyRing, String lSalt) {
        keyring = keyRing;
        userid = userId;
        contextid = contextId;
        salt = lSalt;
    }

    /**
     * @param gen PGPKeyRingGenerator
     * @param pub PublicKey
     * @param priv PrivateKey
     * @param Password Password for the private keys
     * @param Salt Salt used for the passwords.
     */
    public GuardKeys(PGPKeyRingGenerator gen, String encryptedPrivateKey, PublicKey pub, String password, String salted) {
        keyring = gen.generateSecretKeyRing();
        setPublicKeyRing(gen.generatePublicKeyRing());
        publicKey = pub;
        salt = salted;
        privkey = encryptedPrivateKey;
    }

    /**
     * Sets the PGPSecretKeyRing
     *
     * @param ring the PGPSecretKeyRing
     */
    public void setPGPSecretKeyRing(PGPSecretKeyRing ring) {
        keyring = ring;
    }

    /**
     * Asserts whether the private key is <code>null</code>
     *
     * @return true if the private key is <code>null</code>; false otherwise
     */
    public boolean privKeyNull() {
        return privkey == null;
    }

    /**
     * Set the already encoded private RSA
     *
     * @param priv
     */
    public void setEncodedPrivate(String priv) {
        privkey = priv;
    }

    /**
     * Set the RSA public key from stored string
     *
     * @param data
     */
    public void setPublicKeyFrom64String(String data) throws OXException {
        if (data == null) {
            publicKey = null;
            return;
        }
        try {
            KeyFactory kf = KeyFactory.getInstance("RSA");
            // for private keys use PKCS8EncodedKeySpec; for public keys use X509EncodedKeySpec
            X509EncodedKeySpec ks = new X509EncodedKeySpec(Base64.decodeBase64(data));
            publicKey = kf.generatePublic(ks);
        } catch (InvalidKeySpecException e) {
            throw KeysExceptionCodes.INVALID_KEY_SPECIFICATION.create(e);
        } catch (NoSuchAlgorithmException e) {
            throw KeysExceptionCodes.NO_SUCH_ALGORITHM.create(e);
        }
    }

    /**
     * Return public RSA key
     *
     * @return
     */
    public PublicKey getPublicKey() {
        return publicKey;
    }

    /**
     * Get Base64 representation of Public key
     *
     * @return
     */
    public String getEncodedPublic() {
        if (publicKey == null) {
            return (null);
        }
        return Base64.encodeBase64String(publicKey.getEncoded());
    }

    /**
     * Get the Base64 Encoded and encrypted Private key
     *
     * @return
     */
    public String getEncodedPrivate() {
        return privkey;
    }

    /**
     * Set RSA Public Key
     *
     * @param pub
     */
    public void setPublic(PublicKey pub) {
        publicKey = pub;
    }

    /**
     * Return the Public Key
     *
     * @return
     */
    public PGPPublicKey getPGPPublicKey() {
        Iterator<PGPPublicKey> keys = getPublicKeyRing().getPublicKeys();
        PGPPublicKey key = null;
        while (keys.hasNext()) {
            key = keys.next();
            if (key.isEncryptionKey() && !(key.isMasterKey())) {
                return key;
            }
        }
        return getPublicKeyRing().getPublicKey();

    }

    /**
     * Get Base64 Encoded String
     *
     * @return
     * @throws IOException
     */
    public String getEncodedPGPPublic() throws OXException {
        try {
            return Base64.encodeBase64String(getPGPPublicKey().getEncoded());
        } catch (IOException e) {
            throw KeysExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    /**
     * Return the secret Key
     *
     * @return
     */
    public PGPSecretKey getPGPSecretKey() {
        if (keyring == null) {
            return null;
        }
        return keyring.getSecretKey();
    }

    public PGPSecretKey getPGPSigningKey() {
        //return PGPUtil.getSigningKey(keyring);
        Iterator<PGPSecretKey> it = keyring.getSecretKeys();
        PGPSecretKey toreturn = null;
        while (it.hasNext()) {
            PGPSecretKey key = it.next();
            if (key.isMasterKey()) {
                toreturn = key;  // return master key only if no usable sub-keys found
            } else if (key.isSigningKey() && !key.getPublicKey().hasRevocation()) {
                return (key);
            }
        }
        return toreturn;
    }

    public PGPSecretKeyRing getPGPSecretKeyRing() {
        return keyring;
    }

    /**
     * Get Base64 Encoded Secret Keys
     *
     * @return
     * @throws OXException
     */
    public String getEncodedPGPSecret() throws OXException {
        if (keyring == null) {
            return null;
        }
        try {
            return Base64.encodeBase64String(keyring.getEncoded());
        } catch (IOException e) {
            throw KeysExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    /**
     * Set PGPSecret info from base64 string
     *
     * @param keystring
     * @return
     * @throws OXException
     */
    public void setPGPSecretFromString(String keystring) throws OXException {
        if (keystring == null) {
            keyring = null;
            return;
        }
        try {
            keyring = new PGPSecretKeyRing(Base64.decodeBase64(keystring), new BcKeyFingerprintCalculator());
        } catch (IOException e) {
            throw KeysExceptionCodes.IO_ERROR.create(e, e.getMessage());
        } catch (PGPException e) {
            throw KeysExceptionCodes.PGP_ERROR.create(e, e.getMessage());
        }

    }

    /**
     * Get public key from base64 String
     *
     * @param keystring Base64String
     * @return
     * @throws OXException
     */
    public PGPPublicKey getPGPPublicKeyFromString(String keystring) throws OXException {
        try {
            PGPPublicKeyRing pkr = new PGPPublicKeyRing(Base64.decodeBase64(keystring), new BcKeyFingerprintCalculator());
            return pkr.getPublicKey();
        } catch (IOException e) {
            throw KeysExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    public void setSalt(String salted) {
        salt = salted;
    }

    public String getSalt() {
        return salt;

    }

    public void setPGPKeyRingFromAsc(String ascdata) throws OXException {
        try {
            setPublicKeyRing(PGPPublicKeyRingFactory.create(ascdata));
        } catch (IOException e) {
            throw KeysExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    public String exportPGPPublicKeyRing() throws OXException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        ArmoredOutputStream arm = new ArmoredOutputStream(out);
        try {
            arm.write(getPublicKeyRing().getEncoded());
            arm.close();
            out.close();
        } catch (IOException e) {
            throw KeysExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
        return new String(out.toByteArray());
    }

    /**
     * Returns the length of the RSA public key
     *
     * @return the length of the RSA public key
     */
    public int getKeyLen() {
        RSAPublicKey rsapub = (RSAPublicKey) publicKey;
        return (rsapub.getModulus().bitLength());
    }

    /**
     * Returns the PGP public key ring
     *
     * @return the PGP public key ring
     */
    public PGPPublicKeyRing getPublicKeyRing() {
        return publicKeyRing;
    }

    public PGPPublicKeyRing setPublicKeyRing(PGPPublicKeyRing pubring) {
        this.publicKeyRing = pubring;
        return pubring;
    }

    public boolean isPasswordNeeded() {
        return passwordNeeded;
    }

    public void setPasswordNeeded(boolean passwordNeeded) {
        this.passwordNeeded = passwordNeeded;
    }

    public int getUserid() {
        return userid;
    }

    public void setUserid(int userid) {
        this.userid = userid;
    }

    public int getContextid() {
        return contextid;
    }

    public void setContextid(int contextid) {
        this.contextid = contextid;
    }

    public Date getLastup() {
        return lastup;
    }

    public void setLastup(Date lastup) {
        this.lastup = lastup;
    }

    public String getMisc() {
        return misc;
    }

    public void setMisc(String misc) {
        this.misc = misc;
    }

    public JsonObject getSettings() {
        return settings;
    }

    public void setSettings(JsonObject settings) {
        this.settings = settings;
    }

    public String getEmail() {
        return email;
    }

    public void setEmail(String email) {
        this.email = email;
    }

    public String getQuestion() {
        return question;
    }

    public void setQuestion(String question) {
        this.question = question;
    }

    public String getAnswer() {
        return answer;
    }

    public void setAnswer(String answer) {
        this.answer = answer;
    }

    public String getRecovery() {
        return recovery;
    }

    public void setRecovery(String recovery) {
        this.recovery = recovery;
    }

    public long getKeyid() {
        return keyid;
    }

    public void setKeyid(long keyid) {
        this.keyid = keyid;
    }

    public boolean isCurrent() {
        return current;
    }

    public void setCurrent(boolean current) {
        this.current = current;
    }

    public boolean isInline() {
        return inline;
    }

    public void setInline(boolean inline) {
        this.inline = inline;
    }

    public int getVersion() {
        return version;
    }

    public void setVersion(int version) {
        this.version = version;
    }

    public String getLanguage() {
        return language;
    }

    public void setLanguage(String language) {
        this.language = language;
    }

    public boolean isRecoveryAvail() {
        return recoveryAvail;
    }

    public void setRecoveryAvail(boolean recoveryAvail) {
        this.recoveryAvail = recoveryAvail;
    }

}
