/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keys;

import java.util.List;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import com.openexchange.exception.OXException;
import com.openexchange.guard.keys.dao.GuardKeys;
import com.openexchange.guard.keys.dao.KeySignatureVerificationResult;

/**
 * {@link PGPKeyService}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public interface PGPKeyService {

    /**
     * Update the specified PGP keys for the specified user
     * 
     * @param id The user identifier
     * @param cid The context identifier
     * @param keys The PGP keys to update
     * @throws OXException
     */
    void updatePGPKeys(int id, int cid, GuardKeys keys) throws OXException;

    /**
     * Get the public keyring associated with the id.
     * 
     * @param reqId the public keyring identifier
     * @param userid The user identifier
     * @param cid The context identifier
     * @return The PGP public key ring
     * @throws OXException
     */
    PGPPublicKeyRing getPublicKeyRingById(long reqId, int userid, int cid) throws OXException;

    /**
     * Get the public keyring associated with the hex id.
     * 
     * @param hexId The hex identifier for the public keyring
     * @param userid The user identifier
     * @param cid The context identifier
     * @return The PGP public key ring associated with the specified hex identifier
     * @throws OXException
     */
    PGPPublicKeyRing getPublicKeyRingById(String hexId, int userid, int cid) throws OXException;

    /**
     * Search for public keys for email
     * 
     * @param email The email address to search
     * @return A list with all PGP public key rings found for the specified e-mail address
     * @throws OXException
     */
    List<PGPPublicKeyRing> getPublicKeyRing(String email) throws OXException;

    /**
     * Update the PGP Key record updating PGP Secret key and recovery on duplicate
     * 
     * @param id The user identifier
     * @param cid The context identifier
     * @param key The key to update
     * @param recovery The recovery password
     * @param duplicate The duplicate
     * @throws OXException
     */
    void updateDuplicatePGPKey(int id, int cid, GuardKeys key, String recovery, Long duplicate) throws OXException;

    /**
     * Exports the specified PGP public key ring and returns it as string
     * 
     * @param pgpKeyRing
     * @return The PGP public key ring
     * @throws OXException
     */
    String export(PGPPublicKeyRing pgpKeyRing) throws OXException;

    /**
     * Get the private PGP key from the specified key ring
     * 
     * @param password The password to decrypt the private key
     * @param salt the salt
     * @param keyRing The key ring to retrieve the PGP key from
     * @return The private PGP key
     * @throws OXException
     */
    PGPPrivateKey getPGPPrivateKey(String password, String salt, PGPSecretKeyRing keyRing) throws OXException;

    /**
     * Retrieves the private PGP key with the specified identifier
     * 
     * @param password The password to decrypt the private key
     * @param salt the salt
     * @param keyRing The key ring to retrieve the PGP key from
     * @param id The key identifier
     * @return The private PGP key
     * @throws OXException
     */
    PGPPrivateKey getPGPPrivateKey(String password, String salt, PGPSecretKeyRing keyRing, long id) throws OXException;

    /**
     * Retrieve the signing key from the specified secret key ring.
     * 
     * @param keyRing The secret PGP key ring
     * @param password The password
     * @param salt The salt
     * @return The signing key
     * @throws OXExcepion
     */
    PGPPrivateKey getPGPSignKey(PGPSecretKeyRing keyRing, String password, String salt) throws OXException;

    /**
     * Check all signatures of the keys in the specified key ring
     * 
     * @param id The user identifier
     * @param cid The context identifier
     * @param keyRing The key ring
     * @return A list with the verification results
     * @throws OXException
     */
    List<KeySignatureVerificationResult> checkSignatures(int id, int cid, PGPPublicKeyRing keyRing) throws OXException;

    /**
     * Check all the signatures of the specified key
     * 
     * @param id The user identifier
     * @param cid The context identifier
     * @param key The key
     * @return A list with the verification results
     * @throws OXException
     */
    List<KeySignatureVerificationResult> checkSignatures(int id, int cid, PGPPublicKey key) throws OXException;

    /**
     * Add a a new user identifier to the specified public key ring. The specified private key is used for signing.
     * 
     * @param privateKey The private key to sign with
     * @param publicKeyRing The public key ring
     * @param newId The new user identifier
     * @return The public key ring with the new public key
     * @throws OXException
     */
    PGPPublicKeyRing addUID(PGPPrivateKey privateKey, PGPPublicKeyRing publicKeyRing, String userId) throws OXException;

    /**
     * Checks for a bad user identifier
     * 
     * @param keys The {@link GuardKeys}
     * @param password The password
     * @throws OXException
     */
    void checkBadUID(GuardKeys keys, String password) throws OXException;

    /**
     * Create a public key ring from a private key
     * 
     * @param secretKeyRing The secret key ring
     * @return The generated public key ring
     * @throws OXException
     */
    PGPPublicKeyRing createPublicKeyRingFromPrivateKey(PGPSecretKeyRing secretKeyRing) throws OXException;

    /**
     * Revokes the PGP Public keys associated with the GuardKeys and returns with the revocation certificate
     * 
     * @param keys The {@link GuardKeys} containing the public keys
     * @param password The password
     * @param reason The revocation reason
     * @return The updated {@link GuardKeys}
     * @throws OXException
     */
    GuardKeys revokePGPPublicKeys(GuardKeys keys, String password, String reason) throws OXException;

    /**
     * Sign the public key
     * 
     * @param keyToSign The key to sign
     * @param keys The {@link GuardKeys}
     * @param password The password
     * @return The signed public key
     * @throws OXException
     */
    PGPPublicKey signPGPPublicKey(PGPPublicKey keyToSign, GuardKeys keys, String password) throws OXException;

    /**
     * 
     * @param secretKey The secret key
     * @param privateKey The private key
     * @param keyToSign The key to sign
     * @return The public key ring
     * @throws OXException
     */
    PGPPublicKeyRing signPGPPublicKeyRing(PGPSecretKey secretKey, PGPPrivateKey privateKey, PGPPublicKey keyToSign) throws OXException;

    /**
     * Exports the pgp private key as string
     * 
     * @param keys The {@link GuardKeys}
     * @param password The password
     * @return The pgp key as string
     * @throws OXException
     */
    String exportPGPPrivateKey(GuardKeys keys, String password) throws OXException;
    
    /**
     * Imports the specified pgp private key
     * @param keys The {@link GuardKeys} containing the pgp private key
     * @param pgpPrivateAsc The pgp private asc
     * @param pgpPassword The PGP password
     * @param guardPassword The guard password
     * @return
     * @throws OXException
     */
    GuardKeys importPGPPrivateKey(GuardKeys keys, String pgpPrivateAsc, String pgpPassword, String guardPassword) throws OXException;
}
