/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keys;

import java.security.PrivateKey;
import org.bouncycastle.openpgp.PGPKeyPair;
import com.openexchange.exception.OXException;
import com.openexchange.guard.keys.dao.GuardKeys;

/**
 * {@link GuardKeyService}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public interface GuardKeyService {

    /**
     * Gets the {@link GuardKeys} for the specified user
     *
     * @param id The user identifier
     * @param cid The context identifier
     * @return The {@link GuardKeys}
     * @throws OXException
     */
    GuardKeys getKeys(int id, int cid) throws OXException;

    /**
     * Gets the {@link GuardKeys} for the specified user from the specified shard
     *
     * @param id The user identifier
     * @param cid The context identifier
     * @param shard The shard identifier
     * @return The {@link GuardKeys}
     * @throws OXException
     */
    GuardKeys getKeys(int id, int cid, int shard) throws OXException;

    /**
     * Gets the {@link GuardKeys} for the specified user
     *
     * @param id The user identifier
     * @param cid The context identifier
     * @param keyId The key identifier
     * @param current flag indicating that the current key should be returned
     * @return The {@link GuardKeys}
     * @throws OXException
     */
    GuardKeys getKeys(int id, int cid, long keyId, boolean current) throws OXException;

    /**
     * Retrieve keys from Mater KeyTable
     *
     * @param id The user identifier
     * @param cid The context identifier
     * @param shard The shard identifier
     * @param keyId The key identifier
     * @param current flag indicating whether to retrieve the current key
     * @return The GuardKeys
     * @throws OXException
     */
    GuardKeys getKeys(int id, int cid, int shard, String keyId, boolean current) throws OXException;

    /**
     * Stores the specified key
     *
     * @param keys The key to store
     * @param userCreated flag indicating whether the user is created
     * @return The stored keys
     * @throws OXException
     */
    GuardKeys storeKeys(GuardKeys keys, boolean userCreated) throws OXException;

    /**
     * Gets the key for the user with the specified e-mail address
     *
     * @param email The e-mail address of the user
     * @return The key
     * @throws OXException
     */
    GuardKeys getKeysFromEmail(String email) throws OXException;

    /**
     * Gets the key with the specified identifier for the user with the specified e-mail address
     *
     * @param email The e-mail address
     * @param keyId The key identifier
     * @return The key
     * @throws OXException
     */
    GuardKeys getKeysFromEmail(String email, long keyId) throws OXException;

    /**
     * Updates the specified keys for the specified user
     *
     * @param id The user identifier
     * @param cid The context identifier
     * @param keys The keys to update
     * @param recovery The recovery password
     * @param reset flag indicating whether to reset the password
     * @throws OXException
     */
    void updateKeys(int id, int cid, GuardKeys keys, String recovery, boolean reset) throws OXException;

    /**
     * Create a recovery password
     *
     * @param keys The key for which to create the recovery password
     * @param password The password to use
     * @return The recovery password as string
     * @throws OXException
     */
    String createRecovery(GuardKeys keys, String password) throws OXException;

    /**
     * Generate RSA/PGP keys for user. Use the RSA key cache
     *
     * @param name Name of user
     * @param email Email address
     * @param password Password to protect keys
     * @return The generated key pair
     */
    GuardKeys generateKeyPair(String name, String email, String password) throws OXException;

    /**
     * Generate RSA/PGP keys for user.
     *
     * @param name Name of user
     * @param email Email address
     * @param password Password to protect keys
     * @param failover If no keys in cache, then generate real-time
     * @return The generated key pair
     */
    GuardKeys generateKeyPair(String name, String email, String password, boolean failover) throws OXException;

    /**
     * Generate RSA/PGP keys for user
     *
     * @param name The name of the user
     * @param email The email address of the user
     * @param password The password to protect the keys
     * @return The generated {@link GuardKeys}
     * @throws OXException
     */
    GuardKeys generateKeys(String name, String email, String password) throws OXException;

    /**
     * Generate RSA/PGP keys for user
     *
     * @param name The name of the user
     * @param email The email address of the user
     * @param password The password to protect the keys
     * @param rsaMaster
     * @param rsaEncrypt
     * @return The generated {@link GuardKeys}
     * @throws OXException
     */
    GuardKeys generateKeys(String name, String email, String password, PGPKeyPair rsaMaster, PGPKeyPair rsaEncrypt) throws OXException;

    /**
     * Adds the specified {@link GuardKeys} key
     *
     * @param key The key to add
     * @throws OXException
     */
    void addNewKey(GuardKeys key) throws OXException;

    /**
     * Updates the PIN for the specified guest user
     *
     * @param id The user identifier
     * @param cid The context identifier
     * @param pin The new PIN
     * @throws OXException
     */
    void updatePin(int id, int cid, String pin) throws OXException;

    /**
     * Removes the PIN from the specified guest
     * 
     * @param email the email of the guest user / guest key to remove the PIN for
     * @throws OXExeption
     */
    void removePin(String email) throws OXException;

    /**
     * Recovers the password hash for the specified user
     *
     * @param id The user identifier
     * @param cid The context identifier
     * @return The password hash
     * @throws OXException
     */
    String recoverPasswordHash(int id, int cid) throws OXException;

    /**
     * Resets password for an account with new password, stores new keys, and stores new recovery
     *
     * @param email The e-mail address of the user
     * @param newpass The new password
     * @throws OXException
     */
    void resetPassword(String email, String newpass) throws OXException;

    /**
     * Changes the key password and generates new {@link GuardKeys}
     *
     * @param oldPass The old password
     * @param newPass The new password
     * @param oldKey The old key
     * @return The new {@link GuardKeys}
     * @throws OXException
     */
    GuardKeys changePassword(String oldPass, String newPass, GuardKeys oldKey) throws OXException;

    /**
     * Changes the key password and generates a new {@link GuardKeys}
     *
     * @param oldPassHash The old hashed password
     * @param newPass The new password (unhashed)
     * @param oldKey The old key
     * @return The new {@link GuardKeys}
     * @throws OXException
     */
    GuardKeys changePasswordWithRecovery(String oldPassHash, String newPass, GuardKeys oldKey) throws OXException;

    /**
     * Store the security question and answer for the specified user
     *
     * @param id The user identifier
     * @param cid The context identifier
     * @param question The question
     * @param answer The answer
     * @return true if the question was successfully stored; false otherwise
     * @throws OXException
     */
    boolean storeQuestion(int id, int cid, String question, String answer) throws OXException;

    /**
     * Encrypt the private key with the specified password and salt
     *
     * @param privateKey The private key to encrypt
     * @param password The password to encrypt it with
     * @param salt The salt to use
     * @return The encrypted private key as string
     * @throws OXException
     */
    String encryptPrivateKey(PrivateKey privateKey, String password, String salt) throws OXException;

    /**
     * Decrypt the private key with the specified password and salt. In addition the Password will be extended by the salt.
     *
     * @param privateKey the encrypted private key
     * @param password The password to decrypt it with
     * @param salt The salt
     * @param logBadAttempt flag indicating whether to log a bad decryption attempt
     *
     * @return The decrypted {@link PrivateKey}
     * @throws OXException
     */
    PrivateKey decryptPrivateKey(String privateKey, String password, String salt, boolean logBadAttempt) throws OXException;

    /**
     * Decrypt the private key with the specified password hash.
     *
     * @param privateKey the encrypted private key
     * @param password The password hash (This is the salted password)
     * @param salt The salt
     * @param logBadAttempt flag indicating whether to log a bad decryption attempt
     *
     * @return The decrypted {@link PrivateKey}
     * @throws OXException
     */
    PrivateKey decryptPrivateKeyWithHash(String privateKey, String hash, String salt, boolean logBadAttempt) throws OXException;

    /**
     * Generates a password
     * 
     * @return String with a generated password
     * @throws OXException
     */
    String generatePassword() throws OXException;

    /**
     * Returns if password recovery is enabled for the given user/context id.
     * 
     * @param userId The id of the user
     * @param contextId The id of the context
     * @return <code>true</code> if enabled, otherwise <code>false</code>
     * @throws OXException
     */
    boolean isRecoveryEnabled(int userId, int contextId) throws OXException;
}
