/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.hkp.internal;

import java.util.Iterator;
import java.util.List;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.LongUtil;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.hkp.exception.HkpExceptionCodes;
import com.openexchange.guard.hkp.osgi.Services;
import com.openexchange.guard.hkp.servlets.HkpServlet;
import com.openexchange.guard.hkp.utils.CheckWhitelist;
import com.openexchange.guard.keys.PGPKeyService;
import com.openexchange.guard.keys.PublicKeyService;
import com.openexchange.guard.keys.dao.RecipKey;

/**
 *
 * {@link HKPIndex} - The HKP "index" operation requests a list of keys on the keyserver
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class HKPIndex {

    private static final Logger logger = LoggerFactory.getLogger(HkpServlet.class);

    /**
     * Internal Method for assert that the mail does not contain some special characters
     *
     * @param email The mail
     * @return email If the mail does not contain any unwanted characters
     * @throws OXException In case email contains some unwanted characters
     */
    private String assertUnwantedCharacters(String email) throws OXException {
        if (email.contains("%")) {
            throw HkpExceptionCodes.INVALID_EMAIL_REQUEST.create();
        }
        return email;
    }

    /**
     * Creates string representation from a Public key ring list
     *
     * @param keyrings the list to create the string for
     * @return keyrings as string representation
     */
    private String keyRingsToString(List<PGPPublicKeyRing> keyrings) {
        StringBuilder resp = new StringBuilder();
        resp.append("info:1:" + keyrings.size() + "\r\n");
        for (int i = 0; i < keyrings.size(); i++) {
            Iterator<PGPPublicKey> keys = keyrings.get(i).getPublicKeys();
            while (keys.hasNext()) {
                PGPPublicKey key = keys.next();
                String id = LongUtil.longToHexString(key.getKeyID());
                String algo = Integer.toString(key.getAlgorithm());
                String keylen = Integer.toString(key.getBitStrength());
                long cr = key.getCreationTime().getTime() / 1000;
                String creation = Long.toString(cr);
                String exp = key.getValidSeconds() == 0 ? "" : Long.toString((key.getValidSeconds()) + cr);
                String pub = "pub:" + id + ":" + algo + ":" + keylen + ":" + creation + ":" + exp + ":";
                resp.append(pub + "\r\n");
                // create primary key line
                Iterator<?> it = key.getUserIDs();
                while (it.hasNext()) {
                    resp.append("uid:\"" + (String) it.next() + "\":" + creation + ":" + exp + ":\r\n");
                    // get userids
                }
            }
        }
        return resp.toString();
    }

    /**
     * "The (HKP) index operation requests a list of keys on the keyserver that match the text or key ID in the "search" variable."
     *
     * @param requestIP The IP of the caller
     * @param search The search pattern
     * @return The key list formatted string, or null if the search did not found any keys
     * @throws OXException
     */
    public String index(String requestIP, String search) throws OXException {

        if (search.contains("@")) {

            //Searching for the given email
            PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
            List<PGPPublicKeyRing> keyrings = pgpKeyService.getPublicKeyRing(assertUnwantedCharacters(search)); // zero cid for local search only

            if (keyrings.size() > 0) {
                logger.debug("Sending list of keys, length = " + keyrings.size());
                return keyRingsToString(keyrings);
            } else {
                //No keys found - check if the requester is on the white list; In this case we create a new PGP key, if the requestED user is an OX user
                logger.debug("No Keys found, check whitelist against " + requestIP);
                // If calling server in whitelist, check if OX member to create
                if (CheckWhitelist.isWhiteListed(requestIP)) {

                    logger.debug("IP in whitelist, checking against OX backend");

                    GuardConfigurationService guardConfigurationService = Services.getService(GuardConfigurationService.class);
                    String passwordFromAddress = guardConfigurationService.getProperty(GuardProperty.passwordFromAddress);
                    String passwordFromName = guardConfigurationService.getProperty(GuardProperty.passwordFromName);

                    //Check and see if we can get public key from the backend if member
                    //This will create a key for a requested user if it is an OX user; but not a guest
                    PublicKeyService publicKeyService = Services.getService(PublicKeyService.class);
                    RecipKey key = publicKeyService.getPubKey(assertUnwantedCharacters(search), 0, 0, passwordFromAddress, true, true /* remote=true -> do not create guest accounts or query other HKP servers */, null, passwordFromName, null, -1, -1, "");
                    if (key != null) {
                        //Lookup the new created key
                        keyrings = pgpKeyService.getPublicKeyRing(assertUnwantedCharacters(search));
                        logger.debug("Sending list of keys, length = " + keyrings.size());
                        return keyRingsToString(keyrings);
                    }
                }
            }
        }
        return null;
    }
}
