
package com.openexchange.guard.guestupgrade.internal;

import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.database.utils.GuardConnectionHelper;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.guestupgrade.GuestUpgradeService;
import com.openexchange.guard.guestupgrade.exceptions.GuardGuestUpgraderExceptionCodes;
import com.openexchange.guard.guestupgrade.osgi.Services;
import com.openexchange.guard.keys.GuardKeyService;
import com.openexchange.guard.keys.dao.GuardKeys;
import com.openexchange.guard.keys.storage.KeyTableStorage;
import com.openexchange.guard.keys.storage.PGPKeysStorage;
import com.openexchange.guard.oxapi.Utilities;

/**
 * {@link GuestUpgradeServiceImpl}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class GuestUpgradeServiceImpl implements GuestUpgradeService {

    private static final Logger logger = LoggerFactory.getLogger(GuestUpgradeServiceImpl.class);

    @Override
    public void upgrade(String email) throws OXException {

        //Getting key by email
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        GuardKeys keys = keyService.getKeysFromEmail(email);
        if (keys == null) {
            logger.error("Unable to find guest account " + email);
            throw GuardGuestUpgraderExceptionCodes.UNKNOWN_GUEST_ACCOUNT.create(email);
        }

        //Getting user information from OX backend
        Utilities util = new Utilities();
        JsonObject userdat = util.getUser(email);
        if (userdat.has(email)) {
            JsonObject user = userdat.get(email).getAsJsonObject();
            int oxUserId = user.get("uid").getAsInt();
            int oxUserCid = user.get("cid").getAsInt();
            if (oxUserId != keys.getUserid() || oxUserCid != keys.getContextid()) {
                try {
                    upgrade(keys.getUserid(), keys.getContextid(), oxUserId, oxUserCid);
                    logger.info(email + " Upgraded to OX account");

                } catch (Exception e) {
                    logger.error("Error upgrading account " + email);
                    throw GuardGuestUpgraderExceptionCodes.UPGRADE_ERROR.create(e, email, e.getMessage());
                }
            }
            else {
                logger.error("Already upgraded " + email);
                throw GuardGuestUpgraderExceptionCodes.ALREADY_UPGRADED.create(email);
            }
        }
        else {
            logger.error("No OX account found for " + email);
            throw GuardGuestUpgraderExceptionCodes.UNKNOWN_OX_ACCOUNT.create(email);
        }
    }

    @Override
    public void upgrade(int guestUserId, int guestContextId, int userId, int contextId) throws OXException {

        try {
            try (GuardConnectionHelper connections = new GuardConnectionHelper(contextId, guestContextId, guestUserId)) {

                //Start transactions
                connections.start();

                //Getting the guest's keys
                KeyTableStorage keyTableStorage = Services.getService(KeyTableStorage.class);
                List<GuardKeys> guestKeys = keyTableStorage.getKeysForUser(guestUserId, guestContextId);

                if (!guestKeys.isEmpty()) {
                    //Removing the guest keys from the guest-shard key table
                    keyTableStorage.deleteAllForUser(connections.getGuestConnection(), guestUserId, guestContextId);

                    for (GuardKeys guestKey : guestKeys) {
                        //Setting the new user id for each guest key
                        guestKey.setUserid(userId);
                        guestKey.setContextid(contextId);
                        //Adding it to the main key table
                        keyTableStorage.insert(connections.getUserConnection(), guestKey, true);

                        //Updating the email mapping
                        EmailStorage emailStorage = Services.getService(EmailStorage.class);
                        emailStorage.insertOrUpdate(connections.getGuardConncetion(), guestKey.getEmail(), guestKey.getContextid(), guestKey.getUserid(), 0 /* regular ox guard user */);

                        //Updating PGP mapping
                        PGPKeysStorage pgpKeysStorage = Services.getService(PGPKeysStorage.class);
                        pgpKeysStorage.updateContextIdByKeyId(connections.getGuardConncetion(), guestKey.getKeyid(), contextId);

                        connections.commit();
                    }
                }
            }
        } catch (Exception e) {
            throw GuardGuestUpgraderExceptionCodes.DB_ERROR.create(e,e.getMessage());
        }
    }
}
