
package com.openexchange.guard.common.util;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.openexchange.exception.OXException;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;

/**
 * Utility class
 */
public final class JsonUtil {

    private static final Logger LOG = LoggerFactory.getLogger(JsonUtil.class);

    /**
     * Utility to extract JSON data from a web response
     * 
     * @param webResponse the response to extract data from
     * @return The parsed JSON object
     */
    public static JsonObject extractObject(String webResponse) {
        int i = webResponse.indexOf("{");
        int j = webResponse.lastIndexOf("}");
        String jsonString = webResponse.substring(i, j + 1);
        JsonParser parser = new JsonParser();
        JsonObject result = (JsonObject) parser.parse(jsonString);
        return result;
    }

    /**
     * Parses the specified data as JsonObject
     * 
     * @param data
     * @return
     * @throws IllegalArgumentException if the specified data cannot be parsed as a JsonObject
     */
    public static JsonObject parseAsJsonObject(String data) {
        JsonParser jsonParser = new JsonParser();
        JsonElement element = jsonParser.parse(data);
        if (!element.isJsonNull() && element.isJsonObject()) {
            return element.getAsJsonObject();
        }
        throw new IllegalArgumentException("Cannot parse the provided data as JsonObject: " + data);
    }

    /**
     * 
     * @param json
     * @param name
     * @return
     * @throws OXException
     */
    public static String getStringFromJson(JsonObject json, String name) throws OXException {
        return getStringFromJson(json, name, false);
    }

    /**
     * Get String from JsonObject
     * 
     * @param json
     * @param name
     * @param mandatory
     * @return
     * @throws OXException
     */
    public static String getStringFromJson(JsonObject json, String name, boolean mandatory) throws OXException {
        if ((json == null) || (!json.has(name))) {
            if (mandatory) {
                throw GuardCoreExceptionCodes.JSON_PARAMETER_MISSING.create(name);
            } else {
                return null;
            }
        }
        try {
            return json.get(name).getAsString();
        } catch (Exception ex) {
            LOG.error("Problem getting string parameter " + name + "from json", ex);
            throw GuardCoreExceptionCodes.JSON_ERROR.create(ex, ex.getMessage());
        }
    }

    /**
     * 
     * @param json
     * @param name
     * @return
     * @throws OXException
     */
    public static int getIntFromJson(JsonObject json, String name) throws OXException {
        return getIntFromJson(json, name, false);
    }

    /**
     * 
     * @param json
     * @param name
     * @return
     * @throws OXException
     */
    public static boolean getBooleanFromJson(JsonObject json, String name) throws OXException {
        return getBooleanFromJson(json, name, false);
    }

    /**
     * 
     * @param json
     * @param name
     * @param mandatory
     * @return
     * @throws OXException
     */
    public static boolean getBooleanFromJson(JsonObject json, String name, boolean mandatory) throws OXException {
        if (!json.has(name)) {
            if (mandatory) {
                throw GuardCoreExceptionCodes.JSON_PARAMETER_MISSING.create(name);
            } else {
                return false;
            }
        }
        try {
            return json.get(name).getAsBoolean();
        } catch (Exception ex) {
            LOG.error("Problem getting string parameter " + name + "from json", ex);
            throw GuardCoreExceptionCodes.JSON_ERROR.create(ex, ex.getMessage());
        }
    }

    /**
     * Get int from JsonObject
     * 
     * @param json
     * @param name
     * @param mandatory
     * @return
     * @throws OXException
     */
    public static int getIntFromJson(JsonObject json, String name, boolean mandatory) throws OXException {
        if (!json.has(name)) {
            if (mandatory) {
                throw GuardCoreExceptionCodes.JSON_PARAMETER_MISSING.create(name);
            } else {
                return 0;
            }
        }
        try {
            return json.get(name).getAsInt();
        } catch (Exception ex) {
            LOG.error("Problem getting int parameter " + name + "from json", ex);
            throw GuardCoreExceptionCodes.JSON_ERROR.create(ex, ex.getMessage());
        }
    }

    public static String getError(String response) {
        try {
            int i = response.indexOf("{");
            int j = response.lastIndexOf("}");
            response = response.substring(i, j + 1);
            JsonObject resp = parseAsJsonObject(response);
            if (resp.has("error_desc")) {
                return resp.get("error_desc").getAsString();
            } else {
                return "";
            }
        } catch (Exception ex) {
            return "";
        }
    }
}
