/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.cipher.internal;

import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SecureRandom;
import javax.crypto.Cipher;
import javax.crypto.NoSuchPaddingException;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.cipher.exceptions.GuardCipherExceptionCodes;
import com.openexchange.guard.cipher.osgi.Services;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;

/**
 * {@link AbstractGuardCipherService}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
abstract class AbstractGuardCipherService implements GuardCipherService {

    protected final GuardCipherAlgorithm cipherAlgorithm;

    /**
     * Characters for random password. Excluding 'O' and 'I' for now
     */
    private static final String passChars = "0123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghjklmnpqrstuvwxyz";

    /**
     * Initialises a new {@link AbstractGuardCipherService}.
     * 
     * @param cipherAlgorithm The cipher algorithm
     * @throws OXException If the specified cipher algorithm does not exist, or if the key length of the specified algorithm exceeds the maximum defined
     */
    public AbstractGuardCipherService(GuardCipherAlgorithm cipherAlgorithm) throws OXException {
        super();

        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
        int keyLength = configService.getIntProperty(cipherAlgorithm.getKeyLengthPropertyName());
        try {
            int maxAllowedKeyLength = Cipher.getMaxAllowedKeyLength(cipherAlgorithm.name());
            if (keyLength > maxAllowedKeyLength) {
                throw GuardCipherExceptionCodes.INVALID_KEY_LENGTH.create(cipherAlgorithm.getTransformation(), keyLength, maxAllowedKeyLength);
            }
        } catch (NoSuchAlgorithmException e) {
            throw GuardCipherExceptionCodes.UNKNOWN_CIPHER_ALGORITHM.create(e, cipherAlgorithm.getTransformation());
        }

        this.cipherAlgorithm = cipherAlgorithm;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.crypto.internal.GuardCryptoService#decrypt(java.lang.String, java.security.Key)
     */
    @Override
    public String decrypt(String data, Key secretKey) {
        return decrypt(data, secretKey, true);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.crypto.internal.GuardCryptoService#generateRandomPassword()
     */
    @Override
    public String generateRandomPassword() throws OXException {
        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
        int length = configService.getIntProperty(GuardProperty.newPassLength);

        SecureRandom random = new SecureRandom();
        StringBuilder newPassword = new StringBuilder(length);
        for (int i = 0; i < length; i++) {
            newPassword.append(passChars.charAt(random.nextInt(passChars.length())));
        }

        return newPassword.toString();
    }

    /**
     * Get the Cipher
     * 
     * @return The cipher
     * @throws OXException
     */
    protected Cipher getCipher() throws OXException {
        try {
            return Cipher.getInstance(cipherAlgorithm.getTransformation(), cipherAlgorithm.getProvider());
        } catch (NoSuchAlgorithmException | NoSuchProviderException | NoSuchPaddingException e) {
            throw GuardCipherExceptionCodes.UNKNOWN_CIPHER_ALGORITHM.create(e, cipherAlgorithm.getTransformation());
        }
    }
}
