
package com.openexchange.guard.servlets.authentication;

import javax.servlet.http.HttpServletRequest;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.oxapi.SessionInformation;

/**
 * {@link RestSessionValidator} Validates the sessionID using the OX Rest API
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4
 */
public class RestSessionValidator implements SessionValidator {

    private static final String JSON_CONTEXT = "context";
    private static final String JSON_USER = "user";
    private static final String[] SESSION_PARAMETER_NAMES = { "session" };
    
    private String getSession(HttpServletRequest request) {

        //Getting session from parameter
        for (String sessionIdentifier : SESSION_PARAMETER_NAMES) {
            String session = request.getParameter(sessionIdentifier);
            if (session != null && !session.isEmpty()) {
                return session;
            }
        }

        //..or as fallback from the cookie
        return new OxCookie(request.getCookies()).getSessionID();
    }
        
    @Override
    public
    GuardUserSession validateSession(HttpServletRequest request) throws OXException {
        //Getting the session id from the request
        String sessionId = getSession(request);
        String userAgent = request.getHeader("User-Agent");
        if (sessionId != null && !sessionId.isEmpty()) {
            //Checking if session is valid 
            if(new Api(new OxCookie(request.getCookies()),sessionId,userAgent).verifyLogin()) {
                //Getting user-id and context-id from session - instead of trusting client input
                JsonObject sessionObject = new SessionInformation().getSessionInformation(sessionId);
                if (sessionObject != null && sessionObject.has(JSON_CONTEXT) && sessionObject.has(JSON_USER)) {
                    return new GuardUserSession(sessionId, sessionObject.get(JSON_CONTEXT).getAsInt(), sessionObject.get(JSON_USER).getAsInt());
                }
            }
        }
        return null;
    }
}
