/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgp.decryption.impl;

import java.io.IOException;
import java.io.OutputStream;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Part;
import javax.mail.internet.MimeMessage;
import org.bouncycastle.openpgp.PGPException;
import com.openexchange.exception.OXException;
import com.openexchange.guard.pgp.PGPResult;
import com.openexchange.guard.pgp.PGPUtils;
import com.openexchange.guard.pgp.exceptions.GuardPGPExceptionCodes;

/**
 * Decrypts PGP MIME messages
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class PGPMimeDecrypter {

    private boolean isGuestMessage(MimeMessage message) throws MessagingException {
        return message.getHeader("X-OXGUARD-GUEST") != null;
    }

    /**
     * decrypts the OX Guard part
     * @param message the OX Guard part to decrypt
     * @param userId the user's ID
     * @param contextId the context ID
     * @param password the password to use for decrypting
     * @param outputStream the outputstream to write the decrypted data to
     * @throws PGPException
     * @throws OXException
     */
    public PGPResult decryptPart(Part pgpPart,int userId, int contextId, String password, OutputStream outputStream) throws IOException, MessagingException, OXException, PGPException {
        return PGPUtils.decryptFile(pgpPart.getInputStream(),outputStream,userId,contextId,password,false /*Don't close the stream*/);
    }

    /**
     * decrypts the OX Guard part of the given MIME message
     * @param message the message to search for the OX Guard part to decrypt
     * @param userId the user's ID
     * @param contextId the context ID
     * @param password the password to use for decrypting
     * @param outputStream the outputstream to write the decrypted data to
     * @throws IOException
     * @throws MessagingException
     * @throws PGPException
     * @throws OXException
     */
    public PGPResult decryptMessage(MimeMessage message, int userId, int contextId, String password, OutputStream outputStream) throws MessagingException, IOException, OXException, PGPException {

        //Check if we have a Multipart message
        Object content = message.getContent();
        if(!(content instanceof Multipart)) {
            //No mime message in multipart
            throw GuardPGPExceptionCodes.MIME_NOT_MULTIPART_ERROR.create();
        }
        Multipart multiPartContent = (Multipart)content;

        //Extracting the OX Guard part from the mail message
        Part pgpPart = null;
        if(isGuestMessage(message)){
            pgpPart = new PGPMimeMessageExtractor().extractGuestMessagePart(multiPartContent);
        }
        else {
            pgpPart = new PGPMimeMessageExtractor().extractPGPMessagePart(multiPartContent);
        }
        if(pgpPart == null) {
            //not guard data in mime email
            throw GuardPGPExceptionCodes.MIME_NO_GUARDPART_ERROR.create();
        }

        //Decrypting the OX Guard part of the mail message
        return decryptPart(pgpPart,userId,contextId,password,outputStream);

    }
}
