/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgp;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.security.SignatureException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.IOUtils;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.common.util.IDNUtil;
import com.openexchange.guard.common.util.ListUtil;
import com.openexchange.guard.common.util.LongUtil;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.keys.GuardKeyService;
import com.openexchange.guard.keys.PGPKeyService;
import com.openexchange.guard.keys.PGPPublicKeyRingFactory;
import com.openexchange.guard.keys.dao.GuardKeys;
import com.openexchange.guard.keys.dao.OGPGPKey;
import com.openexchange.guard.keys.dao.PGPKeys;
import com.openexchange.guard.keys.storage.OGPGPKeysStorage;
import com.openexchange.guard.keys.storage.PGPKeysStorage;
import com.openexchange.guard.mail.Attachment;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.ApiResponse;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.pgp.decryption.GuardEMailDecryptionService;
import com.openexchange.guard.pgp.decryption.PGPDecryptionResult;
import com.openexchange.guard.pgp.decryption.impl.GuardEMailDecryptionServiceImpl;
import com.openexchange.guard.pgp.parser.AsciiArmoredKeyParser;
import com.openexchange.guard.ratifier.GuardRatifierService;

public class PGPPublicHandler {

    public static final int MAX_KEY_FILE_UPLOAD_SIZE = 1000000; /* 1 MB */

    private static Logger LOG = LoggerFactory.getLogger(PGPPublicHandler.class);

    public void incomingPGPForm(HttpServletRequest request, HttpServletResponse response, int userid, int cid, OxCookie cookie) throws Exception {
        FileItemFactory fif = new DiskFileItemFactory();
        ServletFileUpload sfu = new ServletFileUpload(fif);
        List<FileItem> files = sfu.parseRequest(request);
        String emailRestriction = ServletUtils.getStringParameter(request, "emailaddress");  // Optional uid restriction
        //Check authentication
        Api ap = new Api(cookie, request);
        if (!ap.verifyLogin()) {
            LOG.info("User not logged into UI");
            ServletUtils.sendNotAcceptable(response, "Must be logged in to UI");
            return;
        }

        //Iterate through attachments
        Iterator<FileItem> iterator = files.iterator();
        while (iterator.hasNext()) {
            FileItem fi = iterator.next();

            //Check the file size
            if (fi.getSize() > MAX_KEY_FILE_UPLOAD_SIZE) {
                ServletUtils.sendNotAcceptable(response, "Key size exceeded");
                return;
            }

            //Parsing the input stream for public keys rings and using the factory to create them
            List<PGPPublicKeyRing> publicKeys = PGPPublicKeyRingFactory.create(new AsciiArmoredKeyParser().parse(fi.getInputStream()));

            if (publicKeys.size() > 0) {
                //If the client provided an email filter we only consider parsed public keys which userids are in the filter
                String[] emails = emailRestriction == null ? null : emailRestriction.split(",");
                if (emails != null) {
                    publicKeys = getKeysForEmail(publicKeys, Arrays.asList(emails));
                    if (publicKeys.size() == 0) {
                        LOG.debug("User email address not found in the uploaded public keys");
                        ServletUtils.sendNotAcceptable(response, "Bad UID");
                        return;
                    }
                }

                // Start response array
                StringBuilder responseSB = new StringBuilder();
                responseSB.append("[");

                //Saving all the parsed key-rings for the user
                int count = 0;
                for (PGPPublicKeyRing publicKey : publicKeys) {
                    //Saving each key
                    if (saveKeyData(publicKey, cid, userid)) {
                        //Convert the key to JSON and append to the response
                        JsonArray keydata = PgpHandler.getKeyInfo(publicKey);
                        responseSB.append(keydata.toString());
                        //Comma separated, if more than one key
                        if (count != publicKeys.size() - 1) {
                            responseSB.append(",");
                        }
                        count++;
                    } else {
                        LOG.warn("Unable to save uploaded public key data");
                        ServletUtils.sendNotAcceptable(response, "Problem importing key, possible bad data");
                        return;
                    }
                }
                responseSB.append("]");  // Close the array
                ServletUtils.sendHtmlOK(response, responseSB.toString());
            } else {
                LOG.info("Bad data");
                ServletUtils.sendNotAcceptable(response, "Bad data");
                return;
            }
        }
    }

    /**
     * Filters a list of public keys for uids
     *
     * @param publicKeys the list of keys to filter
     * @param emails the filter emails
     * @return those keys from publicKeys which uids are in emails
     */
    private List<PGPPublicKeyRing> getKeysForEmail(List<PGPPublicKeyRing> publicKeys, List<String> emails) {
        ArrayList<PGPPublicKeyRing> ret = new ArrayList<PGPPublicKeyRing>();

        for (PGPPublicKeyRing key : publicKeys) {
            Iterator<String> uidIterator = key.getPublicKey().getUserIDs();

            while (uidIterator.hasNext()) {
                String uid = uidIterator.next().toLowerCase();
                for (String email : emails) {
                    if (email.contains("@")) { // If email address, check if in userid
                        if (uid.contains(email.toLowerCase())) {
                            ret.add(key);
                        }
                    }
                }
            }
        }
        return ret;
    }

    private boolean saveKeyData(PGPPublicKeyRing ring, int cid, int userid) throws Exception {
        if (!ring.getPublicKeys().hasNext()) {
            return (false);  // If no public keys, return false
        }
        JsonArray keydata = PgpHandler.getKeyInfo(ring);
        JsonArray ids = new JsonArray();
        List<String> keyhexids = new ArrayList<String>();
        for (int i = 0; i < keydata.size(); i++) {
            JsonObject key = keydata.get(i).getAsJsonObject();
            JsonArray keyids = key.get("ids").getAsJsonArray();
            keyhexids.add(key.get("Key").getAsString());
            if (keyids.size() > 0) {
                ids = keyids;
            }
        }
        if (keyhexids.size() == 0) {
            return false;
        }
        boolean added = false;
        for (int j = 0; j < ids.size(); j++) {
            InternetAddress addr = parseEmailAdress(ids.get(j).getAsString());
            try {
                addr.validate();
            } catch (Exception e) {
                continue;
            }
            String email = IDNUtil.decodeEmail(addr.getAddress());

            //Adding the new uploaded key to the storage
            PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
            String asciiData = pgpKeyService.export(ring);
            OGPGPKeysStorage ogpgpKeysStorage = Services.getService(OGPGPKeysStorage.class);
            ogpgpKeysStorage.insertOrUpdate(userid, cid, email, keyhexids, asciiData);

            added = true;
        }
        return added;
    }

    private InternetAddress parseEmailAdress(String address) throws AddressException, UnsupportedEncodingException {
        try {
            if (address.contains("<")) {
                String name = address.substring(0, address.indexOf("<"));
                if (!name.contains("\"")) {
                    address = "\"" + name.trim() + "\" " + address.substring(address.indexOf("<"));
                }
            }
        } catch (Exception e) {
            LOG.error("Problem parsing address ", e);
        }
        InternetAddress[] recips = InternetAddress.parse(IDNUtil.aceEmail(address), false);
        if (recips.length > 0) {
            return new InternetAddress(recips[0].getAddress(), recips[0].getPersonal(), "UTF-8");
        }
        return null;
    }

    public void saveKeyAttachment(HttpServletRequest request, HttpServletResponse response, int userid, int cid, OxCookie ck) throws Exception {
        String folder = ServletUtils.getStringParameter(request, "folder", true);
        String emailid = ServletUtils.getStringParameter(request, "emailid", true);
        String attach = ServletUtils.getStringParameter(request, "attach", true);
        Api ap = new Api(ck, request);
        String session = request.getParameter("session");
        String userAgent = request.getHeader("User-Agent");
        if (!ap.verifyLogin()) {
            LOG.info("User not logged into UI");
            ServletUtils.sendNotAcceptable(response, "Must be logged in to UI");
            return;
        }
        PGPPublicKeyRing ring = null;
        String epass = ServletUtils.getStringParameter(request, "epass");
        String data = null;
        if (epass == null) {  // If not encrypted, retrieve plaintext
            try(ApiResponse plainAttachment = ap.getPlainAttachment(emailid, attach, folder)){
                Attachment att = new Attachment();
                att.setContent(plainAttachment.readContent());
                data = new String(att.getContent());
            }
        } else {
            // Handle encrypted public key
            String filename = ServletUtils.getStringParameter(request, "filename", true);
            boolean inline = ServletUtils.getBooleanParameter(request, "inline",false);
            String jSessionId = ck.getJSESSIONID();
            String password = PGPUtils.decodeEPass(epass, jSessionId);
            if (password == null) {
                ServletUtils.sendNotAcceptable(response, "bad password");
                return;
            }
            byte[] bytedata = null;
            if (inline) {  // pgp inline attachment

                //Get a Stream to the attachment
                Api api = new Api(ck, session, userAgent);
                Attachment attachment = new Attachment();
                try(ApiResponse plainAttachment = api.getPlainAttachment(emailid, attach, folder)){
                    attachment.setContent(plainAttachment.readContent());
                    InputStream attachmentStream = new ByteArrayInputStream(attachment.getContent());

                    ByteArrayOutputStream output = new ByteArrayOutputStream();
                    PGPUtils.decryptFile(attachmentStream, output, userid, cid, password, false);
                    output.close();
                    bytedata = output.toByteArray();
                }
            } else {  // pgp mime attachment
                GuardEMailDecryptionService decryptionService = new GuardEMailDecryptionServiceImpl();
                PGPDecryptionResult decryptionResult = decryptionService.decryptMimeAttachmentByName(ck, session, userAgent, folder, emailid, filename, userid, cid, password);
                bytedata = IOUtils.toByteArray(decryptionResult.getInputStream());
            }
            if (bytedata == null || bytedata.length < 2) {
                LOG.warn("Unable to decode inline pgp public key attachment");
                ServletUtils.sendNotAcceptable(response, "unable to decode");
                return;
            }
            data = new String(bytedata, "UTF-8");
        }
        ring = PGPPublicKeyRingFactory.create(data);
        if (ring == null) {
            LOG.info("Problem reading key data");
            ServletUtils.sendNotAcceptable(response, "Problem reading key data");
            return;
        }
        if (saveKeyData(ring, cid, userid)) {
            ServletUtils.sendHtmlOK(response, PgpHandler.getKeyInfo(ring).toString());
        } else {
            LOG.info("Problem saving key data");
            ServletUtils.sendNotAcceptable(response, "Problem saving");
        }

    }

    public void getPubKeyDetail(HttpServletRequest request, HttpServletResponse response, int userid, int cid) throws Exception {
        String ids = ServletUtils.getStringParameter(request, "ids", true).trim();
        String[] idList = ids.trim().split("\\s+");
        boolean guard = ServletUtils.getBooleanParameter(request, "guard",false);
        if (guard) {  // If we are to check the Guard tables, do it here
            JsonObject guardJson = getGuardPubKeyDetail(ids);
            if (guardJson != null) {
                guardJson.addProperty("guard", true);
                ServletUtils.sendHtmlOK(response, guardJson.toString());
                return;
            }
        }
        // Otherwise check uploaded
        OGPGPKeysStorage ogpgpKeysStorage = Services.getService(OGPGPKeysStorage.class);
        List<OGPGPKey> ogPGPKeys = ogpgpKeysStorage.getForUserByIds(userid, cid, Arrays.asList(idList));
        OGPGPKey ogPGPKey = null;
        //Getting the first key
        if (ogPGPKeys.size() > 0) {
            ogPGPKey = ogPGPKeys.get(0);
        }

        if (ogPGPKey != null) {
            PGPPublicKeyRing keyring = PGPPublicKeyRingFactory.create(ogPGPKey.getPublicPGPAscData());
            JsonArray keydata = PgpHandler.getKeyInfo(keyring);
            JsonObject retJson = new JsonObject();
            retJson.add("data", keydata);
            retJson.add("share", new JsonPrimitive(ogPGPKey.getShareLevel()));
            retJson.add("inline", new JsonPrimitive(ogPGPKey.isInline()));
            if (keydata.size() > 0) {
                ServletUtils.sendHtmlOK(response, retJson.toString());
                return;
            }
        }
        ServletUtils.sendNotAcceptable(response, "Fail");
    }

    private JsonObject getGuardPubKeyDetail(String id) throws OXException, SignatureException, PGPException {
        PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
        String[] ids = id.split(" ");
        for (String single_id : ids) {
            PGPPublicKeyRing keyring = pgpKeyService.getPublicKeyRingById(single_id, 0, 0);
            if (keyring != null) {
                JsonArray keydata = PgpHandler.getKeyInfo(keyring);
                JsonObject retJson = new JsonObject();
                retJson.add("data", keydata);
                if (keydata.size() > 0) {
                    return (retJson);
                }
            }
        }
        return (null);
    }

    public void getPubKeyList(HttpServletRequest request, HttpServletResponse response, int userid, int cid) throws Exception {
        StringBuilder searchlist = new StringBuilder();
        String emails = ServletUtils.getStringParameter(request, "emails");
        ArrayList<String> emailArray = new ArrayList<String>();
        if (emails != null) {
            String[] emailList = emails.split(",");
            for (String email : emailList) {
                if (!email.trim().equals("")) {
                    GuardRatifierService validatorService = Services.getService(GuardRatifierService.class);
                    validatorService.validate(email);

                    if (searchlist.length() == 0) {
                        searchlist.append("'" + dbformat(email) + "'");
                    } else {
                        searchlist.append(", '" + dbformat(email) + "'");
                    }
                    emailArray.add(email);
                }
            }
        }
        ServletUtils.sendHtmlOK(response, getPubKeyList(userid, cid, searchlist.toString(), emailArray).toString());
    }

    private String dbformat(String data) {
        if (data == null) {
            return data;
        }
        data = data.replace("'", "''");
        data = data.replace("`", "``");
        data = data.replace("\\", "\\\\");
        return data;
    }

    public JsonArray getPubKeyList(int userid, int cid, String searchlist, ArrayList<String> emailArray) throws Exception {
        OGPGPKeysStorage ogpgpKeysStorage = Services.getService(OGPGPKeysStorage.class);
        List<OGPGPKey> pgpKeys = (emailArray == null || emailArray.isEmpty()) ? ogpgpKeysStorage.getForUser(userid, cid) : ogpgpKeysStorage.getForUserByEmail(userid, cid, emailArray);

        JsonArray ret = new JsonArray();
        for (OGPGPKey pgpKey : pgpKeys) {
            JsonObject jsonKey = new JsonObject();
            jsonKey.addProperty("ids", ListUtil.listToString(pgpKey.getKeyIds(), ListUtil.WHITE_SPACE_SEP));
            jsonKey.addProperty("email", pgpKey.getEmail());
            jsonKey.addProperty("share", pgpKey.getShareLevel());
            jsonKey.addProperty("shared", (userid == pgpKey.getUserId() ? false : true));  // mark if this is a shared key
            jsonKey.addProperty("inline", pgpKey.isInline());
            ret.add(jsonKey);
        }

        if (emailArray.size() > 0) {
            GuardKeyService keyService = Services.getService(GuardKeyService.class);
            for (int i = 0; i < emailArray.size(); i++) {
                String email = emailArray.get(i);
                GuardKeys gkey = keyService.getKeysFromEmail(email);
                if (gkey != null) {
                    JsonArray info = PgpHandler.getKeyInfo(gkey.getPublicKeyRing());
                    StringBuilder idBuilder = new StringBuilder();
                    for (int j = 0; j < info.size(); j++) {
                        JsonObject keyinfo = (JsonObject) info.get(j);
                        if (j > 0) {
                            idBuilder.append(" ");
                        }
                        idBuilder.append(keyinfo.get("Key").getAsString());
                    }
                    JsonObject key = new JsonObject();
                    key.addProperty("ids", idBuilder.toString());
                    key.addProperty("email", email);
                    key.addProperty("share", false);
                    key.addProperty("guard", true);
                    key.addProperty("inline", gkey.isInline());
                    ret.add(key);
                }
            }
        }
        return (ret);
    }

    public void deleteKey(HttpServletRequest request, HttpServletResponse response, int userid, int cid) throws Exception {
        String ids = ServletUtils.getStringParameter(request, "ids", true);
        try {
            String[] idList = ids.trim().split("\\s+");
            OGPGPKeysStorage ogpgpKeysStorage = Services.getService(OGPGPKeysStorage.class);
            ogpgpKeysStorage.delete(userid, cid, Arrays.asList(idList));

            ServletUtils.sendHtmlOK(response, "OK");
        } catch (Exception ex) {
            LOG.error("Problem deleting public pgp key ", ex);
            ServletUtils.sendNotAcceptable(response, "Fail");
        }
    }

    public void shareKey(HttpServletRequest request, HttpServletResponse response, int userid, int cid) throws Exception {
        String ids = ServletUtils.getStringParameter(request, "ids", true);
        String[] idList = ids.trim().split("\\s+");
        String share = ServletUtils.getStringParameter(request, "share", true);
        int sharelevel = share.toLowerCase().trim().equals("true") ? 1 : 0;
        try {
            OGPGPKeysStorage ogpgpKeysStorage = Services.getService(OGPGPKeysStorage.class);
            ogpgpKeysStorage.updateShareLevel(userid, cid, Arrays.asList(idList), sharelevel);

            ServletUtils.sendHtmlOK(response, "OK");
        } catch (Exception ex) {
            LOG.error("Problem sharing public pgp key ", ex);
            ServletUtils.sendNotAcceptable(response, "Fail");
        }
    }

    public void changeInline(HttpServletRequest request, HttpServletResponse response, int userid, int cid) throws Exception {
        String ids = ServletUtils.getStringParameter(request, "ids", true);
        String[] idList = ids.trim().split("\\s+");
        String inlineStr = ServletUtils.getStringParameter(request, "inline", true);
        boolean isInline = inlineStr.toLowerCase().trim().equals("true") ? true : false;
        try {
            OGPGPKeysStorage ogpgpKeysStorage = Services.getService(OGPGPKeysStorage.class);
            ogpgpKeysStorage.updateInlineMode(userid, cid, Arrays.asList(idList), isInline);

            ServletUtils.sendHtmlOK(response, "OK");
        } catch (Exception ex) {
            LOG.error("Problem changing inline status public pgp key ", ex);
            ServletUtils.sendNotAcceptable(response, "Fail");
        }
    }

    public static GuardKeys getKeyForPGP(int id, int cid, long keyid) throws Exception {
        return (getKeyForPGP(id, cid, keyid, false));
    }

    /**
     * Return the Guard Keys populated with PGP data
     *
     * @param id
     * @param cid
     * @param keyid
     * @param incoming - true if current private key search only
     * @return
     * @throws OXException
     * @throws Exception
     */
    public static GuardKeys getKeyForPGP(int userid, int cid, long keyid, boolean incoming) throws OXException  {
        //TODO: Use a DI-Container for resolving
        PGPKeysStorage pgpKeysStorage = Services.getService(PGPKeysStorage.class);
        PGPKeys keyMapping = pgpKeysStorage.getByKeyIdInContext(cid, keyid);
        if (keyMapping != null) {
            GuardKeyService keyService = Services.getService(GuardKeyService.class);
            GuardKeys keys = keyService.getKeys(userid, cid, keyMapping.getKeyId(), !incoming);  // If incoming, we want all of the keys, not just the current
            return (keys);
        }
        if (incoming) {
            return (null);
        }
        return (getUploadedPublic(userid, cid, LongUtil.longToHexStringTruncated(keyid)));
    }

    public static GuardKeys getUploadedPublic(int userid, int cid, String pubid) throws OXException {
        OGPGPKeysStorage ogpgpKeysStorage = Services.getService(OGPGPKeysStorage.class);
        List<OGPGPKey> ogPGPKeys = ogpgpKeysStorage.getForUserByIds(userid, cid, Arrays.asList(new String[] { pubid }));
        for (OGPGPKey key : ogPGPKeys) {
            GuardKeys ret = new GuardKeys();
            ret.setEmail(key.getEmail());
            try {
                ret.setPublicKeyRing(PGPPublicKeyRingFactory.create(key.getPublicPGPAscData()));
            } catch (IOException e) {
                throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
            }
            ret.setContextid(cid);
            ret.setUserid(userid);
            if (ret.getUserid() == userid) {
                return ret;
            }
        }
        return null;
    }

}
