/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgp;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.security.PrivateKey;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.IOUtils;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPUserAttributeSubpacketVector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.common.util.JsonUtil;
import com.openexchange.guard.handler.ResponseHandler;
import com.openexchange.guard.internal.Auth;
import com.openexchange.guard.internal.GuardSettings;
import com.openexchange.guard.internal.Settings;
import com.openexchange.guard.internal.UserData;
import com.openexchange.guard.keys.GuardKeyService;
import com.openexchange.guard.keys.PGPKeyService;
import com.openexchange.guard.keys.dao.GuardKeys;
import com.openexchange.guard.keys.dao.PGPKeys;
import com.openexchange.guard.keys.storage.KeyTableStorage;
import com.openexchange.guard.keys.storage.PGPKeysStorage;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.pgp.exceptions.GuardPGPExceptionCodes;

public class PGPPrivateKeyHandler {

    private static Logger LOG = LoggerFactory.getLogger(PGPPrivateKeyHandler.class);

    public void incomingPGPForm(HttpServletRequest request, HttpServletResponse response, int userid, int cid, OxCookie cookie) throws Exception {
        FileItemFactory fif = new DiskFileItemFactory();
        ServletFileUpload sfu = new ServletFileUpload(fif);
        List<FileItem> files = sfu.parseRequest(request);
        Iterator<FileItem> iterator = files.iterator(); // Iterate through attachments
        String privdata = "";
        String pubdata = "";
        String pgppassword = "";
        String password = null;
        String language = ServletUtils.getStringParameter(request, "language");

        //Check authentication
        Api ap = new Api(cookie, request);
        if (!ap.verifyLogin()) {
            LOG.info("User not logged into UI");
            ServletUtils.sendNotAcceptable(response, "Must be logged in to UI");
            return;
        }

        while (iterator.hasNext()) {
            FileItem fi = iterator.next();
            String name = fi.getFieldName();
            if (name.equals("pgp")) {
                pgppassword = fi.getString("UTF-8");
                continue;
            }
            if (name.equals("password")) {
                password = fi.getString("UTF-8");
                if (password.equals("undefined")) {
                    password = null;
                }
                continue;
            }
            name = fi.getName();
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            InputStream input = fi.getInputStream();
            IOUtils.copy(input, out); // Copy stream to byte array
            String data = "\r\n" + new String(out.toByteArray()) + "\r\n";

            int start = data.indexOf("-----BEGIN PGP PRIV");
            if (start >= 0) {
                int end = data.indexOf("-----END PGP PRIVATE KEY BLOCK", start);
                end = data.indexOf("\r", end);
                privdata = data.substring(start, end);
            }
            if (data.contains("PUBLIC KEY")) {
                int pstart = data.indexOf("-----BEGIN PGP PUBLIC");
                int pend = data.indexOf("-----END PGP PUBLIC KEY BLOCK", pstart);
                pend = data.indexOf("\r", pend);
                pubdata = data.substring(pstart, pend);
            }
            out.close();
            input.close();
        }
        if (privdata.length() < 2 && pubdata.length() < 2) {
            LOG.info("Bad data");
            ServletUtils.sendNotAcceptable(response, "Bad data");
            return;
        }
        GuardKeys newkey = new GuardKeys();
        PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
        try { // If private key not empty, then import

            if (!privdata.equals("") && password != null && pgppassword != null) {
                try {
                    newkey = pgpKeyService.importPGPPrivateKey(newkey, privdata, pgppassword, password);
                } catch (OXException ex) {
                    ServletUtils.sendNotAcceptable(response, "Bad password");
                    return;
                }
            }
            if (newkey == null) {
                ServletUtils.sendNotAcceptable(response, "Failed to import private key");
                return;
            }
        } catch (Exception ex) {
            LOG.error("Unable to open pgp private file for inport ", ex);
            ServletUtils.sendNotAcceptable(response, "Bad pgp");
            return;
        }
        if (pubdata.equals("")) {
            PGPPublicKeyRing publicKeyRing = pgpKeyService.createPublicKeyRingFromPrivateKey(newkey.getPGPSecretKeyRing());
            newkey.setPublicKeyRing(publicKeyRing);

        } else {
            newkey.setPGPKeyRingFromAsc(pubdata);
            if (newkey.getPublicKeyRing() == null) {
                LOG.info("Problem with pub keys");
                ServletUtils.sendNotAcceptable(response, "Problem with pub keys");
                return;
            }
        }
        try {
            Api api = new Api(cookie, request);
            newkey.setEmail(api.getPrimary());
            newkey.setContextid(api.getContextId());
            newkey.setUserid(userid);
            newkey.setLanguage(language);
            //TODO:
            //version,question,answer,settings?

            // Mkae sure the IDs of the new public key are same as the user
            GuardKeyService keyService = Services.getService(GuardKeyService.class);
            if (verifyID(newkey)) {
                Long dupl;
                boolean duplicate = false;
                if ((dupl = isDuplicate(newkey)) != 0) {  // If public key already registered, possible duplicate
                    duplicate = true;
                }

                // Create recovery
                if (password != null) {
                    if (GuardSettings.checkCreateRecovery(cookie, request)) {
                        newkey.setRecovery(keyService.createRecovery(newkey, password));
                    }
                }

                if (!duplicate) {  // Not duplicate, add
                    keyService.addNewKey(newkey);
                } else {
                    GuardKeys existing = PGPPublicHandler.getKeyForPGP(userid, cid, dupl, true);

                    if (!PGPUtils.verifyPassword(password, existing.getSalt(), existing.getPGPSecretKeyRing())) {
                        ServletUtils.sendNotAcceptable(response, "Bad Merge Password");
                        return;
                    }

                    //////////// HANDLE MERGE/DUPLICATE public keys
                    if ((newkey.setPublicKeyRing(handleDuplicatePublic(newkey, existing, dupl))) == null) {
                        LOG.error("Problem handling duplicate public key");
                        ServletUtils.sendNotAcceptable(response, "duplicate error");
                        return;
                    }
                    if (newkey.getPGPSecretKeyRing() == null) {  // If overwriting public only, and we have a private for that key, keep it and only update PubRing
                        if (existing.getPGPSecretKeyRing() != null) {
                            KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
                            ogKeyTableStorage.updatePublicKey(newkey);
                        }
                    } else {
                        // Update the key
                        newkey = updateOldGuardKey(newkey, existing, password);
                        if (password != null) {  // RSA public key changed and need to rehash the recovery password
                            if (GuardSettings.checkCreateRecovery(cookie, request)) {
                                newkey.setRecovery(keyService.createRecovery(newkey, password));
                            }
                        }
                        pgpKeyService.updateDuplicatePGPKey(userid, cid, newkey, newkey.getRecovery(), dupl);
                    }
                }

            } else {
                LOG.info("Bad key id");
                ServletUtils.sendNotAcceptable(response, "Bad ID");
                return;
            }
        } catch (Exception e) {
            if (e.getMessage() != null) {
                if (e.getMessage().contains("Duplicate")) {
                    ServletUtils.sendNotAcceptable(response, "duplicate");
                    LOG.debug("Duplicate private key");
                    return;

                }
            }
            LOG.error("Error adding pgp private key ", e);
            ServletUtils.sendNotAcceptable(response, e.getMessage());
            return;
        }
        ServletUtils.sendHtmlOK(response, "OK");
    }

    /**
     * Verify the public key has the same email as the user
     *
     * @param gkey
     * @return
     */
    private boolean verifyID(GuardKeys gkey) {
        Iterator keys = gkey.getPublicKeyRing().getPublicKeys();
        PGPPublicKey key = null;
        while (keys.hasNext()) {
            key = (PGPPublicKey) keys.next();
            Iterator it = key.getUserIDs();
            int i = 0;
            while (it.hasNext()) {
                String id = (String) it.next();
                if (id.toLowerCase().contains(gkey.getEmail().toLowerCase())) {
                    return (true);
                }
            }
        }
        return (false);
    }

    /**
     * Handle an upload of a public key that already exists
     *
     * @param gkey
     * @param dupl
     * @return
     */

    public PGPPublicKeyRing handleDuplicatePublic(GuardKeys gkey, GuardKeys existing, long dupl) {

        try {
            Iterator<PGPPublicKey> keys = gkey.getPublicKeyRing().getPublicKeys();
            // get the guardkeys that has the duplicated public key

            while (keys.hasNext()) {
                PGPPublicKey keyToImport = keys.next();
                long importID = keyToImport.getKeyID();
                PGPPublicKey toMerge = existing.getPublicKeyRing().getPublicKey(importID);
                if (toMerge != null) {
                    existing.setPublicKeyRing(PGPPublicKeyRing.removePublicKey(existing.getPublicKeyRing(), toMerge));
                    toMerge = mergeKeys(toMerge, keyToImport);
                    if (toMerge == null) {
                        return null;
                    }
                    existing.setPublicKeyRing(PGPPublicKeyRing.insertPublicKey(existing.getPublicKeyRing(), toMerge));
                } else {
                    existing.setPublicKeyRing(PGPPublicKeyRing.insertPublicKey(existing.getPublicKeyRing(), keyToImport));
                }
            }
            PGPKeysStorage keysStorage = Services.getService(PGPKeysStorage.class);
            keysStorage.addPublicKeyIndex(existing.getContextid(), existing.getKeyid(), existing.getEmail(), existing.getPublicKeyRing());
            return existing.getPublicKeyRing();
        } catch (Exception e) {
            LOG.error("Problem merging duplicate public keys, ", e);
            return existing.getPublicKeyRing();
        }
    }

    /**
     * Since the salt changes, we need to re-encrypt the old Guard proprietary priv key.
     *
     * @param newkey
     * @param existing
     * @param password
     * @return
     */
    private GuardKeys updateOldGuardKey(GuardKeys newkey, GuardKeys existing, String password) {
        if (existing.privKeyNull()) {
            return (newkey);
        }
        try {
            GuardKeyService guardKeyService = Services.getService(GuardKeyService.class);
            PrivateKey privateKey = guardKeyService.decryptPrivateKey(existing.getEncodedPrivate(), password, existing.getSalt(), true);
            String encryptedPK = guardKeyService.encryptPrivateKey(privateKey, password, existing.getSalt());
            newkey.setEncodedPrivate(encryptedPK);
            newkey.setPublic(existing.getPublicKey());
        } catch (Exception ex) {
            LOG.error("Problem copying over old Guard Password ", ex);
            return (newkey);
        }
        return (newkey);
    }

    /**
     * Delete the private/public key pair for the user
     *
     * @param request
     * @param response
     * @param ck
     * @throws Exception
     */
    public void DeletePrivateKey(HttpServletRequest request, HttpServletResponse response, int userid, int cid, OxCookie ck) throws Exception {
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
        Api ap = new Api(ck, request);
        if (!ap.verifyLogin()) {
            ServletUtils.sendNotAcceptable(response, "Must be logged in");
            LOG.info("Delete private key...not logged in");
            return;
        }
        Settings settings = new Auth().getPermissions(ck, request);
        if (settings.isNoDeletePrivate()) {
            ServletUtils.sendNotAcceptable(response, "Function disabled by configuration");
            return;
        }
        String keystring = json.get("keyid").getAsString();
        String auth;
        {
            JsonElement authElement = json.get("auth");
            if (authElement == null) {
                LOG.error("The 'auth' is missing from the JSON payload");
                ServletUtils.sendNotAcceptable(response, "The 'auth' is missing from the JSON payload");
                return;
            }
            auth = authElement.getAsString();
        }

        long keyid = 0l;
        try {
            keyid = Long.parseLong(keystring);
        } catch (Exception e) {
            LOG.error("Bad keyid");
            ServletUtils.sendNotAcceptable(response, "Bad keyid");
            return;
        }
        // Verify auth
        UserData data = new UserData(auth, ck);
        if (data.getUserid() < 0) {
            ServletUtils.sendNotAcceptable(response, "Bad password");
            LOG.info("Bad password");
            return;
        }
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        GuardKeys key = keyService.getKeys(userid, cid, keyid, false);
        // Make sure right password for this key
        if (!PGPUtils.verifyPassword(data.getEncr_password(), key.getSalt(), key.getPGPSecretKeyRing())) {
            ServletUtils.sendNotAcceptable(response, "Bad password");
            LOG.info("Bad password");
            return;
        }
        boolean iscurrent = key.isCurrent();

        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        ogKeyTableStorage.delete(key);

        PGPKeysStorage pgpKeysStorage = Services.getService(PGPKeysStorage.class);
        pgpKeysStorage.deleteByKeyId(keyid);

        //if key was the current key, then upgrade highest version key to current
        if (iscurrent) {
            resetCurrentKey(userid, cid);
        }
        ServletUtils.sendHtmlOK(response, "OK");
        LOG.debug("Success deleting private key");

    }

    /**
     * Resets the "current" key to the key which has the highest version
     *
     * @param userid the user's id
     * @param cid the context id
     * @throws Exception
     */
    private void resetCurrentKey(int userid, int cid) throws Exception {
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);

        //Getting the keys with the highest version from the user, and mark it as the "current" key
        GuardKeys highestKey = ogKeyTableStorage.getHighestVersionKeyForUser(userid, cid);
        if (highestKey != null) {
            ogKeyTableStorage.setCurrentKey(highestKey);
        }
    }

    /**
     * Revoke a private key for the user
     *
     * @param request
     * @param response
     * @param ck
     * @throws Exception
     */
    public void revokePrivateKey(HttpServletRequest request, HttpServletResponse response, int userid, int cid, OxCookie ck) throws Exception {
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
        Api ap = new Api(ck, request);
        if (!ap.verifyLogin()) {
            ServletUtils.sendNotAcceptable(response, "Must be logged in");
            LOG.info("Delete private key...not logged in");
            return;
        }
        String keystring = json.get("keyid").getAsString();
        String auth = json.get("auth").getAsString();
        String reason = json.get("reason").getAsString();
        long keyid = 0l;
        try {
            keyid = Long.parseLong(keystring);
        } catch (Exception e) {
            LOG.error("Bad keyid");
            ServletUtils.sendNotAcceptable(response, "Bad keyid");
            return;
        }
        // Verify auth
        UserData data = new UserData(auth, ck);
        if (data.getUserid() < 0) {
            ServletUtils.sendNotAcceptable(response, "Bad password");
            LOG.info("Bad password");
            return;
        }
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        GuardKeys key = keyService.getKeys(userid, cid, keyid, false);
        // Make sure right password for this key
        if (!PGPUtils.verifyPassword(data.getEncr_password(), key.getSalt(), key.getPGPSecretKeyRing())) {
            ServletUtils.sendNotAcceptable(response, "Bad password");
            LOG.info("Bad password");
            return;
        }
        PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
        key = pgpKeyService.revokePGPPublicKeys(key, data.getEncr_password(), reason);

        KeyTableStorage keyTableStorage = Services.getService(KeyTableStorage.class);
        keyTableStorage.updatePublicKey(key);

        ServletUtils.sendHtmlOK(response, "OK");
        LOG.info("Success deleting private key");

    }

    /**
     * Delete a local public key only that was uploaded by the user for his/her use
     * Private key must not exist
     *
     * @param request
     * @param response
     * @param ck
     * @throws Exception
     */
    public void deleteLocalPublic(HttpServletRequest request, HttpServletResponse response, int userid, int cid, OxCookie ck) throws Exception {
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
        Api ap = new Api(ck, request);
        if (!ap.verifyLogin()) {
            ServletUtils.sendNotAcceptable(response, "Not logged in");
            LOG.info("Fail, not logged in");
            return;
        }
        String keystring = JsonUtil.getStringFromJson(json, "keyid", true);
        long keyid = 0l;
        try {
            keyid = Long.parseLong(keystring);
        } catch (Exception ex) {
            ServletUtils.sendNotAcceptable(response, "Bad keyid");
            LOG.error("Bad keyid");
            return;
        }

        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        GuardKeys key = keyService.getKeys(userid, cid, keyid, false);
        if (key == null) {
            ServletUtils.sendNotAcceptable(response, "Unable to find key");
            LOG.info("Fail, unable to find key");
            return;
        }
        if (key.getPGPSecretKey() != null) {
            ServletUtils.sendNotAcceptable(response, "Private key exists");
            LOG.error("Attempt to delete public when private key exists");
            return;
        }

        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        ogKeyTableStorage.delete(key);

        PGPKeysStorage pgpKeysStorage = Services.getService(PGPKeysStorage.class);
        pgpKeysStorage.deleteByKeyIdInContext(keyid, cid);

        //if key was the current key, then upgrade highest version key to current:
        if (key.isCurrent()) {
            resetCurrentKey(userid, cid);
        }
    }

    /**
     * Add/Merge certifications of two public keys
     *
     * @param old
     * @param newKey
     * @return
     * @throws PGPException
     */
    private PGPPublicKey mergeKeys(PGPPublicKey old, PGPPublicKey newKey) throws PGPException {
        if (old.getKeyID() != newKey.getKeyID()) {
            LOG.error("Unable to merge keys, different keyids ");
            return (null);
        }
        ArrayList<PGPSignature> oldsigs = new ArrayList<PGPSignature>();
        Iterator oldsigI = old.getSignatures();
        while (oldsigI.hasNext()) {
            oldsigs.add((PGPSignature) oldsigI.next());
        }

        Iterator<String> userids = old.getUserIDs();
        // Loop through userids of the old certificate and see if there are any new signatures
        while (userids.hasNext()) {
            String uid = userids.next();
            Iterator<PGPSignature> newsigs = newKey.getSignaturesForID(uid);
            if (newsigs != null) {
                while (newsigs.hasNext()) {
                    PGPSignature newsig = newsigs.next();
                    if (!hasSig(newsig, oldsigs)) {
                        old = PGPPublicKey.addCertification(old, uid, newsig);
                    }
                }
            }

        }
        userids = newKey.getUserIDs();
        // Loop through userids of the NEW certificate and see if there are any new signatures
        while (userids.hasNext()) {
            String uid = userids.next();
            Iterator<PGPSignature> newsigs = newKey.getSignaturesForID(uid);
            while (newsigs.hasNext()) {
                PGPSignature newsig = newsigs.next();
                if (!hasSig(newsig, oldsigs)) {
                    old = PGPPublicKey.addCertification(old, uid, newsig);
                }
            }
        }
        // Loop through user attributes and see if any new signatures
        Iterator<PGPUserAttributeSubpacketVector> vectors = old.getUserAttributes();
        while (vectors.hasNext()) {
            PGPUserAttributeSubpacketVector vector = vectors.next();
            Iterator<PGPSignature> newsigs = newKey.getSignaturesForUserAttribute(vector);
            while (newsigs.hasNext()) {
                PGPSignature newsig = newsigs.next();
                if (!hasSig(newsig, oldsigs)) {
                    old = PGPPublicKey.addCertification(old, vector, newsig);
                }
            }
        }
        // Loop through user attributes and see if any new signatures
        vectors = newKey.getUserAttributes();
        while (vectors.hasNext()) {
            PGPUserAttributeSubpacketVector vector = vectors.next();
            Iterator<PGPSignature> newsigs = newKey.getSignaturesForUserAttribute(vector);
            if (newsigs != null) {
                while (newsigs.hasNext()) {
                    PGPSignature newsig = newsigs.next();
                    if (!hasSig(newsig, oldsigs)) {
                        old = PGPPublicKey.addCertification(old, vector, newsig);
                    }
                }
            }
        }

        // Check for revoke signatures
        Iterator<PGPSignature> newsigs = newKey.getSignaturesOfType(PGPSignature.KEY_REVOCATION);
        while (newsigs.hasNext()) {
            PGPSignature newsig = newsigs.next();
            if (!hasSig(newsig, oldsigs)) {
                old = PGPPublicKey.addCertification(old, newsig);
            }
        }
        return (old);
    }

    private boolean hasSig(PGPSignature sig, ArrayList<PGPSignature> sigs) throws PGPException {
        for (int i = 0; i < sigs.size(); i++) {
            if (Arrays.equals(sig.getSignature(), sigs.get(i).getSignature())) {
                return (true);
            }
        }
        return (false);
    }

    /**
     * Check if public key already exists in our index
     *
     * @param gkey
     * @return 0 if not duplicate, otherwise keyid of the duplicate key
     * @throws Exception
     */

    public long isDuplicate(GuardKeys gkey) throws Exception {
        List<Long> ids = new ArrayList<Long>();
        Iterator<PGPPublicKey> keys = gkey.getPublicKeyRing().getPublicKeys();
        while (keys.hasNext()) {
            ids.add(keys.next().getKeyID());
        }

        //TODO: Use a DI Container for resolving this
        PGPKeysStorage pgpKeysStorage = Services.getService(PGPKeysStorage.class);
        List<PGPKeys> keyMappings = pgpKeysStorage.getByIds(ids);
        if (keyMappings.size() > 0) {
            return keyMappings.get(0).getKeyId();
        }
        return 0;

    }

    /**
     * Make the keyid the current key
     *
     * @param id
     * @param cid
     * @param keyid
     * @return
     * @throws Exception
     */
    public static void makeCurrent(int id, int cid, String keyid) throws Exception {
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);

        //Getting the key which should be marked as "current"
        GuardKeys key = ogKeyTableStorage.getKeyForUserById(id, cid, keyid);
        if (key != null) {
            //Removing the current flag from all keys
            ogKeyTableStorage.unsetCurrentFlag(id, cid);

            //Setting the current flag to the key
            ogKeyTableStorage.setCurrentKey(key);
        } else {
            throw new IllegalArgumentException("Key Not Found");
        }
    }

    /**
     * Make a selected Private/Public key pair the current one to be used.
     *
     * @param request
     * @param response
     * @param ck
     * @throws Exception
     */
    public void makeCurrent(HttpServletRequest request, HttpServletResponse response, int userid, int cid, OxCookie ck) throws Exception {
        Api ap = new Api(ck, request);
        if (!ap.verifyLogin()) {
            ServletUtils.sendNotAcceptable(response, "Must be logged in");
            return;
        }

        String keyid = ServletUtils.getStringParameter(request, "keyid", true);
        makeCurrent(userid, cid, keyid);
    }

    public void addUID(HttpServletRequest request, HttpServletResponse response, OxCookie ck, int userid, int cid) throws Exception {
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request);
        Api ap = new Api(ck, request);
        if (!ap.verifyLogin()) {
            ServletUtils.sendNotAcceptable(response, "Must be logged in");
            return;
        }
        String keyid = ServletUtils.getStringParameter(request, "keyid", true);
        String name = JsonUtil.getStringFromJson(json, "name", true);
        String email = JsonUtil.getStringFromJson(json, "email", true);
        String password = JsonUtil.getStringFromJson(json, "password", true);

        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        GuardKeys keys = keyService.getKeys(userid, cid, Long.parseLong(keyid), false);
        if (keys == null) {
            throw GuardPGPExceptionCodes.KEY_NOT_FOUND.create(keyid);
        }

        PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
        PGPPrivateKey pkey;
        pkey = pgpKeyService.getPGPSignKey(keys.getPGPSecretKeyRing(), password, keys.getSalt());

        String uid = name + " <" + email + ">";
        PGPPublicKeyRing newring = pgpKeyService.addUID(pkey, keys.getPublicKeyRing(), uid);
        if (newring == null) {
            ServletUtils.sendNotAcceptable(response, "error adding");
            return;
        }
        keys.setPublicKeyRing(newring);
        pgpKeyService.updatePGPKeys(userid, cid, keys);
        ServletUtils.sendHtmlOK(response, "OK");
    }

}
