/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.drive;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.FileItemIterator;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.lang.time.StopWatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.common.util.JsonUtil;
import com.openexchange.guard.drive.exceptions.GuardDriveMessages;
import com.openexchange.guard.internal.Auth;
import com.openexchange.guard.internal.Settings;
import com.openexchange.guard.internal.Settings.permissions;
import com.openexchange.guard.internal.UserData;
import com.openexchange.guard.keys.GuardKeyService;
import com.openexchange.guard.keys.dao.GuardKeys;
import com.openexchange.guard.keys.dao.RecipKey;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.ox.streaming.StreamHandlerFactory;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.oxapi.streaming.FileItemInputStream;
import com.openexchange.guard.oxapi.streaming.GuardStreamUtils;
import com.openexchange.guard.oxapi.streaming.JsonStreamHandler;
import com.openexchange.guard.oxapi.streaming.StreamHandler;
import com.openexchange.guard.translation.GuardTranslationService;

public class FileHandler {

    private static Logger LOG = LoggerFactory.getLogger(FileHandler.class);

    /**
     * Handling drive file uploads from a client
     *
     * @param request The request sent by the client
     * @param response The response
     * @param cookie the cookie
     * @param update Whether the file replaces an existing file or is a new file
     * @throws IOException
     */
    public void encryptUpload(HttpServletRequest request, HttpServletResponse response, OxCookie cookie, int userid, int cid, boolean update) throws Exception {
        //Checking if the user is allowed to use files
        if (!new Auth().checkPermission(cookie, request, Settings.permissions.FILE)) {
            LOG.error("not authorized");
            ServletUtils.sendNotAcceptable(response, "not authorized");
            return;
        }

        String sessionID = ServletUtils.getStringParameter(request, "session", true);
        Api api = new Api(cookie, sessionID, request.getHeader("User-Agent"));
        FileItemFactory fileItemFactory = new DiskFileItemFactory();
        ServletFileUpload servletFileUpload = new ServletFileUpload(fileItemFactory);
        String fileName = null;
        String contentType = null;
        FileItemIterator fileItemIterator = servletFileUpload.getItemIterator(request);

        try (FileItemInputStream uploadStream = new FileItemInputStream("file", fileItemIterator); FileItemInputStream jsonStream = new FileItemInputStream("json", fileItemIterator);) {
            //Getting the file name of the file to upload
            //NOTE: This opens the stream of the first item;
            //      This is OK since it is the first item and therefore
            //      we'll not skip something
            fileName = uploadStream.getItemName();
            contentType = uploadStream.getContentType();

            //Getting the user's key for encrypting the file
            //TODO: What happens when uploading a file to a shared folder?
            //      Encrypting the file for more recipients?
            //      We should move this code in it's own class then
            GuardKeyService keyService = Services.getService(GuardKeyService.class);
            GuardKeys userKeys = keyService.getKeys(userid, cid);
            if (userKeys == null) {
                throw new Exception(String.format("Key not found for userid %d within context %d", userid, cid));
            }

            //Adding the user to the list of recipients
            ArrayList<RecipKey> recipients = new ArrayList<RecipKey>();
            recipients.add(new RecipKey(userKeys));

            //Creating a Json handler which adds and additional
            //meta-data object to the Json stream during later
            //processing
            JsonObject meta = new JsonObject();
            meta.addProperty("Encrypted", true);
            meta.addProperty("OwnerCid", cid);
            meta.addProperty("OwnerId", userid);
            meta.addProperty("OrigMime", contentType);
            JsonStreamHandler jsonHandler = new JsonStreamHandler("meta", meta);

            //encrypting the input stream of the upload item
            LOG.debug("Encrypting new upload ...");
            StopWatch sw = new StopWatch();
            sw.start();
            FileEncrypter encryptor = new FileEncrypter(api);
            JsonObject ret;
            if (update) {
                String fileId = ServletUtils.getStringParameter(request, "id", true);
                ret = encryptor.encryptUpdate(uploadStream, fileName, fileId, contentType, jsonStream, jsonHandler, recipients);
            } else {
                ret = encryptor.encryptNew(uploadStream, fileName, contentType, jsonStream, jsonHandler, recipients);
            }
            sw.stop();
            LOG.debug(String.format("Encrypting done for new upload [Time: %d ms]", sw.getTime()));

            //Sending response
            sendOK(response, ret.toString());
        }
    }

    private String getFileJson(String itemId, String mime, String folder, String description, int id, int cid) {
        Gson gson = new GsonBuilder().create();
        JsonObject json = new JsonObject();
        JsonObject meta = new JsonObject();
        meta.addProperty("OGID", itemId);
        meta.addProperty("Encrypted", true);
        meta.addProperty("OwnerCid", cid);
        meta.addProperty("OwnerId", id);
        meta.addProperty("OrigMime", mime);
        json.add("folder_id", new JsonPrimitive(folder));
        json.addProperty("description", description);
        json.add("meta", meta);
        String data = gson.toJson(json);
        return (data);
    }

    public void getFile(HttpServletRequest request, HttpServletResponse response, int userId, int cid, OxCookie cookie) throws Exception {
        String fileId = ServletUtils.getStringParameter(request, "id", true);
        String folderId = ServletUtils.getStringParameter(request, "folder", true);
        String version = ServletUtils.getStringParameter(request, "version");
        String auth = ServletUtils.getStringParameter(request, "auth");
        String lang = ServletUtils.getStringParameter(request, "lang");
        String fileName = ServletUtils.getStringParameter(request, "filename", true);
        boolean download = ServletUtils.getBooleanParameter(request, "download", false);

        //Get further meta data from the object to decrypt
        Api ap = new Api(cookie, request);
        String data = ap.getInfoItemJson(fileId, folderId, version);
        if (!JsonUtil.getError(data).equals("")) {
            throw new Exception(JsonUtil.getError(data));
        }
        Gson gson = new Gson();
        JsonObject json = gson.fromJson(data, JsonObject.class);
        JsonObject jdata = json.get("data").getAsJsonObject();
        JsonObject meta = jdata.get("meta").getAsJsonObject();
        long fileSize = jdata.get("file_size").getAsLong();
        String fileType = jdata.get("file_mimetype").getAsString();
        if (meta.has("OrigMime")) {  // If we stored the original mime type in meta, use this
            fileType = meta.get("OrigMime").getAsString();
        }
        //Authentication
        UserData authData;
        if (auth.length() < 20) {
            String password = ServletUtils.getStringParameter(request, "password", true);
            authData = new UserData();
            authData.setEncr_password(password);
            authData.setUserid(userId);
        } else {
            authData = new UserData(auth, cookie);
        }
        if (authData.getUserid() == -1) {
            GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
            sendFail(response, translationService.getTranslation(GuardDriveMessages.UNABLE_TO_DECODE_BAD_AUTH, lang));
            return;
        }

        //Creating an appropriated stream handler for file decryption
        StreamHandler streamHandler = StreamHandlerFactory.createStreamHandler(fileName, userId, cid, authData.getEncr_password(), cookie.getJSESSIONID());
        //Preparing response
        fileName = fileName.replace(".pgp", "").replace(".grd2", "").replace(".grd", "");
        String newContentType = streamHandler.getNewContentType(fileType, fileName);
        response.setContentType(newContentType);
        response.addHeader("Access-Control-Allow-Origin", "*");
        if (download) {
            response.setHeader("Content-Disposition", "attachment; filename=\"" + fileName + "\"");
        }

        OutputStream outputStream = response.getOutputStream();
        try (InputStream inputStream = ap.openDriveItem(fileId, fileSize, folderId, version)) {
            //Just writing the input stream to the output stream of the response.
            //Encryption is done one the fly by the streamHandler
            GuardStreamUtils.Copy(inputStream, outputStream, streamHandler);
        } catch (IOException ex) {
            if (ex.getMessage() != null) {
                // TODO will likely need translation
                if (ex.getMessage().startsWith("No secret key")) {
                    outputStream.write(ex.getMessage().getBytes());
                    response.setStatus(HttpServletResponse.SC_OK);
                    return;
                }
            }
            throw ex;
        }

        response.setStatus(HttpServletResponse.SC_OK);
    }

    /**
     * Send answer to response stream
     *
     * @param response
     * @param answer String to put in HTML
     */
    private void sendOK(HttpServletResponse response, String answer) {
        response.setContentType("text/html");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        try {
            response.getWriter().println(answer);
        } catch (IOException e) {
            LOG.error("Error while writing 'OK' answer to response", e);
        }
    }

    private void sendFail(HttpServletResponse response, String answer) {
        response.setContentType("text/html");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_NOT_ACCEPTABLE);
        try {
            response.getWriter().println(answer);
        } catch (IOException e) {
            LOG.error("Error while writing 'fail' answer to response", e);
        }

    }

    public void encryptFile(HttpServletRequest request, HttpServletResponse response, OxCookie cookie, int userid, int cid) throws Exception {
        String id = ServletUtils.getStringParameter(request, "id", true);
        String folder = ServletUtils.getStringParameter(request, "folder", true);
        String version = ServletUtils.getStringParameter(request, "version");
        if (!new Auth().checkPermission(cookie, request, permissions.FILE)) {
            LOG.error("not authorized");
            ServletUtils.sendNotAcceptable(response, "not authorized");
            return;
        }

        //Get meta data from the file to encrypt
        Api api = new Api(cookie, request);
        String data = api.getInfoItemJson(id, folder, version);
        Gson gson = new Gson();
        JsonObject json = gson.fromJson(data, JsonObject.class);
        if (json.has("error")) {
            throw new Exception(json.get("error").getAsString());
        }
        JsonObject jdata = json.get("data").getAsJsonObject();
        long fileSize = jdata.get("file_size").getAsLong();
        String fileName = jdata.get("filename").getAsString();
        String fileType = jdata.get("file_mimetype").getAsString();

        //Getting the user's key
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        GuardKeys userKeys = keyService.getKeys(userid, cid);
        if (userKeys == null) {
            throw new Exception(String.format("Key not found for userid %d within context %d", userid, cid));
        }

        //Adding the user to the list of recipients
        ArrayList<RecipKey> recipients = new ArrayList<RecipKey>();
        recipients.add(new RecipKey(userKeys));

        //populating new JSON meta data
        String itemID = CipherUtil.getUUID();
        String description = (jdata.get("description") == null) ? "" : jdata.get("description").getAsString();
        String newjson = getFileJson(itemID, fileType, folder, description, userid, cid);
        JsonStreamHandler jsonHandler = new JsonStreamHandler();
        LOG.debug(String.format("Starting encryption for file %s in folder %s ...", id, folder));
        StopWatch sw = new StopWatch();
        sw.start();

        try (InputStream jsonStream = new ByteArrayInputStream(newjson.getBytes()); InputStream remoteInputStream = api.openDriveItem(id, fileSize, folder, version)) {
            // TODO: Make the FileEncrypted and FileDecrypter OSGi services
            FileEncrypter fileEncrypter = new FileEncrypter(api);
            //Encrypting the item
            JsonObject retJson = fileEncrypter.encryptNew(remoteInputStream, fileName, fileType, jsonStream, jsonHandler, recipients);
            sw.stop();
            long ms = sw.getTime();
            LOG.debug(String.format("Encryption done for file %s in folder %s [Time: %d ms]", id, folder, ms));

            //deleting the old item
            api.deleteItem(id, folder);

            //sending response
            sendOK(response, retJson.toString());
        }
    }

    /**
     * Removes the encryption from a drive file
     *
     * @param request The request sent by a client
     * @param response The response
     * @param cookie The cookie data
     * @throws GuardMissingParameter If the requests was not correct and complete
     */
    public void removeEncryption(HttpServletRequest request, HttpServletResponse response, int userid, int cid, OxCookie cookie) throws Exception {

        //parse parameters
        String fileId = ServletUtils.getStringParameter(request, "id", true);
        String folderId = ServletUtils.getStringParameter(request, "folder", true);
        String version = ServletUtils.getStringParameter(request, "version");
        String auth = ServletUtils.getStringParameter(request, "auth");
        String lang = ServletUtils.getStringParameter(request, "language");
        String fileName = ServletUtils.getStringParameter(request, "filename", true);

        //Get further meta data from the object to decrypt
        Api ap = new Api(cookie, request);
        String data = ap.getInfoItemJson(fileId, folderId, version);
        Gson gson = new Gson();
        JsonObject json = gson.fromJson(data, JsonObject.class);
        JsonObject jdata = json.get("data").getAsJsonObject();
        long fileSize = jdata.get("file_size").getAsLong();
        String fileType = jdata.get("file_mimetype").getAsString();

        //Authentication
        //TODO: Do we have to check for permission here, like in encrypt file??
        UserData authData;
        if (auth.length() < 20) {
            String password = ServletUtils.getStringParameter(request, "password", true);
            authData = new UserData();
            authData.setEncr_password(password);
            authData.setUserid(userid);
        } else {
            authData = new UserData(auth, cookie);
        }
        if (authData.getUserid() == -1) {
            GuardTranslationService translationService = Services.getService(GuardTranslationService.class);
            sendFail(response, translationService.getTranslation(GuardDriveMessages.UNABLE_TO_DECODE_BAD_AUTH, lang));
            return;
        }

        //New JSON
        String description = (jdata.get("description") == null) ? "" : jdata.get("description").getAsString();
        JsonObject newJsonContent = new JsonObject();
        newJsonContent.add("folder_id", new JsonPrimitive(folderId));
        newJsonContent.addProperty("description", description);
        String newjson = newJsonContent.toString();
        JsonStreamHandler jsonHandler = new JsonStreamHandler();

        try (InputStream jsonStream = new ByteArrayInputStream(newjson.getBytes())) {
            //De-Crypting the item
            LOG.debug(String.format("Decrypting file %s in folder %s ...", fileId, folderId));
            StopWatch sw = new StopWatch();
            sw.start();
            FileDecrypter fileDecrypter = new FileDecrypter(ap);

            JsonObject retJson = fileDecrypter.decrypt(fileId, fileSize, folderId, version, fileName, userid, cid, authData.getEncr_password(), fileType, jsonStream, jsonHandler);
            sw.stop();
            long ms = sw.getTime();
            LOG.debug(String.format("Decryption done for file %s in folder %s [Time: %d ms]", fileId, folderId, ms));

            //deleting the old file
            ap.deleteItem(fileId, folderId);

            //sending response
            sendOK(response, retJson.toString());
        }
    }

    public void getfolder(HttpServletRequest request, HttpServletResponse response, OxCookie oxCookie) {
        try {
            Api ap = new Api(oxCookie, request);
            String data = ap.getFolder(Integer.parseInt(request.getParameter("id")));
            if (!data.contains("error")) {
                ServletUtils.sendHtmlOK(response, data);
                return;
            }

        } catch (Exception ex) {
            LOG.error("Error getting folder " + request.getParameter("id"), ex);
        }
        ServletUtils.sendNotAcceptable(response, "error");
    }

    public void addFolderShare(HttpServletRequest request, HttpServletResponse response, OxCookie oxCookie) {
        try {
            Api ap = new Api(oxCookie, request);
            int id = Integer.parseInt(request.getParameter("id"));
            String data = ap.getFolder(id);
            if (data.contains("error")) {
                ServletUtils.sendNotAcceptable(response, "error retrieving folder perm");
                return;
            }
            String[] users = request.getParameter("users").split(",");
            JsonArray permissions = getPermissions(data);
            for (int i = 0; i < users.length; i++) {
                int user = Integer.parseInt(users[i]);
                boolean found = false;
                permloop: for (int j = 0; j < permissions.size(); j++) {
                    if (permissions.get(j).getAsJsonObject().get("entity").getAsInt() == user) {
                        // update
                        int bits = permissions.get(j).getAsJsonObject().get("bits").getAsInt();
                        bits = bits | 257;
                        JsonObject perm = permissions.get(j).getAsJsonObject();
                        perm.addProperty("bits", bits);
                        JsonArray newpermissions = new JsonArray();
                        for (int c = 0; c < permissions.size(); c++) {
                            if (c != j) {
                                newpermissions.add(permissions.get(c).getAsJsonObject());
                            }
                        }
                        newpermissions.add(perm);
                        permissions = newpermissions;
                        found = true;
                        break permloop;
                    }
                }
                if (!found) {
                    JsonObject newperm = new JsonObject();
                    newperm.addProperty("entity", user);
                    newperm.addProperty("bits", 257);
                    newperm.addProperty("group", false);
                    permissions.add(newperm);
                }
            }
            Gson gson = new Gson();
            JsonObject newdata = new JsonObject();
            newdata.add("permissions", permissions);
            String addresult = ap.updateFolder(id, gson.toJson(newdata));
            if (addresult.contains("error")) {
                LOG.error("failed to add share");
                LOG.error(addresult);
                ServletUtils.sendNotAcceptable(response, "error adding folder perm");
                return;
            }
            ServletUtils.sendHtmlOK(response, "OK");
            return;
        } catch (Exception ex) {
            LOG.error("Error adding folder share", ex);
        }
        ServletUtils.sendNotAcceptable(response, "error");
    }

    private JsonArray getPermissions(String data) {
        Gson gson = new Gson();
        JsonObject json = gson.fromJson(data, JsonObject.class);
        JsonObject jdata = json.get("data").getAsJsonObject();
        JsonArray permissions = jdata.get("permissions").getAsJsonArray();
        return (permissions);
    }

}
