
package com.openexchange.guard.drive;

import java.io.InputStream;
import java.util.ArrayList;
import org.apache.http.entity.ContentType;
import com.google.gson.JsonObject;
import com.openexchange.guard.keys.dao.GuardKeys;
import com.openexchange.guard.keys.dao.RecipKey;
import com.openexchange.guard.ox.streaming.PGPStreamEncryptionHandler;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.streaming.JsonStreamHandler;
import com.openexchange.guard.oxapi.streaming.StreamHandler;

/**
 * Class for encrypting a drive file
 */
public class FileEncrypter {

    private final Api api;

    /**
     * Constructor
     * 
     * @param api HTTP API access to drive
     */
    public FileEncrypter(Api api) {
        this.api = api;
    }

    /**
     * Encrypts an input stream and uploads the encrypted content to the same drive file
     *
     * @param inputStream The stream to encrypt
     * @param fileName The name of the file to encrypt
     * @param fileId The file identifier of the file that is going to be updated
     * @param fileType The content type of the file to encrypt
     * @param jsonStream The stream to a JSON object which contains meta data which should be written to the new file
     * @param jsonHandler A handler which can change the JSON data during processing the jsonStream
     * @param recipients A list of recipients to encrypt the file for
     * @return The JSON response from the backend if the upload of the encrypted item was successful
     * @throws Exception On an error during encryption or down/uploading
     */
    public JsonObject encryptUpdate(InputStream inputStream, String fileName, String fileId, String fileType, InputStream jsonStream, JsonStreamHandler jsonHandler, ArrayList<RecipKey> recipients) throws Exception {
        return encrypt(inputStream, fileName, fileId, fileType, jsonStream, jsonHandler, recipients, null, /* no signing key: do not sign */
        null, /* no signing key password: do not sign */
        true);
    }

    /**
     * Encrypts an input stream and uploads the encrypted content to a new drive file
     *
     * @param inputStream The stream to encrypt
     * @param fileName The name of the file to encrypt
     * @param fileType The content type of the file to encrypt
     * @param jsonStream The stream to a JSON object which contains meta data which should be written to the new file
     * @param jsonHandler A handler which can change the JSON data during processing the jsonStream
     * @param recipients A list of recipients to encrypt the file for
     * @return The JSON response from the backend if the upload of the encrypted item was successful
     * @throws Exception On an error during encryption or down/uploading
     */
    public JsonObject encryptNew(InputStream inputStream, String fileName, String fileType, InputStream jsonStream, JsonStreamHandler jsonHandler, ArrayList<RecipKey> recipients) throws Exception {
        return encrypt(inputStream, fileName, null, fileType, jsonStream, jsonHandler, recipients, null, /* no signing key: do not sign */
        null, /* no signing key password: do not sign */
        false);
    }

    /**
     * Encrypts and signs an input stream and uploads the encrypted content to a new drive file
     *
     * @param inputStream The stream to the item to encrypt
     * @param fileName The name of the file to encrypt
     * @param fileType The content type of the file to encrypt
     * @param jsonStream The stream to a JSON object which contains meta data which should be written to the new file
     * @param jsonHandler A handler which can change the JSON data during processing the jsonStream
     * @param recipients A list of recipients to encrypt the file for
     * @param signingKey The key to sign the item with, or null
     *            if the item should not be signed
     * @param signingKeyPassword The password for using the signing key,
     *            or null if signingKey is null
     * @return The JSON response from the backend if the upload of the encrypted item was successful
     * @throws Exception On an error during encryption or down/uploading
     */
    private JsonObject encrypt(InputStream inputStream, String fileName, String fileId, String fileType, InputStream jsonStream, JsonStreamHandler jsonHandler, ArrayList<RecipKey> recipients, GuardKeys signingKey, String signingKeyPassword, boolean update) throws Exception {

        //Setup up the encryption handling using PGP as default encryption mode
        StreamHandler streamEncryptor = null;
        if (signingKey == null) {
            //encryption only
            streamEncryptor = new PGPStreamEncryptionHandler(recipients, false /* armored */);
            //guard encryption for testing purpose
            //You can enable this for testing purpose using the legacy guard encryption
            //streamEncryptor = new GuardStreamEcryptionHandler(recipients.get(0).userid,
            //                                                  recipients.get(0).cid);
        } else {
            //encryption and signing
            streamEncryptor = new PGPStreamEncryptionHandler(recipients, signingKey, signingKeyPassword, true /* armored */);
        }

        String newItemName = streamEncryptor.getNewName(fileName);
        ContentType newContentType = ContentType.create(streamEncryptor.getNewContentType(fileType, fileName));

        //Upload the stream while encrypting the data
        if (update) {
            return api.uploadDriveUpdatedStreamItem(inputStream, streamEncryptor, newItemName, fileId, newContentType, jsonStream, jsonHandler);
        } else {
            return api.uploadDriveNewStreamItem(inputStream, streamEncryptor, newItemName, newContentType, jsonStream, jsonHandler);
        }
    }
}
