
package com.openexchange.guard.drive;

import java.io.InputStream;
import org.apache.http.entity.ContentType;
import com.google.gson.JsonObject;
import com.openexchange.guard.ox.streaming.StreamHandlerFactory;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.streaming.JsonStreamHandler;
import com.openexchange.guard.oxapi.streaming.StreamHandler;

/**
 * Class for decrypting a drive file
 */
public class FileDecrypter {

    private final Api api;

    /**
     * Constructor
     *
     * @param api HTTP API access to drive
     */
    public FileDecrypter(Api api) {
        this.api = api;
    }

    /**
     * Decrypts a drive file
     *
     * @param fileId The id of the file to decrypt
     * @param fileSize The size of the file to decrypt
     * @param folderId The id of the file's foder
     * @param version The version to decrypt
     * @param fileName The name of the file to decrypt
     * @param userId The id of the user to decrypt the file for
     * @param cid
     * @param password The password of the user's private key
     * @param encryptedExtraPassword an encrypted, optional (used only with legacy Guard files),
     *            additional password which was set for the file
     * @param fileType The file type
     * @param jsonStream The stream to a JSON object which contains meta data which should be written to the new file
     * @param jsonHandler A handler which can change the JSON data during processing the jsonStream
     * @throws Exception on an error
     * @return The JSON response from the backend if the upload of the decrypted item was successful
     */
    public JsonObject decrypt(String fileId, long fileSize, String folderId, String version, String fileName, int userId, int cid, String password, String fileType, InputStream jsonStream, JsonStreamHandler jsonHandler) throws Exception {

        try (InputStream remoteInputStream = api.openDriveItem(fileId, fileSize, folderId, version)) {

            //Creating a stream handler which is doing the decryption while processing the stream
            StreamHandler streamDecryptor = StreamHandlerFactory.createStreamHandler(fileName, userId, cid, password, api.getCookie().getJSESSIONID());
            String newItemName = streamDecryptor.getNewName(fileName);
            ContentType newContentType = ContentType.create(streamDecryptor.getNewContentType(fileType, fileName));

            //Upload the stream while decrypting the data
            JsonObject json = api.uploadDriveNewStreamItem(remoteInputStream, streamDecryptor, newItemName, newContentType, jsonStream, jsonHandler);
            return json;
        }
    }
}
