/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.oxapi.streaming;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.Charset;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.content.AbstractContentBody;

/**
 * A content body which can be used in HTTP request to
 * read from an input stream
 *
 * @author Benjamin Gruedelbach
 */
public class StreamedContentBody extends AbstractContentBody {

    private final InputStream inputStream;
    private final StreamHandler handler;
    private final int inputBufferSize;
    private final String filename;
    private final ContentType contentType;

    /**
     * Constructor
     *
     * @param inputStream An input stream representing the remote file to process
     * @param contentType The content type
     * @param handler A handler defining the concrete data processing
     *            method to apply for the InputStream
     */
    public StreamedContentBody(InputStream inputStream, int inputBufferSize, ContentType contentType, StreamHandler handler, String filename) {
        super(contentType.getMimeType());

        if (inputStream == null) {
            throw new NullPointerException("File inputstream is null");
        }
        if (handler == null) {
            throw new NullPointerException("Encryption handleris null");
        }
        this.contentType = contentType;
        this.inputStream = inputStream;
        this.inputBufferSize = inputBufferSize;
        this.handler = handler;
        this.filename = filename;
    }

    @Override
    public String getFilename() {
        return filename;
    }

    @Override
    public void writeTo(OutputStream os) throws IOException {
        handler.processStream(inputStream, os, inputBufferSize);
    }

    @Override
    public long getContentLength() {
        //-1 because of chunked upload
        return -1;
    }

    @Override
    public String getTransferEncoding() {
        return "UTF-8";
    }

    @Override
    public String getCharset() {
        final Charset charset = this.contentType.getCharset();
        return charset != null ? charset.name() : null;
    }
}
