/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.management.osgi;

import javax.management.ObjectName;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.GuardMaintenanceService;
import com.openexchange.guard.management.maintenance.GuardMaintenanceMBean;
import com.openexchange.guard.management.maintenance.internal.GuardMaintenanceMBeanImpl;
import com.openexchange.guard.management.statistics.GuardServerStatisticsMBean;
import com.openexchange.guard.management.statistics.internal.GuardServerStatisticsMBeanImpl;
import com.openexchange.management.ManagementService;

/**
 * {@link MBeansServiceTracker}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class MBeansServiceTracker implements ServiceTrackerCustomizer<ManagementService, ManagementService> {

    private BundleContext context;

    private ObjectName statisticsObjectName;

    private ObjectName maintenanceObjectName;

    private GuardMaintenanceService guardMaintenanceService;

    /**
     * Initialises a new {@link MBeansServiceTracker}.
     */
    public MBeansServiceTracker(BundleContext context, GuardMaintenanceService guardMaintenanceService) {
        super();
        this.context = context;
        this.guardMaintenanceService = guardMaintenanceService;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.osgi.util.tracker.ServiceTrackerCustomizer#addingService(org.osgi.framework.ServiceReference)
     */
    @Override
    public ManagementService addingService(ServiceReference<ManagementService> reference) {
        Logger logger = LoggerFactory.getLogger(MBeansServiceTracker.class);
        ManagementService managementService = context.getService(reference);
        try {
            statisticsObjectName = new ObjectName(GuardServerStatisticsMBean.DOMAIN, GuardServerStatisticsMBean.KEY, GuardServerStatisticsMBean.VALUE);
            managementService.registerMBean(statisticsObjectName, new GuardServerStatisticsMBeanImpl());
            logger.info("GuardServerStatisticsMBean successfully registered");

            maintenanceObjectName = new ObjectName(GuardMaintenanceMBean.DOMAIN, GuardMaintenanceMBean.KEY, GuardMaintenanceMBean.VALUE);
            managementService.registerMBean(maintenanceObjectName, new GuardMaintenanceMBeanImpl(this.guardMaintenanceService));
            logger.info("GuardMaintenanceMBean successfully registered");

            return managementService;
        } catch (Exception e) {
            logger.error("Could not register GuardServerStatisticsMBean", e);
        }

        context.ungetService(reference);
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.osgi.util.tracker.ServiceTrackerCustomizer#modifiedService(org.osgi.framework.ServiceReference, java.lang.Object)
     */
    @Override
    public void modifiedService(ServiceReference<ManagementService> reference, ManagementService service) {
        // no-op

    }

    /*
     * (non-Javadoc)
     * 
     * @see org.osgi.util.tracker.ServiceTrackerCustomizer#removedService(org.osgi.framework.ServiceReference, java.lang.Object)
     */
    @Override
    public void removedService(ServiceReference<ManagementService> reference, ManagementService service) {
        Logger logger = LoggerFactory.getLogger(MBeansServiceTracker.class);
        if (service != null) {
            if (statisticsObjectName != null) {
                try {
                    service.unregisterMBean(statisticsObjectName);
                    logger.info("GuardServerStatisticsMBean successfully unregistered");
                } catch (OXException e) {
                    logger.warn("Could not unregister GuardServerStatisticsMBean", e);
                }
            }
            if (maintenanceObjectName != null) {
                try {
                    service.unregisterMBean(maintenanceObjectName);
                    logger.info("GuardMaintenanceMBean successfully unregistered");
                } catch (OXException e) {
                    logger.warn("Could not unregister GuardMaintenanceMBean", e);
                }
            }
        }
    }
}
