/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keys.util;

import java.util.Date;
import java.util.Iterator;
import org.bouncycastle.bcpg.HashAlgorithmTags;
import org.bouncycastle.bcpg.sig.KeyFlags;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureSubpacketVector;
import org.bouncycastle.openpgp.operator.PBESecretKeyDecryptor;
import org.bouncycastle.openpgp.operator.PBESecretKeyEncryptor;
import org.bouncycastle.openpgp.operator.PGPDigestCalculator;
import org.bouncycastle.openpgp.operator.bc.BcPBESecretKeyDecryptorBuilder;
import org.bouncycastle.openpgp.operator.bc.BcPBESecretKeyEncryptorBuilder;
import org.bouncycastle.openpgp.operator.bc.BcPGPDigestCalculatorProvider;
import org.bouncycastle.openpgp.operator.jcajce.JcePBESecretKeyDecryptorBuilder;
import com.openexchange.guard.common.util.CipherUtil;

/**
 * {@link PGPUtil}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public final class PGPUtil {

    /**
     * Convert byte array fingerprint into standard String format with spacing
     *
     * @param fingerprint
     * @return
     */
    public static String getFingerPrintString(byte[] fingerprint) {
        StringBuffer fpstring = new StringBuffer();
        for (int i = 0; i < fingerprint.length; ++i) {
            String hex = Integer.toHexString((char) fingerprint[i]);
            hex = hex.toUpperCase();
            while (hex.length() < 2) {
                hex = '0' + hex;
            }
            if (hex.length() > 2) {
                hex = hex.substring(hex.length() - 2);
            }
            fpstring.append(hex);
            if (i % 2 == 1) {
                fpstring.append(" ");
            }
        }
        return fpstring.toString();
    }

    public static int getHashAlgorithmTags() {
        return HashAlgorithmTags.SHA256;
    }

    /**
     * Decodes the specified private key with he specified password and salt
     *
     * @param secretKey The secret key to decode
     * @param password The password to decode it with
     * @param salt the salt
     * @return The decoded private key
     * @throws PGPException
     */
    public static PGPPrivateKey decodePrivate(PGPSecretKey secretKey, String password, String salt) throws PGPException {
        return secretKey.extractPrivateKey(new BcPBESecretKeyDecryptorBuilder(new BcPGPDigestCalculatorProvider()).build(CipherUtil.getSHA(password, salt).toCharArray()));
    }

    /**
     * Check if a public key is expired
     *
     * @param key
     * @return
     */
    public static boolean isExpired(PGPPublicKey key) {
        if (key == null) {
            return false;
        }
        if (key.getValidSeconds() == 0) {
            return false;
        }
        Date now = new Date();
        return key.getCreationTime().getTime() + (key.getValidSeconds() * 1000) - now.getTime() < 0;
    }

    /**
     * Check if all keys are expired
     */
    public static boolean checkAllExpired(PGPPublicKeyRing ring) {
        boolean allExpired = true;
        Iterator<PGPPublicKey> it = ring.getPublicKeys();
        while (it.hasNext()) {
            PGPPublicKey key = it.next();
            allExpired = allExpired && isExpired(key);
        }
        return (allExpired);
    }

    /**
     * Duplicates the specified secret key ring. This method can also be used to create a new secret key ring with a different password
     *
     * @param secretKeyRing The secret key ring to duplicate
     * @param decryptorPasswordHash The hashed password for the decryptor
     * @param encryptorPasswordHash The hashed password for the encryptor
     * @param symmetricKeyAlgorithmTag The symmetric key algorithm tag (see PGPEncryptedData.AES_256 and PGPEncryptedData.AES_128)
     * @return The duplicated {@link PGPSecretKeyRing}
     * @throws PGPException
     */
    public static PGPSecretKeyRing duplicateSecretKeyRing(PGPSecretKeyRing secretKeyRing, String decryptorPasswordHash, String encryptorPasswordHash, int symmetricKeyAlgorithmTag) throws PGPException {
        PGPDigestCalculator sha256Calc = new BcPGPDigestCalculatorProvider().get(getHashAlgorithmTags());
        PBESecretKeyDecryptor oldEncryptor = new JcePBESecretKeyDecryptorBuilder().setProvider("BC").build(decryptorPasswordHash.toCharArray());
        PBESecretKeyEncryptor newEncryptor = new BcPBESecretKeyEncryptorBuilder(symmetricKeyAlgorithmTag, sha256Calc, 0x60).build(encryptorPasswordHash.toCharArray());

        return PGPSecretKeyRing.copyWithNewPassword(secretKeyRing, oldEncryptor, newEncryptor);
    }

    /**
     * Returns the signature type of the specified signature identifier
     *
     * @param signature The signature identifier
     * @return The signature type as string
     */
    public static String getSignatureType(int signature) {
        switch (signature) {
            case -1:
                return ("Fail");
            case 0:
                return ("Missing Public key");
            case PGPSignature.POSITIVE_CERTIFICATION:
                return ("Positive");
            case PGPSignature.CASUAL_CERTIFICATION:
                return ("Casual");
            case PGPSignature.CERTIFICATION_REVOCATION:
                return ("Revoke Cert");
            case PGPSignature.DEFAULT_CERTIFICATION:
                return ("Default Cert");
            case PGPSignature.DIRECT_KEY:
                return ("Direct");
            case PGPSignature.KEY_REVOCATION:
                return ("Revoke");
            case PGPSignature.SUBKEY_REVOCATION:
                return ("Sub revoke");
        }
        return ("");
    }

    /**
     * Gets the flags of a given key
     *
     * @param key the key to get the flags for
     * @return the flags for the given key
     */
    public static int getKeyFlags(PGPPublicKey key) {
        Iterator<PGPSignature> signatures = key.getSignatures();
        while (signatures.hasNext()) {
            PGPSignature signature = signatures.next();
            PGPSignatureSubpacketVector packet = signature.getHashedSubPackets();
            if(packet != null) {
                return packet.getKeyFlags();
            }
        }
        return 0;
    }

    /**
     * Checks whether a key has the given flags or not
     *
     * @param key the key
     * @param flag the flag to check
     * @see https://tools.ietf.org/html/rfc4880#section-5.2.3.21
     * @return True if the key has the given flag
     */
    public static boolean keyHasFlag(PGPPublicKey key, int flag) {
        int existingFlags = getKeyFlags(key);
        return (existingFlags & flag) > 0;
    }

    /**
     * Checks whether a key is meant to be an encryption key.
     *
     * Note: This method checks if the preferred usage of the key is encrypting.
     * Use {@link org.bouncycastle.openpgp.PGPPublicKey#isEncryptionKey} for checking if a key is technical able to encrypt
     *
     * @param key the key to check
     * @return true, if the key is meant to be an encryption key, false otherwise
     */
    public static boolean isEncryptionKey(PGPPublicKey key) {
        //Check if the key has flags
        if (getKeyFlags(key) > 0) {
            //Check for encryption Flags
            return keyHasFlag(key, KeyFlags.ENCRYPT_COMMS | KeyFlags.ENCRYPT_STORAGE);
        } else {
            //Fallback if flags do not exist (for older keys or if the key creation software did not create them)
            return key.isEncryptionKey();
        }
    }

    /**
     * Returns the ring's singing key
     * @param keyRing The key ring
     * @return the ring's signing sub key or the master key, if no sub key is marked for singing
     */
    public static PGPSecretKey getSigningKey(PGPSecretKeyRing keyRing){
        PGPSecretKey ret = getSigningSubKey(keyRing);
        if(ret == null){
            //If no signing subkey was found we are using the master key
            ret = keyRing.getSecretKey();
        }
        return ret;
    }

    /**
     * Returns a ring's subkey which is meant to be used as signing key
     * @param keyRing The ring to get the signing sub key for
     * @return the signing sub key, or null if no signing sub key was found
     */
    public static PGPSecretKey getSigningSubKey(PGPSecretKeyRing keyRing){
        Iterator<PGPSecretKey> iter = keyRing.getSecretKeys();
        while(iter.hasNext()){
            PGPSecretKey secretKey = iter.next();
            if(!secretKey.isMasterKey() /*only check sub keys*/){
                PGPPublicKey publicKey = secretKey.getPublicKey();
                boolean isSigningKey = keyHasFlag(publicKey,KeyFlags.SIGN_DATA);
                if(isSigningKey && !publicKey.hasRevocation()){
                    return secretKey;
                }
            }
        }
        return null;
    }
}
