/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keys.storage.remoteKeyCache;

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.ListUtil;
import com.openexchange.guard.database.GuardDatabaseService;
import com.openexchange.guard.database.utils.DBUtils;
import com.openexchange.guard.keys.PGPKeyService;
import com.openexchange.guard.keys.PGPPublicKeyRingFactory;
import com.openexchange.guard.keys.exceptions.KeysExceptionCodes;
import com.openexchange.guard.keys.osgi.Services;
import com.openexchange.guard.keys.storage.RemoteKeyCacheStorage;
import com.openexchange.guard.ratifier.GuardRatifierService;

public class RemoteKeyCacheStorageImpl implements RemoteKeyCacheStorage {

    private final static String VALUES_LIST = "{@VALUES}";

    @Override
    public int deleteOld(int days) throws OXException {
        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);
        Connection connection = guardDatabaseService.getWritableForGuard();

        PreparedStatement stmt = null;
        ResultSet executeQuery = null;
        try {
            stmt = connection.prepareStatement(RemoteKeyCacheSql.SELECT_OLD_STMT);
            stmt.setInt(1, days);
            executeQuery = stmt.executeQuery();

            DBUtils.closeSQLStuff(stmt);

            ArrayList<Long> deletingIds = new ArrayList<Long>();
            while (executeQuery.next()) {
                deletingIds.add(executeQuery.getLong("id"));
            }

            if (deletingIds.size() > 0) {
                String sql = RemoteKeyCacheSql.DELETE_REMOTE_KEYS_STMT;
                stmt = connection.prepareStatement(sql.replace(VALUES_LIST, ListUtil.listToDbValuesList(deletingIds.size())));
                Integer[] idsArr = new Integer[deletingIds.size()];
                for (int i = 0; i < idsArr.length; i++) {
                    stmt.setInt(i, idsArr[i]);
                }
                stmt.executeUpdate();
                DBUtils.closeSQLStuff(stmt);

                sql = RemoteKeyCacheSql.DELETE_REMOTE_KEY_CONTENT_STMT;
                stmt = connection.prepareStatement(sql.replace(VALUES_LIST, ListUtil.listToDbValuesList(deletingIds.size())));
                for (int i = 0; i < idsArr.length; i++) {
                    stmt.setInt(i, idsArr[i]);
                }
                stmt.executeUpdate();
                DBUtils.closeSQLStuff(stmt);
            }
            return deletingIds.size();
        } catch (SQLException e) {
            throw KeysExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(stmt, executeQuery);
            guardDatabaseService.backWritableForGuard(connection);
        }
    }

    @Override
    public void insert(PGPPublicKeyRing keyRing) throws OXException {
        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);
        Connection connection = guardDatabaseService.getWritableForGuard();

        PreparedStatement stmt = null;
        try {
            connection.setAutoCommit(false);
            stmt = connection.prepareStatement(RemoteKeyCacheSql.INSERT_RING_DATA_STMT);

            PGPKeyService pgpKeyService = Services.getService(PGPKeyService.class);
            String keyRingAsc = pgpKeyService.export(keyRing);
            stmt.setLong(1, keyRing.getPublicKey().getKeyID());
            stmt.setString(2, keyRingAsc);
            stmt.setString(3, keyRingAsc);
            stmt.executeUpdate();

            //Storing each identity related to the key ring
            Iterator<String> ids = keyRing.getPublicKey().getUserIDs();
            Long refid = keyRing.getPublicKey().getKeyID();
            while (ids.hasNext()) {
                String email = ids.next().trim();
                if (email.contains("<")) {
                    email = email.substring(email.indexOf("<") + 1);
                    if (email.endsWith(">")) {
                        email = email.substring(0, email.length() - 1);
                    }
                }
                GuardRatifierService validatorService = Services.getService(GuardRatifierService.class);
                validatorService.validate(email);

                Iterator<PGPPublicKey> keys = keyRing.getPublicKeys();
                while (keys.hasNext()) {
                    PGPPublicKey key = keys.next();
                    if (!key.hasRevocation()) {

                        PreparedStatement stmt2 = null;
                        try {
                            stmt2 = connection.prepareStatement(RemoteKeyCacheSql.INSERT_RING_ID_STMT);

                            stmt2.setLong(1, key.getKeyID());
                            stmt2.setString(2, email);
                            stmt2.setLong(3, refid);
                            stmt2.setLong(4, refid);
                            stmt2.executeUpdate();
                        } finally {
                            DBUtils.closeSQLStuff(stmt2);
                        }
                    }
                }
            }
            connection.commit();
        } catch (SQLException e) {
            throw KeysExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } finally {
            DBUtils.autocommit(connection);
            DBUtils.closeSQLStuff(stmt);
            guardDatabaseService.backWritableForGuard(connection);

        }
    }

    @Override
    public List<PGPPublicKeyRing> getByEmail(String email) throws OXException {
        GuardDatabaseService guardDatabaseService = Services.getService(GuardDatabaseService.class);
        Connection connection = guardDatabaseService.getReadOnlyForGuard();

        PreparedStatement stmt = null;
        ResultSet resultSet = null;
        try {
            stmt = connection.prepareStatement(RemoteKeyCacheSql.SELECT_BY_EMAIL_STMT);
            stmt.setString(1, email);

            resultSet = stmt.executeQuery();
            ArrayList<PGPPublicKeyRing> ret = new ArrayList<PGPPublicKeyRing>();
            while (resultSet.next()) {
                ret.add(PGPPublicKeyRingFactory.create(resultSet.getString("ring")));
            }
            return ret;
        } catch (SQLException e) {
            throw KeysExceptionCodes.SQL_ERROR.create(e, e.getMessage());
        } catch (IOException e) {
            throw KeysExceptionCodes.IO_ERROR.create(e, e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(resultSet, stmt);
            guardDatabaseService.backReadOnlyForGuard(connection);
        }
    }
}
