/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keys.internal.master.internal;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.security.PrivateKey;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.keys.GuardKeyService;
import com.openexchange.guard.keys.GuardMasterKeyService;
import com.openexchange.guard.keys.dao.GuardKeys;
import com.openexchange.guard.keys.dao.GuardMasterKey;
import com.openexchange.guard.keys.internal.master.exceptions.GuardMasterKeyExceptionCodes;
import com.openexchange.guard.keys.osgi.Services;

/**
 * {@link GuardMasterKeyServiceImpl}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class GuardMasterKeyServiceImpl implements GuardMasterKeyService {

    private static final Logger LOG = LoggerFactory.getLogger(GuardMasterKeyServiceImpl.class);

    private GuardMasterKey masterKey;

    /**
     * Initialises a new {@link GuardMasterKeyServiceImpl}.
     * 
     * @throws OXException
     */
    public GuardMasterKeyServiceImpl() throws OXException {
        super();
        initialise();
    }

    /**
     * Initialises the master key
     * 
     * @throws OXException
     */
    private void initialise() throws OXException {
        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
        File f = configService.getPropertyFile("oxguardpass");
        if (f == null || !f.exists()) {
            throw GuardMasterKeyExceptionCodes.MASTER_KEY_FILE_NOT_EXIST.create();
        } else {
            Map<String, String> salts = readMasterKey(f);
            final String mc = salts.get("MC");
            final String rc = salts.get("RC");

            GuardKeyService keyService = Services.getService(GuardKeyService.class);
            GuardKeys passwordKeys = keyService.getKeys(-2, 0);
            if (passwordKeys == null) {
                generateMasterKey(mc, rc);
            }
            readMasterKey(mc, rc);
        }
    }

    /**
     * Reads the master password file 'oxguardpass'
     * 
     * @return
     * @throws OXException
     */
    private Map<String, String> readMasterKey(File file) throws OXException {
        // Read lines from the 'oxguardpass'
        List<String> lines = readMasterKeyFile(file);

        // Get lines
        Map<String, String> tmp = new HashMap<String, String>(2);
        for (String line : lines) {
            String k = line.substring(0, 2);
            String v = line.substring(2);
            tmp.put(k, v);
        }

        if (tmp.isEmpty()) {
            throw GuardMasterKeyExceptionCodes.MALFORMED_MASTER_KEY_FILE.create(file.getAbsolutePath());
        }

        return tmp;
    }

    /**
     * Reads the master key from the database
     * 
     * @param mc The MC salt
     * @param rc The RC salt
     * @throws OXException
     */
    private void readMasterKey(String mc, String rc) throws OXException {
        // Read private and public key from the db
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        PrivateKey privateKey = null;
        try {
            GuardKeys passwordKeys = keyService.getKeys(-2, 0);
            if (passwordKeys != null) {
                privateKey = keyService.decryptPrivateKey(passwordKeys.getEncodedPrivate(), rc, passwordKeys.getSalt(), true);
            }
        } catch (Exception ex) {
            LOG.error("Problem loading password RSA key", ex);
        }

        GuardKeys passwordKeys;
        String publicKey = null;
        try {
            passwordKeys = keyService.getKeys(-2, 0);
            if (passwordKeys != null) {
                publicKey = passwordKeys.getEncodedPublic();
            }
        } catch (Exception e) {
            LOG.error("Error getting public RSA key ", e);
        }

        // Reference the master key
        masterKey = new GuardMasterKey(mc, rc, publicKey, privateKey);
    }

    /**
     * Reads the master key file from the specified file and returns the contents in a List where every entry represents a line
     * 
     * @param file The file to read the master key from
     * @return A list with the contents of the file, where every entry represents a line
     * @throws OXException
     */
    private List<String> readMasterKeyFile(File file) throws OXException {
        List<String> lines;
        try {
            lines = Files.readAllLines(Paths.get(file.getAbsolutePath()), StandardCharsets.UTF_8);
            if (lines.size() != 2) {
                throw GuardMasterKeyExceptionCodes.MALFORMED_MASTER_KEY_FILE.create(file.getAbsolutePath());
            }
        } catch (IOException e) {
            throw GuardMasterKeyExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
        return lines;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.configuration.GuardMasterKeyService#getMasterKey()
     */
    @Override
    public GuardMasterKey getMasterKey() {
        return masterKey;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.configuration.GuardMasterKeyService#masterKeyExists()
     */
    @Override
    public boolean masterKeyExists() throws OXException {
        boolean exists = false;
        final String masterPasswordFile = null;
        File file = new File(masterPasswordFile);
        if (file.exists()) {
            readMasterKeyFile(file);
            exists = true;
        }
        return exists;
    }

    /**
     * Generates the master keys in database with the specified mc and rc salts
     * 
     * @param mcSalt The MC salt
     * @param rcSalt The RC salt
     * @throws OXException
     */
    private void generateMasterKey(String mcSalt, String rcSalt) throws OXException {
        // Generate public and private keys
        PrivateKey privateKey = null;
        String publicKey = null;
        try {
            GuardKeys keys = createMasterKeysNotCached(-1, 0, "dont@remove.com", "master", mcSalt, "", false, true);// Create master key for password recovery
            if (keys == null) {
                LOG.error("Unable to create master keys");
            } else {
                GuardKeyService keyService = Services.getService(GuardKeyService.class);
                privateKey = keyService.decryptPrivateKey(keys.getEncodedPrivate(), mcSalt, keys.getSalt(), true);
            }
            GuardKeys keys2 = createMasterKeysNotCached(-2, 0, "dont@remove.com", "master", rcSalt, "", false, true);// Create master key for client encryption
            if (keys2 == null) {
                LOG.error("Unable to create master keys for client encryption");
            } else {
                publicKey = keys2.getEncodedPublic();
            }
        } catch (Exception ex) {
            LOG.error("Unable to create master keys", ex);
            throw ex;
        }

        // Update reference
        masterKey = new GuardMasterKey(mcSalt, rcSalt, publicKey, privateKey);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.GuardMasterKeyService#createMasterKeysNC(int, int, java.lang.String, java.lang.String, java.lang.String, java.lang.String, boolean, boolean)
     */
    @Override
    public GuardKeys createMasterKeysNotCached(int id, int cid, String email, String name, String password, String language, boolean createRecovery, boolean markAsCurrent) throws OXException {
        return createMasterKeysNotCached(id, cid, email, name, password, language, false, createRecovery, markAsCurrent);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.GuardMasterKeyService#createMasterKeys(int, int, java.lang.String, java.lang.String, java.lang.String, java.lang.String, boolean, boolean, boolean)
     */
    @Override
    public GuardKeys createMasterKeys(int id, int cid, String email, String name, String password, String language, boolean userCreated, boolean createRecovery, boolean markAsCurrent) throws OXException {
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        try {
            GuardKeys keys = keyService.generateKeyPair(name, email, password);
            if (keys != null) {
                keys.setEmail(email);
                keys.setUserid(id);
                keys.setContextid(cid);
                keys.setLanguage(language);
                keys.setRecovery(createRecovery ? keyService.createRecovery(keys, password) : "");
                keys.setCurrent(markAsCurrent);
                keys = keyService.storeKeys(keys, userCreated);
                if ((keys == null) || (keys.getUserid() == 0)) {
                    // Failed to create keys
                    // TODO: fail with exception instead of null?
                    return null;
                }
            } else {
                // Try fail over to generate it in real-time
                return createMasterKeysNotCached(id, cid, email, name, password, language, userCreated, createRecovery, markAsCurrent);
            }
            return keys;
        } catch (Exception e) {
            LOG.error("Error create keys", e);
            if (e.getMessage() != null) {
                if (e.getMessage().contains("Duplicate")) {
                    LOG.error("Corrupt key lookup?  Check table oxguard.og_email");
                }
            }
            return null;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.keys.GuardMasterKeyService#createMasterKeysNC(int, int, java.lang.String, java.lang.String, java.lang.String, java.lang.String, boolean, boolean, boolean)
     */
    @Override
    public GuardKeys createMasterKeysNotCached(int id, int cid, String email, String name, String password, String language, boolean userCreated, boolean createRecovery, boolean markAsCurrent) throws OXException {
        LOG.debug("Not using cache, Creating master keys for " + email);
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        try {
            GuardKeys keys = keyService.generateKeys(name, email, password);
            if (keys != null) {
                //keys.userid = acc.storeKeysORIG(id, cid, email, keys, createRecovery ? createRecovery(keys, password) : "", language, userCreated);
                keys.setEmail(email);
                keys.setUserid(id);
                keys.setContextid(cid);
                keys.setLanguage(language);
                keys.setRecovery(createRecovery ? keyService.createRecovery(keys, password) : "");
                keys.setCurrent(markAsCurrent);
                keys = keyService.storeKeys(keys, userCreated);
                if (keys.getUserid() == 0) {
                    return null; //Failed to create keys
                }
            }
            return (keys);
        } catch (Exception e) {
            if (e.getMessage().contains("Duplicate entry")) {
                try {
                    LOG.debug("Key already exists, using existing");
                    return (keyService.getKeys(id, cid));
                } catch (Exception e1) {
                    LOG.error("Unable to get existing", e1);
                    return null;
                }
            }
            LOG.error("Error creating master keys for " + email, e);
            return null;
        }
    }
}
