
package com.openexchange.guard.keys.internal.hkpclient;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xbill.DNS.SRVRecord;
import com.openexchange.exception.OXException;
import com.openexchange.guard.dns.DNSService;
import com.openexchange.guard.keys.HKPClientService;
import com.openexchange.guard.keys.exceptions.PublicKeyExceptionCodes;
import com.openexchange.guard.keys.osgi.Services;

/**
 *
 * {@link SRVHKPClientService} Service for querying DNS SRV configured HKP Servers
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class SRVHKPClientService implements HKPClientService {

    private static final Logger logger = LoggerFactory.getLogger(RemoteHKPClientService.class);
    private final HKPClientService delegate;

    /**
    *
    * Initializes a new {@link SRVHKPClientService}.
    */
    public SRVHKPClientService() { this(null);}

    /**
     *
     * Initializes a new {@link SRVHKPClientService}.
     * @param delegate The service to delegate search queries to in a public key ring was not found
     */
    public SRVHKPClientService(HKPClientService delegate) {
        this.delegate = delegate;
    }

    /**
     * Internal method to query HKP servers which were configured in the email-domain's DNS SRV record
     *
     * @param clientToken an identification token put into the X-UI-INTERNAL-ACCOUNT-ID header, or null not not set the header
     * @param email the email to query
     * @return a list of found PGP key rings for the given email, an empty collection if no keys were found or the email's domain has no HKP SRV set
     * @throws Exception
     */
    private Collection<PGPPublicKeyRing> findForSRVRecords(String clientToken, String email) throws Exception {
        DNSService dnsService = Services.getService(DNSService.class);
        List<SRVRecord> records = dnsService.getSrvs(email);

        Collection<PGPPublicKeyRing> rings = new ArrayList<PGPPublicKeyRing>();
        int current = 0;
        SRVRecord nextcheck;
        while ((nextcheck = dnsService.findNext(current, records)) != null) {
            String target = nextcheck.getTarget().toString();
            if (target.endsWith(".")) {
                target = target.substring(0, target.length()-1);
            }
            int port = nextcheck.getPort();
            rings = new HKPClient(clientToken).findKeys(target, port, email);
            if (rings != null) {
                return rings;
            }
            current = nextcheck.getPriority();
        }
        return rings;
    }

    /**
     * Internal method to query SRV records
     *
     * @param clientToken an identification token put into the X-UI-INTERNAL-ACCOUNT-ID header, or null not not set the header
     * @param email The email
     * @return A list of public keys for the email
     * @throws Exception
     */
    private Collection<PGPPublicKeyRing> findInternal(String clientToken, String email) throws Exception {

        //Searching DNS SVR records
        Collection<PGPPublicKeyRing> ret = null;
        try {
            ret = findForSRVRecords(clientToken, email);
            if (ret != null && ret.size() > 0) {
                logger.debug("Remote keys found through SRV");
                return ret;
            }
        }
        catch(Exception e) {
            logger.error("Error querying remote HKP server form SRV record", e);
        }

        if(delegate != null){
            ret = delegate.find(clientToken, email);
        }
        return ret;
    }

    @Override
    public Collection<PGPPublicKeyRing> find(String clientToken, String email) throws OXException {
        try {
            return findInternal(clientToken, email);
        } catch (Exception e) {
            throw PublicKeyExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    @Override
    public PGPPublicKeyRing findFirst(String clientToken, String email) throws OXException {
        Collection<PGPPublicKeyRing> results = find(clientToken, email);
        return results != null && results.size() > 0 ? results.iterator().next() : null;
    }

}
