
package com.openexchange.guard.keys.internal.hkpclient;

import java.util.ArrayList;
import java.util.Collection;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.keys.HKPClientService;
import com.openexchange.guard.keys.exceptions.PublicKeyExceptionCodes;
import com.openexchange.guard.keys.osgi.Services;

/**
 * {@link RemoteHKPClientService} - default implementation for querying public HKP servers
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class RemoteHKPClientService implements HKPClientService {

    private static final Logger logger = LoggerFactory.getLogger(RemoteHKPClientService.class);
    private final Collection<String> configuredRemoteHKPServers;
    private final HKPClientService delegate;

    public RemoteHKPClientService() throws OXException {
        this(null);
    }

    public RemoteHKPClientService(HKPClientService delegate) throws OXException {
        this.delegate = delegate;
        this.configuredRemoteHKPServers = getRemoteHKPServers();
    }

    /**
     * Internal method to read configured HKP servers from configuration
     * 
     * @return A list of configured HKP servers
     * @throws OXException
     */
    private Collection<String> getRemoteHKPServers() throws OXException {
        Collection<String> hkpServers = new ArrayList<String>();

        GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
        String urlStrings = guardConfigService.getProperty(GuardProperty.publicPGPDirectory);
        String[] urls = urlStrings.split(",");
        for (String url : urls) {
            url = url.trim();
            if (url.startsWith("htt") || url.startsWith("hkp")) {
                if (!url.endsWith("?")) {
                    if (!url.endsWith("/")) {
                        url += "/";
                    }
                }
                hkpServers.add(url);
            }
        }
        return hkpServers;
    }

    /**
     * Internal method to query a list HKP Servers
     *
     * @param clientToken an identification token put into the X-UI-INTERNAL-ACCOUNT-ID header, or null not not set the header
     * @param hkpServers a list of HKP servers to query
     * @return A list of found keys
     * @throws Exception
     */
    private ArrayList<PGPPublicKeyRing> findForHKPServer(String clientToken, Collection<String> hkpServers, String email) throws Exception {
        ArrayList<PGPPublicKeyRing> ret = new ArrayList<PGPPublicKeyRing>();
        for (String hkpServer : hkpServers) {
            Collection<PGPPublicKeyRing> result = new HKPClient(clientToken).findKeys(hkpServer, email);
            if (!result.isEmpty()) {
                ret.addAll(ret.size(), result);
            }
        }
        return ret;
    }

    /**
     * Lookup public keys for email address. Checks SRV records first, then configured public servers
     *
     * @param clientToken an identification token put into the X-UI-INTERNAL-ACCOUNT-ID header, or null not not set the header
     * @param email
     * @return
     * @throws Exception
     */
    private Collection<PGPPublicKeyRing> findInternal(String clientToken, String email) throws Exception {

        //Searching configured HKP servers
        Collection<PGPPublicKeyRing> ret = null;
        try {
            ret = findForHKPServer(clientToken, configuredRemoteHKPServers, email);
            if (ret != null && ret.size() > 0) {
                logger.debug("Remote keys found in public server");
                return ret;
            }
        } 
        catch (Exception e) {
            logger.error("Error querying remote HKP server", e);
        }

        if (delegate != null) {
            ret = delegate.find(clientToken, email);
        }
        return ret;
    }

    @Override
    public Collection<PGPPublicKeyRing> find(String clientToken, String email) throws OXException {
        try {
            return findInternal(clientToken, email);
        } catch (Exception e) {
            throw PublicKeyExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    @Override
    public PGPPublicKeyRing findFirst(String clientToken, String email) throws OXException {
        Collection<PGPPublicKeyRing> results = find(clientToken, email);
        return results != null && results.size() > 0 ? results.iterator().next() : null;
    }
}
