package com.openexchange.guard.keys.internal.hkpclient;

import java.util.Collection;
import java.util.List;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.keys.HKPClientService;
import com.openexchange.guard.keys.exceptions.PublicKeyExceptionCodes;
import com.openexchange.guard.keys.osgi.Services;
import com.openexchange.guard.keys.storage.RemoteKeyCacheStorage;


/**
 *
 * {@link CachingHKPClientService} Service for querying guard's remote key cache
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class CachingHKPClientService implements HKPClientService {

    private static final Logger logger = LoggerFactory.getLogger(CachingHKPClientService.class);
    private final HKPClientService delegate;

    /**
     * Initializes a new {@link CachingHKPClientService}.
     */
    public CachingHKPClientService() { this(null); }

    /**
     * Initializes a new {@link CachingHKPClientService}.
     * @param delegate The service to delegate search queries to in a public key ring was not found
     */
    public CachingHKPClientService(HKPClientService delegate) {
        this.delegate = delegate;
    }

    /**
     * Internal method to query the OX Guard remote key cache for the given email
     * @param email The email to query
     * @return The key for the given email from the cache, or null if no such key was found in the cache
     * @throws Exception
     */
    private Collection<PGPPublicKeyRing> findInCache(String email) throws Exception  {
        RemoteKeyCacheStorage remoteKeyCacheStorage = Services.getService(RemoteKeyCacheStorage.class);
        List<PGPPublicKeyRing> rings = remoteKeyCacheStorage.getByEmail(email);
        return rings;
    }

    /**
     * Internal method to store a found key ring in the cache for later re-user
     * @param rings the key ring to save to the cache
     * @throws Exception
     */
    private void writeToCache(Collection<PGPPublicKeyRing> rings) throws Exception {
        RemoteKeyCacheStorage remoteKeyCacheStorage = Services.getService(RemoteKeyCacheStorage.class);
        for (PGPPublicKeyRing ring : rings) {
            remoteKeyCacheStorage.insert(ring);
        }
    }

    /**
     * Internal method for searching for public keys within the cache or delegate if not found
     *
     * @param clientToken an identification token put into the X-UI-INTERNAL-ACCOUNT-ID header, or null not not set the header
     * @param email The email to get the public key rings for
     * @return A list for key rings found for the given email
     * @throws Exception
     */
    private Collection<PGPPublicKeyRing> findInternal(String clientToken, String email) throws Exception {

        //Searching keys in the cache
        Collection<PGPPublicKeyRing> ret = null;
        try {
            ret = findInCache(email);
            if (ret != null && ret.size() > 0) {
                logger.debug("Remote keys found in cache");
                return ret;
            }
        }
        catch(Exception e) {
            logger.error("Error querying HKP key cache", e);
        }

        //Search delegate
        if(delegate != null){
            ret = delegate.find(clientToken, email);
        }

        if (ret != null && ret.size() > 0) {
            logger.debug("Writing found keys from HKP server into cache");
            writeToCache(ret);
        }
        return ret;
    }


    @Override
    public Collection<PGPPublicKeyRing> find(String clientToken, String email) throws OXException {
        try {
            return findInternal(clientToken, email);
        } catch (Exception e) {
            throw PublicKeyExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }

    @Override
    public PGPPublicKeyRing findFirst(String clientToken, String email) throws OXException {
        Collection<PGPPublicKeyRing> results = find(clientToken, email);
        return results != null && results.size() > 0 ? results.iterator().next() : null;
    }

}
