/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.hkp.servlets;

import java.io.IOException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;

/**
 *
 * Provides basic HKP (HTTP Keyserver Protocol) functionality letting OX Guard acting as a basic PGP Key Server
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class HkpServlet extends HttpServlet {

    private static final long serialVersionUID = -7405630378381362021L;
    private static final Logger logger = LoggerFactory.getLogger(HkpServlet.class);

    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response) {

        logger.debug("HKP request " + request.getQueryString());
        try {
            String operation = ServletUtils.getStringParameter(request, "op", true);
            String search = ServletUtils.getStringParameter(request, "search", true);

            if (search.contains("*")) { // Wildcard isn't being supported
                logger.debug("Attempt to use wildcard in remote lookup.  Rejected");
                response.setStatus(HttpServletResponse.SC_NOT_IMPLEMENTED);
                return;
            }

            switch (operation) {
                case "index":
                    new HKPIndexAction(request, response, search).doAction();
                    break;
                case "get":
                    new HKPGetAction(response, search).doAction();
                    return;
                case "vindex":
                    //The "vindex" operation (verbose index) shows (in addition to the index operation) signatures of keys,
                    //but this is currently not supported by OX Guard
                    response.setStatus(HttpServletResponse.SC_NOT_IMPLEMENTED);
                    return;
                default:
                    response.setStatus(HttpServletResponse.SC_BAD_REQUEST);
            }
        } catch (OXException ex) {
            if (ex.similarTo(GuardCoreExceptionCodes.PARAMETER_MISSING)) {
                logger.info("HKP request failed due to missing parameter");
                ServletUtils.sendNotAcceptable(response, "Missing Parameter " + ex);
            }
        } catch (Exception e) {
            logger.error("Unexpected error while handling HKP request", e);
            ServletUtils.sendError(response, e.getLocalizedMessage());
        }
    }

    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response) {
        try {
            //OX Guard does not support/allow key uploads via HKP
            //To be RFC compliant we send a 501-not implemented
            logger.debug("HKP post not implemented");
            ServletUtils.sendAnswer(response, HttpServletResponse.SC_NOT_IMPLEMENTED, "HKP post not supported");
        } catch (IOException e) {
            logger.error("Unexpected error while handling HKP request", e);
            ServletUtils.sendError(response, e.getLocalizedMessage());
        }
    }
}
