/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.email.storage.ogEmail;

import java.sql.Connection;
import java.util.List;
import com.openexchange.exception.OXException;

/**
 * {@link EmailStorage} provides access to stored ogEmail mappings
 */
public interface EmailStorage {

    /**
     * Creates a new Email item or updates an item, if there is already an item for the given email
     *
     * @param email the email to create or update the item for
     * @param contextId the context id
     * @param userId the user id
     * @param dbShard the db shard id pointing to the database containing the key information related to this email
     * @throws due an error
     */
    void insertOrUpdate(String email, int contextId, int userId, int dbShard) throws OXException;

    /**
     * Creates a new Email item or updates an item, if there is already an item for the given email
     *
     * @param connection the connection to use
     * @param email the email to create or update the item for
     * @param contextId the context id
     * @param userId the user id
     * @param dbShard the db shard id pointing to the database containing the key information related to this email
     * @throws due an error
     */
    void insertOrUpdate(Connection connection, String email, int contextId, int userId, int dbShard) throws OXException;

    /**
     * Deletes all ogEmail mappings for a given user
     *
     * @param contextId the context id
     * @param userId the user id to delete the mappings for
     * @throws due an error
     */
    void deleteAllForUser(int contextId, int userId) throws OXException;

    /**
     * Deletes all ogEmail mappings for a given context
     *
     * @param contextId the context id to delete the mappings for
     * @throws OXException due an error
     */
    void deleteAllForContext(int contextId) throws OXException;

    /**
     * Gets a mapping for a user's email
     *
     * @param email the email to get the mapping for
     * @return the mapping for the given email, or null if no mapping was found
     */
    Email getByEmail(String email) throws OXException;

    /**
     * Retrieves the e-mail address for the specified user
     *
     * @param userId The user identifier
     * @param cid The context identifier
     * @return The e-mail address
     * @throws OXException
     */
    String getEmail(int userId, int cid) throws OXException;

    /**
     * Gets a list of Email items for a given user
     *
     * @param contextId the context id of the user
     * @param userId the user's id
     * @return a list of Email items for the given user
     * @throws due an error
     */
    List<Email> getById(int contextId, int userId) throws OXException;

    /**
     * Gets the highest existing user id from all Email entries which context id is < 0 (Guests)
     *
     * @return the highest guest id for all Email entries related to a guest account,
     * @throws OXException due an error
     */
    int getHighestGuestId() throws OXException;
}
