/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.database;

import java.sql.Connection;
import com.openexchange.database.Assignment;
import com.openexchange.exception.OXException;
import com.openexchange.guard.database.utils.GuardConnectionWrapper;

/**
 * {@link GuardDatabaseService}
 *
 * @author <a href="mailto:martin.schneider@open-xchange.com">Martin Schneider</a>
 * @since 2.4.0
 */
public interface GuardDatabaseService {

    /**
     * Returns the schema name assigned to the given shardId.
     *
     * @param shardId The id of the shard to return the name for
     * @return String with the shard name related to given id
     */
    String getShardName(int shardId);

    /**
     * Returns a writable connection to registered ox guard database.
     *
     * @return a writable connection to the ox guard database.
     * @throws OXException if no connection can be obtained.
     */
    Connection getWritableForGuard() throws OXException;

    /**
     * Returns a writable connection without timeout to the registered ox guard database. This {@link Connection} should be used to perform administrative things like update tasks.
     *
     * @return a writable connection to the ox guard database.
     * @throws OXException if no connection can be obtained.
     */
    Connection getWritableForGuardNoTimeout() throws OXException;

    /**
     * Returns a read only connection to registered ox guard database.
     *
     * @return a read only connection to the ox guard database.
     * @throws OXException if no connection can be obtained.
     */
    Connection getReadOnlyForGuard() throws OXException;

    /**
     * Returns a read only connection to the ox guard database.
     *
     * @param connection Read only connection to return.
     */
    void backReadOnlyForGuard(Connection connection);

    /**
     * Returns a writable connection to the ox guard database.
     *
     * @param connection Writable connection to return.
     */
    void backWritableForGuard(Connection connection);

    /**
     * Returns a writable connection without timeout to the ox guard database.
     *
     * @param connection Writable connection to return.
     */
    void backWritableForGuardNoTimeout(Connection connection);

    /**
     * Returns a writable connection to the shard database.
     * 
     * @param shardId The identifier of the shard
     * @return a writable connection to the shard database.
     * @throws OXException
     */
    Connection getWritableForShard(int shardId) throws OXException;

    /**
     * Returns a writable connection without timeout to the shard database. This {@link Connection} should be used to perform administrative things like update tasks.
     *
     * @return a writable connection to the ox guard database.
     * @throws OXException if no connection can be obtained.
     */
    Connection getWritableForShardNoTimeout(int shardId) throws OXException;

    /**
     * Returns a read only connection to the shard database.
     * 
     * @param shardId The identifier of the shard
     * @return a read only connection to the shard database.
     * @throws OXException
     */
    Connection getReadOnlyForShard(int shardId) throws OXException;

    /**
     * Returns a read only connection to the shard database.
     *
     * @param shardId The shard id the connection is assigned to.
     * @param connection read only connection to return.
     */
    void backReadOnlyForShard(int shardId, Connection connection);

    /**
     * Returns a writable connection to the shard database.
     *
     * @param shardId The shard id the connection is assigned to.
     * @param connection writable connection to return.
     */
    void backWritableForShard(int shardId, Connection connection);

    /**
     * Returns a writable connection without timeout to the shard database.
     *
     * @param connection Writable connection to return.
     */
    void backWritableForShardNoTimeout(int shardId, Connection connection);

    /**
     * Returns a {@link GuardConnectionWrapper} containing a writable connection to the database the user is associated to.
     * 
     * @param userId The identifier of the user
     * @param contextId The identifier of the context
     * @param shard The identifier of a possible shard
     * @return a {@link GuardConnectionWrapper} with a writable connection and connection related information.
     * @throws OXException
     */
    GuardConnectionWrapper getWritable(int userId, int contextId, int shard) throws OXException;

    /**
     * Returns a writable connection to the destination pool which will be detected based on meta information available within the {@link GuardConnectionWrapper}
     *
     * @param connectionWrapper The wrapper the contained writable connection should be returned for
     */
    void backWritable(GuardConnectionWrapper connectionWrapper);

    /**
     * Returns a {@link GuardConnectionWrapper} containing a read only connection to the database the user is associated to.
     * 
     * @param userId The identifier of the user
     * @param contextId The identifier of the context
     * @param shard The identifier of a possible shard
     * @return a {@link GuardConnectionWrapper} with a read only connection and connection related information.
     * @throws OXException
     */
    GuardConnectionWrapper getReadOnly(int userId, int contextId, int shard) throws OXException;

    /**
     * Returns a read only connection to the destination pool which will be detected based on meta information available within the {@link GuardConnectionWrapper}
     *
     * @param connectionWrapper The wrapper the contained read only connection should be returned for
     */
    void backReadOnly(GuardConnectionWrapper connectionWrapper);

    /**
     * Returns a writable connection to the context database.
     * 
     * @param contextId The identifier of the context database
     * @return a writable connection to the context database.
     * @throws OXException
     */
    Connection getWritable(int contextId) throws OXException;

    /**
     * Returns a read only connection to the context database.
     * 
     * @param contextId The identifier of the context database
     * @return a read only connection to the context database.
     * @throws OXException
     */
    Connection getReadOnly(int contextId) throws OXException;

    /**
     * Returns a read only connection to the context database.
     *
     * @param contextId The context id the connection is assigned to.
     * @param connection read only connection to return.
     */
    void backReadOnly(int contextId, Connection connection);

    /**
     * Returns a writable connection to the context database.
     *
     * @param contextId The context id the connection is assigned to.
     * @param connection writable connection to return.
     */
    void backWritable(int contextId, Connection connection);

    /**
     * Returns a writable connection to the configdb.
     * 
     * @return a writable connection to the configdb.
     * @throws OXException
     */
    Connection getWritableForConfigDB() throws OXException;

    /**
     * Returns a read only connection to the configdb.
     * 
     * @return a read only connection to the configdb.
     * @throws OXException
     */
    Connection getReadOnlyForConfigDB() throws OXException;

    /**
     * Returns a read only connection to the configdb.
     *
     * @param connection read only connection to return.
     */
    void backReadOnlyForConfigDB(Connection connection);

    /**
     * Returns a writable connection to the configdb.
     *
     * @param connection The writable connection to return.
     */
    void backWritableForConfigDB(Connection connection);

    /**
     * Returns a writable connection to the guard MySQL service to register the guard schema.
     * 
     * @return a writable connection to the guard server.
     * @throws OXException
     */
    Connection getWritableForGuardInit() throws OXException;

    /**
     * Returns a writable connection to the shard MySQL service to register the shard schema.
     * 
     * @return a writable connection to the shard server.
     * @throws OXException
     */
    Connection getWritableForShardInit() throws OXException;

    /**
     * Returns a writable connection used to create a new schema (guard or shard).
     *
     * @param connection The writable connection to return.
     */
    void backWritableForInit(Connection connection);

    /**
     * Reloads the underlying database {@link Assignment}s. Should be used after a new database has been registered by using com.openexchange.guard.database.GuardDatabaseService.getWritableForGuardInit() or
     * com.openexchange.guard.database.GuardDatabaseService.getWritableForShardInit()
     * 
     * @throws OXException
     */
    void reloadAssignments() throws OXException;

    /**
     * Returns a connection URL that might get used to create a database connection.
     * 
     * @param hostname The hostname to create a connection URL for
     * @return String with the connection URL that looks like jdbc:mysql://9.9.9.9 (if 9.9.9.9 has been provided)
     */
    String getConnectionUrl(String hostname);
}
