/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.cipher.internal;

import java.security.AlgorithmParameters;
import java.security.Key;
import java.security.spec.KeySpec;
import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.osgi.Services;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;

/**
 * {@link GuardAESCipherServiceImpl}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class GuardAESCipherServiceImpl extends AbstractGuardCipherService {

    private static final Logger logger = LoggerFactory.getLogger(GuardAESCipherServiceImpl.class);

    /**
     * Initialises a new {@link GuardAESCipherServiceImpl}.
     * 
     * @param cipherAlgorithm The cipher algorithm
     * @throws OXException If the specified cipher algorithm does not exist, or if the key length of the specified algorithm exceeds the maximum defined
     */
    public GuardAESCipherServiceImpl(GuardCipherAlgorithm cipherAlgorithm) throws OXException {
        super(cipherAlgorithm);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.crypto.internal.GuardCryptoService#encrypt(java.lang.String, java.security.Key)
     */
    @Override
    public String encrypt(String data, Key secretKey) {
        try {
            Cipher cipher = getCipher();
            cipher.init(Cipher.ENCRYPT_MODE, secretKey);

            AlgorithmParameters parameters = cipher.getParameters();

            byte[] iv = parameters.getParameterSpec(IvParameterSpec.class).getIV();
            byte[] ciphertext = cipher.doFinal(data.getBytes("UTF-8"));

            String encrypted = Base64.encodeBase64String(ciphertext);
            String iVString = Base64.encodeBase64String(iv);
            String combo = iVString + encrypted;

            return combo;
        } catch (Exception e) {
            //TOD throw exception?
            logger.error("Error encrypting", e);
            return "";
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.crypto.GuardCryptoService#encrypt(java.lang.String, java.lang.String)
     */
    @Override
    public String encrypt(String data, String base64Key) {
        SecretKey secretKey = new SecretKeySpec(Base64.decodeBase64(base64Key), cipherAlgorithm.name());
        return encrypt(data, secretKey);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.crypto.GuardCryptoService#encrypt(java.lang.String, java.lang.String, java.lang.String)
     */
    @Override
    public String encrypt(String data, String password, String salt) throws OXException {
        SecretKey secretKey = getKey(password, salt);
        return encrypt(data, secretKey);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.crypto.GuardCryptoService#decrypt(java.lang.String, java.lang.String, java.lang.String)
     */
    @Override
    public String decrypt(String data, String password, String salt) {
        return decrypt(data, password, salt, true);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.cipher.GuardCipherService#decrypt(java.lang.String, java.lang.String, java.lang.String, int)
     */
    @Override
    public String decrypt(String data, String password, String salt, int keyLength) {
        SecretKey secretKey = getKey(password, salt, keyLength);
        return decrypt(data, secretKey);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.crypto.GuardCryptoService#decrypt(java.lang.String, java.lang.String, java.lang.String, boolean)
     */
    @Override
    public String decrypt(String data, String password, String salt, boolean log) {
        try {
            SecretKey secretKey = getKey(password, salt);
            return decrypt(data, secretKey, log);
        } catch (Exception e) {
            //TOD throw exception?
            if (log) {
                logger.info("Bad Pass");
            }
            return "";
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.crypto.GuardCryptoService#decrypt(java.lang.String, java.lang.String)
     */
    @Override
    public String decrypt(String data, String base64Key) {
        SecretKey secretkey = new SecretKeySpec(Base64.decodeBase64(base64Key), cipherAlgorithm.name());
        return decrypt(data, secretkey, true);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.crypto.GuardCryptoService#decrypt(java.lang.String, java.security.Key, boolean)
     */
    @Override
    public String decrypt(String data, Key secretKey, boolean log) {
        try {
            Cipher ciper = getCipher();

            byte[] iv64 = data.substring(0, 24).getBytes("UTF-8");
            byte[] iv = Base64.decodeBase64(iv64);

            data = data.substring(24);

            byte[] databytes = Base64.decodeBase64(data);
            ciper.init(Cipher.DECRYPT_MODE, secretKey, new IvParameterSpec(iv));

            return new String(ciper.doFinal(databytes), "UTF-8");
        } catch (Exception e) {
            //TOD throw exception?
            if (log) {
                logger.info(e.getMessage());
            }
            return "";
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.openexchange.guard.crypto.GuardCryptoService#getRandomKey()
     */
    @Override
    public String getRandomKey() {
        try {
            KeyGenerator keyGen = KeyGenerator.getInstance(cipherAlgorithm.name(), cipherAlgorithm.getProvider());
            keyGen.init(getAESKeyLength());
            SecretKey generatedKey = keyGen.generateKey();

            return Base64.encodeBase64String(generatedKey.getEncoded());
        } catch (Exception e) {
            //TOD throw exception?
            logger.error("Failure while getting random key", e);
            return "";
        }
    }

    ///////////////////////////////////// HELPERS ////////////////////////////////////////////

    /**
     * Returns the AES key length
     * 
     * @return the AES key length
     * @throws OXException
     */
    private int getAESKeyLength() throws OXException {
        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
        return configService.getIntProperty(GuardProperty.aesKeyLength);
    }

    /**
     * Generate key from password and salt
     *
     * @param password The password to generate the key from
     * @param salt The salt
     * @return secretKey The generated SecretKey
     * @throws OXException
     */
    private SecretKey getKey(String password, String salt) throws OXException {
        return getKey(password, salt, getAESKeyLength());
    }

    /**
     * Generate key from password and salt
     *
     * @param password The password to generate the key from
     * @param salt The salt
     * @param keyLength The key length
     * @return secretKey The generated SecretKey
     */
    private SecretKey getKey(String password, String salt, int keyLength) {
        try {
            SecretKeyFactory factory = SecretKeyFactory.getInstance("PBKDF2WithHmacSHA1", "BC");
            KeySpec spec = new PBEKeySpec(password.toCharArray(), salt.getBytes("UTF-8"), 5000, keyLength);
            SecretKey tmpKey = factory.generateSecret(spec);
            SecretKey secretKey = new SecretKeySpec(tmpKey.getEncoded(), "AES");
            return secretKey;
        } catch (Exception e) {
            //TOD throw exception?
            logger.info("Bad Pass Keygen", e);
            return null;
        }
    }
}
