
package com.openexchange.guard.pgp;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.security.NoSuchProviderException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Map;
import java.util.Properties;
import javax.mail.Address;
import javax.mail.BodyPart;
import javax.mail.Message;
import javax.mail.Message.RecipientType;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Session;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import net.htmlparser.jericho.Source;
import org.apache.commons.lang3.StringEscapeUtils;
import org.bouncycastle.openpgp.PGPException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonNull;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.config.OxConfiguration;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.RecipKey;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.exceptions.BadPasswordException;
import com.openexchange.guard.logging.LogAction;
import com.openexchange.guard.mail.GuardMimeMessage;
import com.openexchange.guard.mailcreator.Attachment;
import com.openexchange.guard.mailcreator.Mail;
import com.openexchange.guard.mailcreator.PGPMailCreator;
import com.openexchange.guard.ox.Api;
import com.openexchange.guard.ox.Languages;
import com.openexchange.guard.server.Sender;
import com.openexchange.guard.storage.Storage;
import com.openexchange.guard.util.Core;

public class PGPSender {

    private static Logger logger = LoggerFactory.getLogger(PGPSender.class);
    private boolean draft = false;
    private String draftFolder = null;

    public PGPSender(boolean draft, String draftFolder) {
        this.draft = draft;
        this.draftFolder = draftFolder;
    }

    /**
     * Send PGP Mime email
     *
     * @param emailJson
     * @param attachments
     * @param password
     * @param recipients
     * @param request
     * @param response
     * @param ck
     * @throws MessagingException
     * @throws Exception
     */
    public void sendPGPMail(JsonObject emailJson, ArrayList<Attachment> attachments, String password, ArrayList<RecipKey> recipients, String guestmessage, String pin, int templid, HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie ck) throws MessagingException, Exception {

        boolean inline = Core.getBooleanParameter(request, "pgpinline");

        try {
            int userid = Core.getIntParameter(request, "userid", true);
            int cid = Core.getIntParameter(request, "cid", true);
            LogAction.setUser(userid, cid);
            boolean signit = Core.getBooleanParameter(request, "pgpsign");
            Access acc = new Access();
            GuardKeys senderkey = acc.getKeys(userid, cid);
            if (senderkey == null) {  // Not a good userid and cid
                Core.sendFail(response, "Unable to retrieve sender keys");
                return;
            }

            // Check if inline requested by a recipient
            for (int i = 0; i < recipients.size(); i++) {
                if (recipients.get(i).reqInline()) {
                    inline = true;
                }
            }

            // we need to add the sender to the list of recipients so can be read in sent folder
            RecipKey sender = new RecipKey();
            sender.pgpring = senderkey.pubring;
            recipients.add(sender);

            if (inline && !draft) {  // Drafts will be saved as PGP Mime
                sendPGPInline(emailJson, attachments, senderkey, password, recipients, guestmessage, pin, templid, request, response, ck, signit);
            } else {
                sendPGPMime(emailJson, attachments, senderkey, password, recipients, recipients, guestmessage, pin, templid, request, response, ck);
            }

        } catch (NoSuchProviderException e) {
            logger.error("Error sending email", e);
            Core.sendFail(response, "Fail");
            return;
        } catch (IOException e) {
            logger.error("Problem sending email", e);
            Core.sendFail(response, "Fail");
            return;
        } catch (PGPException e) {
            logger.error("Problem sending email", e);
            Core.sendFail(response, "Fail");
            return;
        }
        if (draft) {
            logger.info("Success draft PGP email");
        } else {
            logger.info("Success send PGP email");
        }
    }

    private void sendPGPInline(JsonObject emailJson, ArrayList<Attachment> attachments, GuardKeys senderkey, String password, ArrayList<RecipKey> recipients, String guestmessage, String pin, int templid, HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie ck, boolean signit) throws Exception {
        ArrayList<RecipKey> guestRecip = new ArrayList<RecipKey>();
        ArrayList<RecipKey> inlineRecip = new ArrayList<RecipKey>();
        int userid = Core.getIntParameter(request, "userid", true);
        int cid = Core.getIntParameter(request, "cid", true);
        LogAction.setUser(userid, cid);
        RecipKey sender = new RecipKey();
        for (int i = 0; i < recipients.size(); i++) {
            RecipKey recip = recipients.get(i);
            if (recip.email == null) {
                sender = recip;
            } else {
                if ((recip.cid < 0) && !isdirect(recip)) {
                    guestRecip.add(recip);
                } else {
                    inlineRecip.add(recip);
                }
            }
        }

        // Send guest emails (these are PGP Mime)
        if (guestRecip.size() > 0) {
            guestRecip.add(sender);
            sendPGPMime(emailJson, attachments, senderkey, password, guestRecip, recipients, guestmessage, pin, templid, request, response, ck);
            // If we have sent some of the emails to a guest, then we need to remove them for the inline emails and mark the headers
            updateJsonRecip(emailJson, inlineRecip);
        }

        // Modify the json for the inline message and encrypt attachments
        if (inlineRecip.size() > 0) {
            inlineRecip.add(sender);
            JsonObject newjson = sendInline(emailJson, attachments, inlineRecip, signit, senderkey, password);
            if (cid > 0) {
                Api ap = new Api(ck, request);
                JsonObject result = ap.sendMail(newjson, attachments);
                if (Core.checkFail(result)) {
                    Core.sendFail(response, "Fail: " + result.get("error_params").getAsString());
                    return;
                } else {
                    // If success, set replied flags
                    handleFlags(emailJson, request, ck, userid, cid);
                }
            } else {
                Sender senderClass = new Sender();
                JsonObject data = emailJson.get("data").getAsJsonObject();
                JsonArray fromarray = data.get("from").getAsJsonArray().get(0).getAsJsonArray();
                if (!senderClass.send(emailJson, attachments, getFromEmailAddress(recipients, fromarray.get(1).getAsString()), sender.lang, Config.getMessageID())) {
                    Core.sendFail(response, "Fail to send to Guest ");
                    return;
                }
            }

        }

        return;
    }

    private void updateJsonRecip(JsonObject emailJson, ArrayList<RecipKey> inlineRecip) throws UnsupportedEncodingException {
        JsonObject data = emailJson.get("data").getAsJsonObject();

        JsonObject headers = data.has("headers") ? data.get("headers").getAsJsonObject() : new JsonObject();
        JsonArray toRecips = data.has("to") ? data.get("to").getAsJsonArray() : new JsonArray();
        if (toRecips.size() > 0) {
            headers.addProperty("X-To-Recips", getRecipsString(toRecips));
        }
        JsonArray ccRecips = data.has("cc") ? data.get("cc").getAsJsonArray() : new JsonArray();
        if (ccRecips.size() > 0) {
            headers.addProperty("X-CC-Recips", getRecipsString(ccRecips));
        }
        JsonArray newToList = new JsonArray();
        JsonArray newCCList = new JsonArray();
        JsonArray newBCCList = new JsonArray();
        for (int i = 0; i < inlineRecip.size(); i++) {
            RecipKey recip = inlineRecip.get(i);
            if (recip.email != null) {
                JsonArray addr = new JsonArray();
                addr.add(new JsonPrimitive(recip.name));
                addr.add(new JsonPrimitive(recip.email));
                if (recip.type == RecipientType.TO) {
                    newToList.add(addr);
                }
                if (recip.type == RecipientType.CC) {
                    newCCList.add(addr);
                }
                if (recip.type == RecipientType.BCC) {
                    newBCCList.add(addr);
                }
            }
        }
        data.remove("to");
        data.remove("cc");
        data.remove("bcc");
        data.add("to", newToList);
        data.add("cc", newCCList);
        data.add("bcc", newBCCList);

    }

    /**
     * Generate comma seperated list of recipients from JsonArray
     *
     * @param recips
     * @return
     * @throws UnsupportedEncodingException
     */
    private String getRecipsString(JsonArray recips) throws UnsupportedEncodingException {
        StringBuilder toString = new StringBuilder();
        for (int i = 0; i < recips.size(); i++) {
            JsonArray recip = recips.get(i).getAsJsonArray();
            if (i > 0) {
                toString.append(", ");
            }
            InternetAddress addr = new InternetAddress(recip.get(1).getAsString(), recip.get(0).getAsString(), "UTF-8");
            toString.append(addr.toString());
        }
        return (toString.toString());
    }

    private void sendPGPMime(JsonObject emailJson, ArrayList<Attachment> attachments, GuardKeys senderkey, String password, ArrayList<RecipKey> recipients, ArrayList<RecipKey> allRecipients, String guestmessage, String pin, int templid, HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie ck) throws Exception {
        int userid = Core.getIntParameter(request, "userid", true);
        int cid = Core.getIntParameter(request, "cid", true);
        LogAction.setUser(userid, cid);
        boolean signit = Core.getBooleanParameter(request, "pgpsign");
        byte[] mime = null;
        String messageId = Config.getMessageID();
        try {
            mime = createMime(emailJson, attachments, allRecipients, messageId);
        } catch (IOException | MessagingException e) {
            logger.error("Error creating Mime email for encrypted email", e);
            Core.sendFail(response, "Error");
            return;
        }
        if (mime == null) {
            Core.sendFail(response, "Error");
            logger.error("Unable to encrypt PGP message");
            return;
        }
        ByteArrayOutputStream encrypted = new ByteArrayOutputStream();

        if (signit) {  // If PGP Signing
            try {
                PGPUtils.signEncryptFile(encrypted, mime, "encrypted.asc", recipients, senderkey, password, true, true);
            } catch (BadPasswordException e) {
                Core.sendFail(response, "Bad password");
                return;
            }
        }
        else {
            PGPUtils.encrypt(encrypted, mime, "encrypted.asc", recipients, true, true);
        }
        String resp = sendMimeEmails(userid, cid, emailJson, encrypted, recipients, allRecipients, guestmessage, pin, messageId, templid, request, ck);
        if (!resp.contains("fail")) {
            handleFlags(emailJson, request, ck, userid, cid);
        } else {
            LogAction.setFail();
            logger.warn("Problem sending email");
            Core.sendFail(response, resp.replace("fail", ""));
            return;
        }
        Core.sendOK(response, resp);
    }

    /**
     * Set original emails as replied to, or delete draft emails that were sent
     *
     * @param emailJson
     * @param request
     * @param ck
     */
    private void handleFlags(JsonObject emailJson, HttpServletRequest request, com.openexchange.guard.server.OxCookie ck, int id, int cid) {
        JsonObject data = emailJson.get("data").getAsJsonObject();
        if (data.has("msgref")) {
            JsonElement msg = data.get("msgref");
            boolean reply = false;
            try {
                if (data.has("headers")) {
                    JsonObject headers = data.get("headers").getAsJsonObject();
                    if (headers.has("In-Reply-To")) {
                        reply = true;
                    }
                }
            } catch (Exception e) {
                logger.error("Error checking in-reply-to headers in email ", e);
            }
            Api ap = new Api(ck, request);
            if (msg.isJsonPrimitive()) {
                ap.handleRef(data.get("msgref").getAsString(), reply, id, cid);
            } else {
                if (msg.isJsonObject()) {
                    JsonObject msgref = msg.getAsJsonObject();
                    ap.handleRef(msgref.get("folder_id").getAsString() + "/" + msgref.get("id").getAsString(), reply, id, cid);
                }
            }
        }
    }

    /**
     * Send the MIME email to OX users or Guest
     *
     * @param userid
     * @param cid
     * @param emailJson
     * @param encrypted
     * @param recipients
     * @param guestmessage
     * @param request
     * @param ck
     * @return
     * @throws MessagingException
     * @throws Exception
     */
    private String sendMimeEmails(int userid, int cid, JsonObject emailJson, ByteArrayOutputStream encrypted, ArrayList<RecipKey> recipients, ArrayList<RecipKey> allRecipients, String guestmessage, String pin, String messageId, int templId, HttpServletRequest request, com.openexchange.guard.server.OxCookie ck) throws MessagingException, Exception {
        JsonObject data = emailJson.getAsJsonObject("data");
        ArrayList<String> failures = new ArrayList<String>();
        Mail mf = new Mail();
        Api ap = new Api(ck, request);
        JsonArray fromarray = data.get("from").getAsJsonArray().get(0).getAsJsonArray();

        String fromName = fromarray.get(0) != JsonNull.INSTANCE ? fromarray.get(0).getAsString().trim() : null;
        String fromEmail = fromarray.get(1).getAsString().trim();

        InternetAddress fromaddr = fromName == null ? new InternetAddress(fromEmail) :
                                                      new InternetAddress(fromEmail,fromName,"UTF-8");

        String subject = data.has("subject") ? data.get("subject").getAsString() : "";
        JsonObject headers = new JsonObject();
        try {
            if (data.has("headers")) {
                headers = data.get("headers").getAsJsonObject();
            }
        } catch (Exception ex) {
            logger.error("Problem extracting headers");
        }

        GuardMimeMessage newMessage = new GuardMimeMessage(Session.getInstance(new Properties()), messageId);
        newMessage.setContent(PGPMailCreator.GetPGPMime(encrypted.toByteArray(), getLanguages(recipients)));

        String uid = EncrLib.getUUID();
        String id = "pgp-" + uid;
        for (int i = 0; i < recipients.size(); i++) {
            if (recipients.get(i).guest == true) {
                saveEmailData(uid, allRecipients, recipients.get(i).userid, recipients.get(i).cid, userid, cid, fromaddr);
            }
        }

        addRecips(newMessage, recipients, true, draft);

        boolean guests = false;  // We need to see if guests are in the list for correct saved sent item
        String resp = null;

        if (!draft) {  // If draft, we don't actually send
            for (int i = 0; i < recipients.size(); i++) {
                if (recipients.get(i).email != null) {  // Make sure legit user to send to
                    RecipKey recip = recipients.get(i);
                    if (!isdirect(recip)) {
                        // Guest account
                        guests = true;
                        Storage.saveEncrObj(
                            recip.userid,
                            recip.cid,
                            id,
                            encrypted.toByteArray());
                        if (recip.newGuestPass != null) {
                            // This is a new Guest account with a new password, need to send notice to the recipient
                            if (com.openexchange.guard.util.Core.checkFail(ap.sendMail(
                                mf.getPasswordEmail(
                                    recip.email,
                                    fromName,
                                    fromName,
                                    fromEmail,
                                    recip.newGuestPass,
                                    recip.lang,
                                    templId,
                                    guestmessage),
                                null).toString())) {
                                failures.add(recip.email);
                                return ("fail");
                            }
                        }
                        JsonObject email = mf.createBlankGuestMail(recip.lang, templId, messageId);
                        if (email == null) {
                            return ("Unable to create template");
                        }
                        email = mf.addTo(email, recip.name, recip.email, recip.type);
                        email = mf.addFrom(email, fromName, fromEmail);
                        email = mf.addSubject(email, subject);
                        email = mf.addPlainText(email, guestmessage);
                        email = mf.noSave(email);
                        try {
                            if (!data.has("headers")) {
                                data.add("headers", new JsonObject());
                            }
                            data.get("headers").getAsJsonObject().addProperty("X-OXGUARD-GUEST", "true");
                        } catch (Exception ex) {
                            logger.error("Problem adding guest header", ex);
                        }
                        try {
                            String url = OxConfiguration.getExternalReaderURL(userid, cid);
                            if (url == null) {
                                url = "https://" + Config.externalReaderPath;
                            }
                            email = mf.addURL(
                                email,
                                url + "?email=" + id +
                                    "&user=" + URLEncoder.encode(recip.email, "ISO-8859-1") + (Languages.customTemplateExists(
                                        templId,
                                        "guesttempl.html") ? ("&templid=" + templId) : "") + "&lang=" + recip.lang);
                        } catch (UnsupportedEncodingException e) {
                            email = mf.addURL(
                                email,
                                "https://" + Config.externalReaderPath + "?email=" + id);
                        }
                        ArrayList<Attachment> attachments = new ArrayList<Attachment>();
                        Attachment guestattach = new Attachment();
                        guestattach.encrContent = encrypted.toByteArray();
                        guestattach.filename = "encrypted.asc";
                        guestattach.type = "application/octet-stream";
                        attachments.add(guestattach);
                        if (cid > 0) {
                            if (com.openexchange.guard.util.Core.checkFail(ap.sendMail(email, attachments).toString())) {
                                return ("fail");
                                //   failures.add(recip.email);
                            }
                        } else {
                            Sender sender = new Sender();
                            if (!sender.send(email, attachments, getFromEmailAddress(recipients, fromEmail), recip.lang, messageId)) {
                                return ("fail");
                            }
                        }

                    }
                }
            }
        }
        if (data.has("priority")) {
            try {
                int priority = data.get("priority").getAsInt();
                newMessage.addHeader("X-Priority", Integer.toString(priority));
                if (priority < 3) {
                    newMessage.addHeader("Importance", "High");
                }
                if (priority > 3) {
                    newMessage.addHeader("Importance", "Low");
                }
            } catch (Exception e) {
                logger.error("problem adding priority to email", e);
            }
        }
        newMessage.setFrom(fromaddr);
        newMessage.setSentDate(new Date());
        newMessage.setSubject(subject, "UTF-8");
        newMessage.saveChanges();
        addReferences(newMessage, headers);
        if (newMessage.getAllRecipients() != null || draft) {  // Standard delivery.  Check we have recipients.

            if (draft) {
                addDraftHeaders(data, newMessage);
            }
            if (cid > 0) { // If not guest sender, we use the OX backend
                ByteArrayOutputStream out = new ByteArrayOutputStream();
                newMessage.writeTo(out);
                if (draft) {
                    logger.debug("Saving draft to " + draftFolder);
                }
                resp = ap.sendMimeMail(out.toString(), draft ? draftFolder : null);
                if (Core.checkFail(resp)) {
                    return ("fail " + Core.getError(resp));
                }
                out.close();
                if (draft) {
                    try {
                        if (data.has("headers")) {
                            if (data.get("headers").getAsJsonObject().has("X-OxGuard-Draft")) {
                                String fullid = data.get("headers").getAsJsonObject().get("X-OxGuard-Draft").getAsString();
                                int i = fullid.lastIndexOf("/");
                                String draftid = fullid.substring(i + 1);
                                String folder = fullid.substring(0, i);
                                ap.deleteDraft(draftid, folder);
                            }
                        }
                    } catch (Exception ex) {
                        logger.error("Problem deleting old draft email", ex);
                    }
                    return (resp);
                }
            } else { // Otherwise, use the SMTP service
                Sender sender = new Sender();
                return (sender.sendMessage(newMessage, getFromEmailAddress(recipients, fromaddr.getAddress()), recipients.get(0).lang) ? "OK" : "fail");
            }
        }
        if ((cid > 0)) {
            createSavedItem(newMessage, recipients, pin, resp, ap);
        }
        if (failures.size() > 0) {
            return ("fail some failed to send");
        }
        return ("OK");
    }

    private String getFromEmailAddress(ArrayList<RecipKey> recipients, String fromaddres) {
        int cid = 0;
        int id = 0;
        for (int i = 0; i < recipients.size(); i++) {// Find at least one member with positive cid, and use that configured address
            if (recipients.get(i).cid > 0) {
                cid = recipients.get(i).cid;
                id = recipients.get(i).userid;
                break;
            }
        }
        return (Mail.getFromAddress(fromaddres, fromaddres, id, cid).get(1));
    }

    /**
     * Need to create combined sent item with all of the recipients, guests, etc
     *
     * @param newmessage
     * @param recipients
     * @param resp
     * @param ap
     * @throws MessagingException
     * @throws IOException
     */
    private void createSavedItem(Message newMessage, ArrayList<RecipKey> recipients, String pin, String resp, Api ap) throws MessagingException, IOException {
        if (resp != null) {  // If we have resp from previously saved email, use to delete
            try {
                JsonObject saved = Core.getJSON(resp);
                if (saved.has("data")) {
                    String folder = saved.get("data").getAsJsonObject().get("folder_id").getAsString();
                    String id = saved.get("data").getAsJsonObject().get("id").getAsString();
                    ap.deleteMail(id, folder);  // First, remove previously saved
                }
            } catch (Exception e) {
                logger.error("Problem getting info to delete saved email", e);
            }
        }
        MimeMessage msg = (MimeMessage) newMessage;
        Address[] x = null;
        msg.setRecipients(Message.RecipientType.TO, x);
        msg.setRecipients(Message.RecipientType.CC, x);
        msg.setRecipients(Message.RecipientType.BCC, x);
        msg.addHeader("X-OxGuard-PIN", pin); // Save pin to sent folder email for retrieval
        addRecips(msg, recipients, true, true);
        msg.saveChanges();
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        msg.writeTo(out);
        ap.sendMimeMail(out.toString(), ap.getSentFolder());
    }

    private void addDraftHeaders(JsonObject data, Message newMessage) throws MessagingException {
        if (!data.has("headers")) {
            return;
        }
        JsonObject headers = data.get("headers").getAsJsonObject();
        for (Map.Entry<String, JsonElement> entry : headers.entrySet()) {
            try {
                if (entry.getValue() != null) {
                    if (entry.getValue().isJsonPrimitive()) {
                        if (addHeader(entry.getKey())) {
                            newMessage.addHeader(entry.getKey(), entry.getValue().getAsString());
                        }
                    }
                }
            } catch (Exception e) {
                logger.error("Non fatal Problem saving draft header: ", e);
            }
        }
    }

    // When saving draft headers, we don't want to save the original DKIM, etc
    private boolean addHeader(String header) {
        try {
            switch (header) {
                case "Message-ID":
                    return (false);
                case "DKIM-Signature":
                    return (false);
                case "Received":
                    return (false);
                case "X-FromIP":
                    return (false);
            }
            return (true);
        } catch (Exception e) {
            return (true);
        }
    }

    /**
     * Add reply-to and refernces headers to outgoing email if present
     *
     * @param newmessage
     * @param headers
     * @throws MessagingException
     */
    private void addReferences(Message newMessage, JsonObject headers) throws MessagingException {
        if (headers == null) {
            return;
        }
        try {
            if (headers.has("In-Reply-To")) {
                String replyto = "";
                if (headers.get("In-Reply-To").isJsonArray()) { // Occasionally, this is sent as an array
                    replyto = headers.get("In-Reply-To").getAsJsonArray().get(0).getAsString();
                } else {
                    replyto = headers.get("In-Reply-To").getAsString();
                }
                newMessage.addHeader("In-Reply-To", replyto);
            }
            if (headers.has("References")) {
                String ref = "";
                if (headers.get("References").isJsonArray()) {
                    ref = headers.get("References").getAsJsonArray().get(0).getAsString();
                } else {
                    ref = headers.get("References").getAsString();
                }
                newMessage.addHeader("References", ref);
            }
        } catch (Exception e) {
            logger.error("Error adding headers to email, ", e);
        }
    }

    /**
     * Get an arraylist of languages from all recipients
     *
     * @param recipients
     * @return
     */
    private ArrayList<String> getLanguages(ArrayList<RecipKey> recipients) {
        ArrayList<String> languages = new ArrayList<String>();
        for (RecipKey recip : recipients) {
            if (!languages.contains(recip.lang)) {
                languages.add(recip.lang);
            }
        }
        return (languages);
    }

    /**
     * Evaluate if Guest recipient wants direct email.
     *
     * @param recip
     * @return
     */
    private static boolean isdirect(RecipKey recip) {
        try {
            if (recip.cid > 0)
            {
                return (true);  // Members get direct
            }
            if (recip.guest == false)
            {
                return (true);  // Have uploaded key, not guest
            }
            if (recip.settings == null) {
                return (false);
            }
            if (!recip.settings.has("direct")) {
                return (false);
            }
            return (recip.settings.get("direct").getAsBoolean());
        } catch (Exception e) {
            logger.error("Trouble checking is direct", e);

            return (false);
        }
    }

    /**
     * Add all of the recipients to the Mail Message
     *
     * @param newMessage
     * @param recipients
     * @param addBcc Don't add BCC if this is for display later
     * @param all Don't check if guest or not -- if just draft, mime header
     * @throws UnsupportedEncodingException
     * @throws MessagingException
     */
    private void addRecips(Message newMessage, ArrayList<RecipKey> recipients, boolean addBcc, boolean all) throws UnsupportedEncodingException, MessagingException {
        ArrayList<Address> to = new ArrayList<Address>();
        ArrayList<Address> cc = new ArrayList<Address>();
        ArrayList<Address> bcc = new ArrayList<Address>();
        for (int i = 0; i < recipients.size(); i++) {
            RecipKey recip = recipients.get(i);
            if (recip.email != null) {
                InternetAddress addr = new InternetAddress(recip.email, recip.name.replace("\"", ""), "UTF-8");
                // If not draft email, only saving the addresses that are directly mailed (IE not guest emails)
                if (all || isdirect(recip)) {
                    if (recip.type == RecipientType.TO) {
                        to.add(addr);
                    }
                    if (recip.type == RecipientType.CC) {
                        cc.add(addr);
                    }
                    if (recip.type == RecipientType.BCC) {
                        bcc.add(addr);
                    }
                }
            }
        }
        if (to.size() > 0) {
            newMessage.addRecipients(RecipientType.TO, to.toArray(new Address[to.size()]));
        }
        if (cc.size() > 0) {
            newMessage.addRecipients(RecipientType.CC, cc.toArray(new Address[cc.size()]));
        }
        if (addBcc && bcc.size() > 0) {
            newMessage.addRecipients(RecipientType.BCC, bcc.toArray(new Address[bcc.size()]));
        }
    }

    private JsonObject sendInline(JsonObject emailJson, ArrayList<Attachment> attachments, ArrayList<RecipKey> recipients, boolean signit, GuardKeys keys, String password) throws Exception {
        JsonObject data = emailJson.getAsJsonObject("data");
        JsonArray mailattach = data.get("attachments").getAsJsonArray();
        for (int i = 0; i < mailattach.size(); i++) {
            JsonObject attach = mailattach.get(i).getAsJsonObject();
            if (attach.has("content")) {
                String content = attach.get("content").isJsonNull() ? "" : attach.get("content").getAsString();
                if (content.equals("")) {
                    content = "  ";  // if blank content, just add some white spaces to establish PGP Email
                }
                if (content.length() > 0) {
                    ByteArrayOutputStream encrypted = new ByteArrayOutputStream();
                    if (signit) {
                        PGPUtils.signEncryptFile(encrypted, content.getBytes("UTF-8"), "encrypted.asc", recipients, keys, password, true, true);
                    } else {
                        PGPUtils.encrypt(encrypted, content.getBytes("UTF-8"), "encrypted.asc", recipients, true, true);
                    }
                    attach.remove("content");
                    String encr = encrypted.toString();
                    String type = attach.get("content_type").getAsString();
                    if (type.contains("altern") || type.contains("html")) {
                        encr = encr.replace("\n", "<br>");
                    }
                    attach.addProperty("content", encr);
                    encrypted.close();
                }
            }
        }
        // Encrypt attachments
        for (int j = 0; j < attachments.size(); j++) {
            Attachment attach = attachments.get(j);
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            if (signit) {
                PGPUtils.signEncryptFile(out, attach.content, attach.filename, recipients, keys, password, true, true);
            } else {
                PGPUtils.encrypt(out, attach.content, attach.filename, recipients, true, true);
            }
            out.close();
            attach.encrContent = out.toByteArray();
            attach.type = "application/octet-stream";
            attach.filename = attach.filename + ".pgp";
            attach.pgp = true;
        }
        return (emailJson);
    }

    private void saveEmailData(String emailID, ArrayList<RecipKey> recipients, int userid, int cid, int senderid, int sendercid, InternetAddress from) throws Exception {
        Access acc = new Access();
        StringBuilder addxml = new StringBuilder();
        addxml.append("<share>");
        for (RecipKey user : recipients) {
            if (user.email != null) {
                addxml.append("<u><e>" + StringEscapeUtils.escapeXml11(user.email) + "</e><i>" + user.userid + "</i><c>" + user.cid + "</c></u>");
            }
        }
        // add sender
        addxml.append("<u><e>" + StringEscapeUtils.escapeXml11(from.getAddress()) + "</e><r>N</r><i>" + senderid + "</i><c>" + sendercid + "</c></u>");
        addxml.append("</share>");
        acc.storeItemID(emailID, userid, cid, senderid, sendercid, 0, 0, addxml.toString(), "");
    }

    /**
     * Add the from, subject, sent time to PGP Mime body
     *
     * @param data
     * @param msg
     */
    private void addHeaders(JsonObject data, MimeMessage msg) {
        try {
            JsonArray fromarray = data.get("from").getAsJsonArray().get(0).getAsJsonArray();
            String senderName = fromarray.get(0) != JsonNull.INSTANCE ? fromarray.get(0).getAsString() : null;
            String senderEmail = fromarray.get(1).getAsString();
            InternetAddress from = senderName == null ? new InternetAddress(senderEmail) :
                                                        new InternetAddress(senderEmail,senderName, "UTF-8");
            String subject = data.has("subject") ? data.get("subject").getAsString() : "";
            msg.setFrom(from);
            msg.setSubject(subject, "UTF-8");
            msg.setSentDate(new Date());
        } catch (Exception ex) {
            logger.error("Error adding headers to PGP Mime encrypted content ", ex);
        }

    }

    private byte[] createMime(JsonObject emailJson, ArrayList<Attachment> attachments, ArrayList<RecipKey> recipients, String messageId) throws IOException, MessagingException {
        JsonObject data = emailJson.getAsJsonObject("data");

        MimeMultipart mp = new MimeMultipart("mixed");
        JsonArray mailattach = data.get("attachments").getAsJsonArray();
        for (int i = 0; i < mailattach.size(); i++) {
            JsonObject attach = mailattach.get(i).getAsJsonObject();
            if (attach.has("content_type") || attach.has("type")) {
                String type = attach.has("content_type") ? attach.get("content_type").getAsString() : attach.get("type").getAsString();
                switch (type.toLowerCase()) {
                    case "alternative":
                        MimeBodyPart multipart = new MimeBodyPart();
                        String htmlcontent = attach.get("content").getAsString();
                        Multipart alternate = new MimeMultipart("alternative");
                        BodyPart pt = new MimeBodyPart();
                        StringReader sr = new StringReader(htmlcontent);
                        Source src = new Source(sr);
                        pt.setContent(src.getRenderer().toString(), "text/plain; charset=utf-8");
                        alternate.addBodyPart(pt);
                        sr.close();
                        BodyPart htm = new MimeBodyPart();
                        htm.setContent(htmlcontent, "text/html; charset=utf-8");
                        alternate.addBodyPart(htm);
                        multipart.setContent(alternate);
                        mp.addBodyPart(multipart);
                        break;

                    case "text/plain":
                        if (attach.has("content")) {
                            BodyPart ptext = new MimeBodyPart();
                            ptext.setContent(attach.get("content").getAsString(), "text/plain; charset=utf-8");
                            mp.addBodyPart(ptext);
                        }
                        break;
                    case "text/html":
                        if (attach.has("content")) {
                            BodyPart html = new MimeBodyPart();
                            html.setContent(attach.get("content").getAsString(), "text/html; charset=utf-8");
                            mp.addBodyPart(html);
                        }
                        break;
                }
            }
        }
        // Add attachments
        for (int i = 0; i < attachments.size(); i++) {
            Attachment attach = attachments.get(i);
            BodyPart pt = new MimeBodyPart();
            if (attach.filename != null) {
                pt.setFileName(attach.filename);
            }
            if (attach.type.contains("text")) {
                pt.setContent(new String(attach.content, "UTF-8"), attach.type);
            } else {
                pt.setContent(attach.content, attach.type);
                if (attach.content_id != null) {
                    pt.setHeader("Content-ID", "<" + attach.content_id + ">");
                    pt.setDisposition("inline");
                }
            }
            mp.addBodyPart(pt);
        }
        Properties prop = new Properties();
        GuardMimeMessage message = new GuardMimeMessage(Session.getInstance(prop), messageId);
        addRecips(message, recipients, false, true);  // add the recipients, but not bcc
        addHeaders(data, message);
        message.setContent(mp);
        message.saveChanges();
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        message.writeTo(out);
        byte[] mime = out.toByteArray();
        out.close();
        return (mime);
    }
}
