/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.ox;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import com.google.gson.JsonObject;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.config.OxConfiguration;
import com.openexchange.guard.translate.GetText;
import com.openexchange.guard.translate.TemplateTransformation;

public class Languages {

	private static Logger logger = LoggerFactory.getLogger(Languages.class);

	public static Map<String, Map<String, String>> avail = new ConcurrentHashMap<String, Map<String, String>>();

    public static String default_code = "";

    final static String templ_location = Config.codePath + "templates";

    final static String i18n_location = Config.codePath + "l10n";

    public Languages() {
        if (avail.size() == 0) { // If not yet populated, get available languages from files
            loadAvail();
            // initconfig();
        }
    }

    /**
     * Read help file using provided language
     * @param lang
     * @return
     * @throws IOException
     */
    public static String getHelpFile (String lang) throws IOException {
        String filename = i18n_location+ "/help/help_" + lang + ".txt";
        File f = new File(filename);
        if (f.exists()) {
            return (GetText.readFile(filename));
        } else {
            logger.warn("no help file found for language " + lang);
            filename = i18n_location+ "/help/help_" + default_code + ".txt";
            f = new File (filename);
            if (f.exists()) {
                return (GetText.readFile(filename));
            } else {
                logger.error("Unable to load default help file");
                return (null);
            }
        }
    }
    /**
     * Get the tranlation of template based on language
     *
     * @param template
     * @param lang Language code, will default if not found
     * @return
     */
    public static String getTemplateTranslation(String template, String lang, List<TemplateTransformation> transformations) {
        return (GetText.processTemplate(templ_location + "/" + template, lang,transformations));
    }

    public static String getTranslation(String text, String lang) {
        return (GetText.gettext(text, lang));
    }

    public static String getTranslation (String template, String lang, int cid, int id) {
    	int templID = OxConfiguration.getTemplateId(cid, id);
    	return (getTranslation(template, lang, templID));
    }

    public static String getTranslation(String template, String lang, int templId) {
        return getTranslation(template,lang,templId,null);
    }

    /**
     * Get translation of template, checking for custom based on templId
     *
     * @param template name of template
     * @param lang lanaguage for translation
     * @param templId of custom template if exists, or default provided
     * @return
     */
    public static String getTranslation(String template, String lang, int templId,List<TemplateTransformation> transformations) {
    	if (templId == 0) {
            return (getTemplateTranslation(template, lang,transformations));
        }
        File f = new File(templ_location + "/" + templId + "-" + template);
        if (f.exists()) {
            return (getTemplateTranslation(templId + "-" + template, lang,transformations));
        } else {
        	logger.warn("no template found for custom template " + template + " id " + templId);
            return (getTemplateTranslation(template, lang,transformations));
        }
    }

    public static boolean customTemplateExists(int templId, String template) {
    	if (templId == 0) {
            return (false);
        }
        File f = new File(templ_location + "/" + templId + "-" + template);
        if (f.exists()) {
            return (true);
        } else {
        	logger.warn("no custom templates found for teplate id " + templId);
            return (false);
        }
    }

    /**
     * Gets the full country name of a language code in the language specified
     *
     * @param code
     * @param lang Language for the output
     * @return
     */
    public String getTranslatedName(String code, String lang) {
        if (!avail.containsKey(code)) {
            if (code.length() > 2) {
                if (avail.containsKey(code.substring(0, 2))) {
                    code = code.substring(0, 2);
                } else {
                    return ("");
                }
            } else {
                return ("");
            }
        }
        Map translations = avail.get(code);
        if (!translations.containsKey(lang)) {
            if (lang.length() > 2) {
                if (translations.containsKey(lang.substring(0, 2))) {
                    lang = lang.substring(0, 2);
                } else {
                    return ("");
                }
            } else {
                return ("");
            }
        }
        return (String) (translations.get(lang));
    }

    // Check if a language is available, if not, return default
    public static String isavail(String code) {
        if (code == null) {
            return (default_code);
        }
        if (avail.containsKey(code)) {
            return (code);
        }
        if (avail.containsKey(code.substring(0, 2))) {
            return (code.substring(0, 2));
        }
        return (default_code);// Not available, so return default
    }

    // Gets Json of available languages
    public static JsonObject getJsonAvail(String lang) {
    	JsonObject langJson = new JsonObject();
        Iterator it = avail.entrySet().iterator();
        int count = 0;
        while (it.hasNext()) {
            Map.Entry ent = (Map.Entry) it.next();
            Map trans = (Map) ent.getValue();
            if (!trans.containsKey(lang)) {
                lang = lang.substring(0, 2);
            }
            if (!trans.containsKey(lang))
             {
                lang = default_code; // If we don't have that translation, server default language
            }
            langJson.addProperty((String) ent.getKey(), (String) trans.get(lang));
        }
        return (langJson);
    }

    /**
     * Load available languages into local memory
     */
    private void loadAvail() {

        String xmlfile = "";

        try {
            FileInputStream inputStream = new FileInputStream(i18n_location + "/languages.xml");
            xmlfile = IOUtils.toString(inputStream);
            inputStream.close();
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            org.w3c.dom.Document doc = builder.parse(new InputSource(new ByteArrayInputStream(xmlfile.getBytes("utf-8"))));
            doc.getDocumentElement().normalize();
            default_code = doc.getElementsByTagName("default").item(0).getTextContent().trim();
            GetText.setDefault(default_code);
            NodeList languages = doc.getElementsByTagName("language");
            for (int i = 0; i < languages.getLength(); i++) {
                Node lang = languages.item(i);
                NodeList elements = lang.getChildNodes();
                String id = elements.item(1).getTextContent();
                File mapping = new File(i18n_location + "/templates-" + id + ".po");
                if (mapping.exists()) {// Make sure our translation file actually exists
                    logger.info("Loading translation file " + mapping.getName());
                    GetText.loadPOfile(i18n_location + "/templates-" + id + ".po", id);
                    Map translated = new HashMap<String, String>();
                    for (int j = 2; j < elements.getLength(); j++) {
                        String langcode = elements.item(j).getNodeName();
                        if (langcode != "#text") {
                            String value = elements.item(j).getTextContent();
                            translated.put(langcode, value); // Put the country name to the translated list by tag
                        }
                    }
                    avail.put(id, translated);
                } else {
                    logger.debug("Unable to load translation file " + mapping.getPath() + "/" + mapping.getName());
                }
            }
            GetText.loadFiles(i18n_location, "main");
        } catch (Exception ex) {
            logger.error("Error loading avaiable translations", ex);

        }
    }


}
