package com.openexchange.guard.database.pgpKeys;

import java.util.List;

/**
 * {@link PGPKeysStorage} provides access to the stored mappings from userid and contextId to a PGP Key ID
 *
 */
public interface PGPKeysStorage {
    /**
     * Gets the mapping for internal user, which means no guest user's are considered
     * @param email the email to get the mappings for
     * @return mappings for the the given email
     * @throws Exception due an error
     */
    List<PGPKeys> getInternalByEmail(String email) throws Exception;

    /**
     * Gets the mapping for a given hexId
     * @param hexId the hexId to get the mapping for
     * @return the mapping for the given hexId, or null if no such mapping was found
     * @throws Exception due an error
     */
    PGPKeys getByHexId(String hexId) throws Exception;

    /**
     * Gets the mapping for a given id
     * @param id the Id to get the mapping for
     * @return the mapping for the given Id, or null if no such mapping was found
     * @throws Exception due an error
     */
    PGPKeys getById(long id) throws Exception;

    /**
     * Gets the mapping for the given ids
     * @param ids the ids of the mappings to get
     * @return the mappings related to the given ids
     * @throws Exception due an error
     */
    List<PGPKeys> getByIds(List<Long> ids) throws Exception;

    /**
     * Gets the mapping for the given key-id in the given context
     * @param contextId the context id
     * @param keyId the key id to get the mapping for
     * @return the mappings related to the given ids
     * @throws Exception due an error
     */
    PGPKeys getByKeyIdInContext(int contextId, long keyId) throws Exception;

    /**
     * Creates a mapping or updates an existing one
     * @param id the id
     * @param email the mail
     * @param hexId the hexId
     * @param contextId the context ID
     * @param keyId the key id
     * @throws Exception due an error
     */
    public void insertOrUpdate(long id,String email,String hexId,int contextId,long keyId) throws Exception;

    /**
     * Deletes a mapping entry for a key
     * @param keyId the id of the key to delete the mapping entry for
     * @throws Exception due an error
     */
    public void deleteByKeyId(long keyId) throws Exception;

    /**
     * Deletes all mappings for a given email
     * @param email the email to delete the mappings for
     * @throws Exception due an error
     */
    public void deleteByEmail(String email) throws Exception;

    /**
     * Deletes a mapping entry for a key in a given context
     * @param keyId the id of the key to delete the mapping entry for
     * @param contextId the context id
     * @throws Exception due an error
     */
    public void deleteByKeyIdInContext(long keyId, int contextId) throws Exception;
}
