package com.openexchange.guard.database.ogPGPKeys;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;
import com.openexchange.guard.database.Sharding;
import com.openexchange.guard.util.ListUtil;

public class RestDbOGPGPKeysStorage implements OGPGPKeysStorage {

    private final static String VALUES_LIST = "{@VALUES}";
    private static Logger logger = LoggerFactory.getLogger(RestDbOGPGPKeysStorage.class);
    
    /**
     * Internal method to convert a string of whitespace separated key ids into a list of key ids
     * @param s the string of whitespace separated key ids
     * @return a list of key ids
     */
    private List<String> stringToList(String s){
        String[] idList = s.trim().split("\\s+");
        return Arrays.asList(idList);
    }

    @Override
    public List<OGPGPKey> getForUser(int userId, int contextId) throws Exception {
        DbQuery db = new DbQuery();
        ArrayList<OGPGPKey> ret = new ArrayList<OGPGPKey>();
        try {
            DbCommand command = new DbCommand(OGPGPKeysSql.SELECT_BY_USER_ID_STMT);
            command.addVariables(userId);
            command.addVariables(contextId);
            command.addVariables(userId);
            db.readFromDB(command, contextId);
            while(db.next()) {
                ret.add(new OGPGPKey(db.rs.getInt("userid"),
                                     contextId,
                                     db.rs.getString("Email"),
                                     stringToList(db.rs.getString("ids")),
                                     db.rs.getString("PGPPublic"),
                                     db.rs.getInt("share_level"),
                                     db.rs.getBit("inline")));
            }
            return ret;
        }
        finally {
            db.close();
        }
    }

    @Override
    public List<OGPGPKey> getForUserByEmail(int userId,
                                            int contextId,
                                            List<String> emails,
                                            boolean shard) throws Exception {
        DbQuery db = new DbQuery();
        ArrayList<OGPGPKey> ret = new ArrayList<OGPGPKey>();
        try{
            String sql = OGPGPKeysSql.SELECT_BY_USER_ID_AND_MAILS_STMT;
            DbCommand command = new DbCommand(sql.replace(VALUES_LIST, ListUtil.listToDbValuesList(emails.size())));
            command.addVariables(userId);;
            command.addVariables(contextId);
            command.addVariables(userId);
            for(String email : emails) {
                command.addVariables(email);
            }

            if(!shard) {
                //OX Guard DB
                db.readFromDB(command, contextId);
            } else {
                //Guest shard
                db.read(command, userId, contextId);
            }
            while(db.next()) {
                ret.add(new OGPGPKey(db.rs.getInt("userid"),
                                     contextId,
                                     db.rs.getString("Email"),
                                     stringToList(db.rs.getString("ids")),
                                     db.rs.getString("PGPPublic"),
                                     db.rs.getInt("share_level"),
                                     db.rs.getBit("inline")));
            }
            return ret;
        } catch (Exception e) {
            if (e.getMessage().contains("doesn't exist")) {
                if (contextId > 0) {
                    logger.info("Creating tables for context " + contextId);
                    Sharding.checkTables(contextId, true);
                }
            } else 
                logger.error("Problem getting PGP Keys ", e);
            return ret;
        }
        finally {
            db.close();
        }
    }

    @Override
    public OGPGPKey getForUserByEmailAndIds(int userId, int contextId, String email, List<String> keyIds) throws Exception {
        DbQuery db = new DbQuery();
        try{
            DbCommand command = new DbCommand(OGPGPKeysSql.SELECT_BY_USER_ID_AND_MAIL_STMT);
            command.addVariables(email);
            command.addVariables(ListUtil.listToString(keyIds, ListUtil.WHITE_SPACE_SEP));
            command.addVariables(contextId);
            command.addVariables(userId);
            db.readFromDB(command, contextId);
            if(db.next()) {
               return new OGPGPKey(db.rs.getInt("userid"),
                                   db.rs.getInt("cid"),
                                   db.rs.getString("Email"),
                                   stringToList(db.rs.getString("ids")),
                                   db.rs.getString("PGPPublic"),
                                   db.rs.getInt("share_level"),
                                   db.rs.getBit("inline"));
            }
            return null;

        }
        finally {
            db.close();
        }
    }

    @Override
    public List<OGPGPKey> getForUserByIds(int userId, int contextId, List<String> keyIds) throws Exception {
        DbQuery db = new DbQuery();
        ArrayList<OGPGPKey> ret = new ArrayList<OGPGPKey>();
        try {
            DbCommand command = new DbCommand(OGPGPKeysSql.SELECT_BY_IDS_STMT);
            command.addVariables(contextId);
            command.addVariables(userId);
            command.addVariables(ListUtil.listToString(keyIds,ListUtil.WHITE_SPACE_SEP) + "%");
            db.readFromDB(command, contextId);
            while(db.next()) {
                ret.add(new OGPGPKey(db.rs.getInt("userid"),
                                    db.rs.getInt("cid"),
                                    db.rs.getString("Email"),
                                    stringToList(db.rs.getString("ids")),
                                    db.rs.getString("PGPPublic"),
                                    db.rs.getInt("share_level"),
                                    db.rs.getBit("inline")));
            }
            return ret;
        }
        finally {
            db.close();
        }
    }

    @Override
    public void InsertOrUpdate(int userId, int contextId, String email, List<String> keyIds, String ascKeyData) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGPGPKeysSql.INSERT_OR_UPDATE_STMT);
            String keyList = ListUtil.listToString(keyIds,ListUtil.WHITE_SPACE_SEP);
            //--for the insert
            command.addVariables(keyList);
            command.addVariables(email);
            command.addVariables(ascKeyData);
            command.addVariables(userId);
            command.addVariables(0); //Default share-level of 0
            command.addVariables(contextId);
            //--for the update
            command.addVariables(ascKeyData);
            command.addVariables(keyList);
            db.writeToDB(command, contextId);
        }
        finally {
            db.close();
        }
    }

    @Override
    public void Delete(int userId, int contextId, List<String> keyIds) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGPGPKeysSql.DELETE_STMT);
            command.addVariables(contextId);
            command.addVariables(userId);
            command.addVariables(ListUtil.listToString(keyIds,ListUtil.WHITE_SPACE_SEP));
            db.write(command, userId, contextId);
        }
        finally {
            db.close();
        }
    }

    @Override
    public void UpdateShareLevel(int userId, int contextId, List<String> keyIds, int shareLevel) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGPGPKeysSql.UPDATE_SHARE_STMT);
            command.addVariables(shareLevel);
            command.addVariables(contextId);
            command.addVariables(userId);
            command.addVariables(ListUtil.listToString(keyIds,ListUtil.WHITE_SPACE_SEP));
            db.write(command, userId, contextId);
        }
        finally {
            db.close();
        }
    }

    @Override
    public void UpdateInlineMode(int userId, int contextId, List<String> keyIds, boolean useInline) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(OGPGPKeysSql.UPDATE_INLINE_STMT);
            command.addVariables(useInline);
            command.addVariables(contextId);
            command.addVariables(userId);
            command.addVariables(ListUtil.listToString(keyIds,ListUtil.WHITE_SPACE_SEP));
            db.write(command, userId, contextId);
        }
        finally {
            db.close();
        }
    }
}
