package com.openexchange.guard.database.fileCache;

import java.util.ArrayList;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;
import com.openexchange.guard.database.OxResultSet;

/**
 * {@link RestDbFileCacheStorage} provides a REST DB implementation of the FileCacheStorage
 */
public class RestDbFileCacheStorage implements FileCacheStorage {

    private static Logger logger = LoggerFactory.getLogger(RestDbFileCacheStorage.class);

    /**
     * Internal method to create a FileCache item from a REST-DB result set
     * @param rs the result set
     * @return a FileCacheItem from the given result set
     * @throws Exception due an error
     */
    private FileCacheItem CreateItemFromDbResult(OxResultSet rs) throws Exception {
        return new FileCacheItem(rs.getString("itemID"),
                                 rs.getInt("userid"),
                                 rs.getString("location"),
                                 rs.getDate("lastDate"));
    }


    @Override
    public FileCacheItem getById(String itemId) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(FileCacheSql.GET_BY_ID_STMT);
            command.addVariables(itemId);
            db.readOG(command);
            if(db.next()) {
                return CreateItemFromDbResult(db.rs);
            }
            return null;
        }
        catch(Exception e) {
            logger.error("Error getting FileCacheItem " + itemId, e);
            throw e;
        }
        finally {
            db.close();
        }
    }

    @Override
    public void Insert(String itemId, int userId, String path) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand com = new DbCommand(FileCacheSql.ADD_STMT);
            com.addVariables(itemId);
            com.addVariables(userId);
            com.addVariables(path);
            db.writeOxGuard(com);
        } catch (Exception e) {
            logger.error("Error adding item " + itemId + " to cache", e);
            throw e;
        }
        finally {
            db.close();
        }
    }

    @Override
    public void updateLastDate(String itemId) throws Exception  {
        DbQuery db = new DbQuery();
        try {
            DbCommand commmand = new DbCommand(FileCacheSql.UPDATE_LAST_DATE_STMT);
            commmand.addVariables(itemId);
            db.writeOxGuard(commmand);
        } catch (Exception e) {
            logger.error("Error updating cache for item " + itemId, e);
            throw e;
        }
        finally {
            db.close();
        }
    }

    @Override
    public List<FileCacheItem> findOld(int days) throws Exception {

        ArrayList<FileCacheItem> ret = new ArrayList<FileCacheItem>();
        DbQuery db = new DbQuery();

        try {
            DbCommand command = new DbCommand(FileCacheSql.SELECT_BY_LAST_DATE_STMT);
            command.addVariables(days);
            db.readOG(command);
            while (db.next()) {
                ret.add(CreateItemFromDbResult(db.rs));
            }
        } catch (Exception e) {
            logger.error("Error reading old FileCache items not accessed since " + days + " days", e);
            throw e;
        }
        finally {
            db.close();
        }
        return ret;
    }


    @Override
    public void delete(String itemId) throws Exception {
        DbQuery db = new DbQuery();
        try {
            DbCommand command = new DbCommand(FileCacheSql.DELETE_STMT);
            command.addVariables(itemId);
            db.writeOxGuard(command);
        }
        catch(Exception e) {
            throw e;
        }
        finally {
            db.close();
        }
    }
}
