/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.database;

import java.util.ArrayList;
import java.util.Iterator;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.openexchange.guard.config.Capabilities;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.config.OxConfiguration;
import com.openexchange.guard.database.ogEmail.OGEmail;
import com.openexchange.guard.database.ogEmail.RestDbOGEmailStorage;
import com.openexchange.guard.database.ogKeyTable.OGKeyTableStorage;
import com.openexchange.guard.database.ogKeyTable.RestDbOGKeyTableStorage;
import com.openexchange.guard.database.userInfo.RestDbUserInfo;
import com.openexchange.guard.database.userInfo.UserInfo;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.mailcreator.Mail;
import com.openexchange.guard.ox.Languages;
import com.openexchange.guard.pgp.PGPPublicHandler;
import com.openexchange.guard.pgp.PgpKeyLookup;
import com.openexchange.guard.pgp.PgpKeys;
import com.openexchange.guard.server.Sender;
import com.openexchange.guard.validator.EmailValidator;

/**
 * Class for looking up public key, and creating keys if needed
 *
 * @author greg
 */
public class GetPubKey {

	private static Logger logger = LoggerFactory.getLogger(GetPubKey.class);

    public RecipKey getPubKey(String email, int senderID, int senderCID, String from, boolean sending, boolean remote) throws Exception {
        return (getPubKey(email, senderID, senderCID, from, sending, remote, "", "", null, null));
    }

    /**
     * Looks up the public key for a recipient. If an OX member, creates if necessary For non-ox members, creates key only if sending
     *
     * @param email
     * @param senderCID, only needed if not remote - used to create guest
     * @param sending Boolean if sending the email
     * @param remote True if from remote lookup - guest accounts wont be created
     * @return
     * @throws Exception
     */
    public RecipKey getPubKey(String email, int senderID, int senderCID, String from, boolean sending, boolean remote, String guestlang, String fromName, String pin, EmailInfo info) throws Exception {

    	if (!EmailValidator.validate(email)){
            return (null); // make sure email address
        }
        // Get connection
        RecipKey result = null;

        // First, we'll check if there is a key uploaded by the user
        GuardKeys key = null;
        // If there is a senderID, check for upload

        if (senderID != 0) {
            key = PGPPublicHandler.getPGPKeysFromEmail(email, senderID, senderCID);
        }
        if (key == null && info != null) {
            key = PGPPublicHandler.getPGPKeysFromEmail(email, info.owner_id, info.owner_cid);
        }
        if (key!=null) {
            result = new RecipKey(key);
            return (result);
        }

        // Next check if we have in our masterKeyTable
        result = checkMaster(email);
        if (result != null)
         {
            return(result);
            // If not found, check to see if we have it in cache for outside user
   //     if ((result == null) && !remote) {
   //        result = checkCache(email);
   //    }
        }

        if (result == null) {
            // If not in our keytable, and not in cache, then let's search the OX user database to see if member
            try {
                com.openexchange.guard.database.Utilities util = new com.openexchange.guard.database.Utilities();
                JsonObject userdat = util.getUser(email);
                if (userdat.has(email)) {
	                JsonObject user = userdat.get(email).getAsJsonObject();
	                int userid = user.get("uid").getAsInt();
	                int cid = user.get("cid").getAsInt();
	                // First, let's check if this is an alias for a known user
	                Access acc = new Access();
	                GuardKeys aliasKey = acc.getKeys(userid, cid);
	                if (aliasKey != null) {
	                    result = recipKeyFromGuardKey(aliasKey);
	                    return (result);
	                }
	                UserInfo userInfo = new RestDbUserInfo();
	                if (userInfo.hasWebmail(userid, cid) && hasGuard(userid, cid)) {  // Verify the user has webmail
	                    result = new RecipKey();
	                    result.userid = userid;
	                    result.cid = cid;
	                    result.lang = user.get("user").getAsJsonObject().get("language").getAsString();
	                    if (user.get("user").getAsJsonObject().has("displayName")) {
	                        result.name = user.get("user").getAsJsonObject().get("displayName").getAsString();
	                    } else {
	                        result.name = "";
	                    }
	                    result.email = email;
	                    if (sending) {
	                        logger.debug("Found id for " + email + " in main ox users, creating key");
	                        Keymanager keyman = new Keymanager();
	                        result.newGuestPass = EncrLib.randomPass(Config.newPassLength);
	                        Mail mf = new Mail();
	                        Sender sender = new Sender();
	                        int templId = OxConfiguration.getTemplateId(result.cid, result.userid);
	                        ArrayList <String> fromemail = Mail.getFromAddress(result.name, result.email, senderID, senderCID);
	                        JsonObject pwEmail = mf.getOxPasswordEmail(
	                                result.email,
	                                fromName != null ? fromName : from,
	                                fromemail.get(0),
	                                fromemail.get(1),
	                                result.newGuestPass,
	                                result.lang,
	                                templId);
	                        if (!sender.send(pwEmail, null)) {
	                            return(null);
	                        }


	                        OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();
	                        GuardKeys currentKeyForUser = ogKeyTableStorage.getCurrentKeyForUser(result.userid, result.cid);
	                        boolean markNewKeyAsCurrent = true;
	                        if(currentKeyForUser != null){
	                            markNewKeyAsCurrent = false;
	                        }

	                        // When creating a new key for a recipient, we will always create a recovery in case of lost email, etc
	                        GuardKeys keys = keyman.createMasterKeys(result.userid, result.cid, email, result.name, result.newGuestPass, result.lang, false, true,markNewKeyAsCurrent);
	                        result.pgpring = keys.pubring;
	                        result.pubkey = (keys == null) ? null : keys.getPublic();

	                    }
	                    result.pgp = true;
	                    return(result);
	                }

                }
            } catch (Exception ex) {
            	logger.error("Error getting public key for " + email, ex);
                result = null;
                return (result);
            }
        }



        if (!remote) {
            result = PgpKeyLookup.getRemoteKeys(email, from);
            if (result != null) {
                return(result);
            }
        }

        if (remote)
         {
            return (result); // If remote lookup, we don't create guest accounts, exit here.
        }

        // Finally, if all other options exhausted, create new guest account
        if (result == null) {
            result = createGuest(email, senderCID, sending, guestlang, pin);
            // Create guest account here
        }
        return (result);
    }

    /**
     * Check if the user has Guard capability enabled
     * @param id
     * @param cid
     * @return
     */
    private static boolean hasGuard (int id, int cid) {
        try {
            Capabilities cap = new Capabilities();
            return (cap.hasUserCapability("guard", id, cid));
        } catch (Exception e) {
            return (true);
        }
    }

    public static RecipKey checkMaster(String email) throws Exception {
        RecipKey result = null;
        //TODO: Use a service locator to resolve this
        RestDbOGEmailStorage restDbOGEmailStorage = new RestDbOGEmailStorage();
        OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();

        //Get the mapping for the specified email
        OGEmail emailMapping = restDbOGEmailStorage.getByEmail(email);
        if (emailMapping != null) {

            //looking for a current key
            GuardKeys currentKey = ogKeyTableStorage.getCurrentKeyForUser(emailMapping.getUserId(), emailMapping.getContextId(), email);
            if(currentKey != null) {
                result = recipKeyFromGuardKey (currentKey);
                logger.debug("Found key for " + email + " in keytable, cid = " + result.cid);
            }
        }
        return (result);
    }

    // Create a RecipKey from GuardKey
    private static RecipKey recipKeyFromGuardKey (GuardKeys key) {
        if(key != null) {
            RecipKey result = new RecipKey();
            result.pubkey = key.getPublic();
            result.userid = key.userid;
            result.cid = key.contextid;
            result.guest = key.contextid < 0;
            result.settings = key.settings;
            result.pgpring = key.pubring;
            if (result.pgpring != null) {
                result.expired = PgpKeys.checkAllExpired(result.pgpring);
                result.pgp = true;
            }
            result.lang = key.language;
            if (result.lang == null) {
                result.lang = Languages.default_code;
            }
            return (result);
        }
        return (null);
    }

    // Create a RecipKey from GuardKey
    private RecipKey createGuest(String email, int senderCID, boolean sending, String guestlang, String pin) {
        RecipKey result = null;
        if (senderCID < 0) return (null); // Guest can't create another Guest account
        if (sending) {
            result = new RecipKey();
            Keymanager manager = new Keymanager();
            result.newGuestPass = EncrLib.randomPass(Config.newPassLength);
            GuardKeys keys = manager.createMasterKeys(0, (senderCID * -1), email, email, result.newGuestPass, Languages.isavail(guestlang), false, true,true);// Guest
            result.pubkey = (keys == null) ? null : keys.getPublic();
            result.pgpring = (keys == null) ? null : keys.pubring;
            result.userid = (keys == null) ? -1 : keys.userid;
            result.cid = (senderCID * -1);
            result.guest = true;
            result.lang = Languages.isavail(guestlang);
            if (pin != null) {
                try {
					Access.updatePin(result.userid, result.cid, pin);
				} catch (Exception e) {
					logger.error("Unable to add PIN ", e);
				}
            }
        } else {
            result = new RecipKey();
            result.pubkey = null;
            result.guest = true;
        }
        return (result);
    }
}
