package com.openexchange.guard.support;

import java.util.ArrayList;
import java.util.List;
import com.openexchange.guard.database.deletedKey.DeletedKey;
import com.openexchange.guard.database.deletedKey.DeletedKeysStorage;
import com.openexchange.guard.database.deletedKey.RestDbDeletedKeysStorage;
import com.openexchange.guard.database.ogEmail.OGEmailStorage;
import com.openexchange.guard.database.ogEmail.RestDbOGEmailStorage;
import com.openexchange.guard.database.ogKeyTable.OGKeyTableStorage;
import com.openexchange.guard.database.ogKeyTable.RestDbOGKeyTableStorage;
import com.openexchange.guard.database.pgpKeys.PGPKeysStorage;
import com.openexchange.guard.database.pgpKeys.RestDbPGPKeysStorage;
import com.openexchange.guard.encr.GuardKeys;

/**
 * {@link UserDeleter} can be used for user or context based key deletion
 */
public class UserDeleter {

    private void backupKeys(List<GuardKeys> keys) throws Exception {
        //Backup keys to deleted key table / backup table
        ArrayList<DeletedKey> deletedKeys = new ArrayList<DeletedKey>();
        for(GuardKeys key : keys){
            deletedKeys.add(new DeletedKey(key));
        }
        if(deletedKeys.size() > 0 ) {
            DeletedKeysStorage deletedKeysStorage = new RestDbDeletedKeysStorage();
            deletedKeysStorage.insert(deletedKeys.toArray(new DeletedKey[deletedKeys.size()]));
        }
    }

    private void removeEmailMappings(int contextId) throws Exception {
        OGEmailStorage ogEmailStorage = new RestDbOGEmailStorage();
        ogEmailStorage.deleteAllForContext(contextId);
    }

    private void removeEmailMappings(int userId, int contextId) throws Exception {
        OGEmailStorage ogEmailStorage = new RestDbOGEmailStorage();
        ogEmailStorage.deleteAllForUser(contextId, userId);
    }

    private void removeKeys(int contextId) throws Exception{
        OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();
        ogKeyTableStorage.deleteAllForContext(contextId);
    }

    private void removeKeys(int userId, int contextId) throws Exception {
        OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();
        ogKeyTableStorage.deleteAllForUser(userId, contextId);
    }

    private void removeKeyMappings(List<GuardKeys> keys) throws Exception {
        for(GuardKeys key : keys) {
            //TODO: Use a service locator for resolving this dependencies
            PGPKeysStorage pgpKeysStorage = new RestDbPGPKeysStorage();
            pgpKeysStorage.deleteByKeyId(key.keyid);
        }
    }

    /**
     * Deletes all Guard PGP keys within a given context.
     * Keys are backed up in {@link DeletedKeysStorage} before deletion
     * @param contextId the id of the context
     * @throws Exception due an error
     */
    public void deleteWithBackup(int contextId) throws Exception {

        //TODO resolve using a DI-Container
        OGKeyTableStorage keyTableStorage = new RestDbOGKeyTableStorage();

        //Backup and remove all keys within the deleted context
        List<GuardKeys> keys = keyTableStorage.getKeysForContext(contextId);
        backupKeys(keys);

        removeKeyMappings(keys);
        removeKeys(contextId);
        //The order matters: removing the email mapping at the end, because it contains the shard id for the other operations
        removeEmailMappings(contextId);
    }

    /**
     * Deletes all Guard PGP keys related to a given user.
     * Keys are backed up in {@link DeletedKeysStorage} before deletion
     * @param userId the user's id
     * @param contextId the context id
     * @throws Exception due an error
     */
    public void deleteWithBackup(int userId, int contextId) throws Exception {

        //TODO: Using a DI-Container for resolving
        OGKeyTableStorage keyTableStorage = new RestDbOGKeyTableStorage();

        List<GuardKeys> keys = keyTableStorage.getKeysForUser(userId, contextId);
        backupKeys(keys);

        removeKeyMappings(keys);
        removeKeys(userId,contextId);
        //The order matters: removing the email mapping at the end, because it contains the shard id for the other operations
        removeEmailMappings(userId,contextId);
    }

    /**
     * Deletes all Guard PGP keys related to a given user without creating a backup.
     * For backing up keys before deletion see {@link #deleteWithBackup(int,int)}
     * @param userId
     * @param contextId
     * @throws Exception
     */
    public void delete(int userId,int contextId) throws Exception {

        //TODO resolve using a DI-Container
        OGKeyTableStorage keyTableStorage = new RestDbOGKeyTableStorage();

        List<GuardKeys> keys = keyTableStorage.getKeysForUser(userId, contextId);
        removeKeyMappings(keys);
        removeKeys(userId,contextId);
        //The order matters: removing the email mapping at the end, because it contains the shard id for the other operations
        removeEmailMappings(userId,contextId);
    }
}
