package com.openexchange.guard.server;

import java.io.IOException;
import java.net.URI;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.ox.Upgrader;
import com.openexchange.guard.support.DeletedKeyExposer;
import com.openexchange.guard.support.UserDeleter;
import com.openexchange.guard.util.Core;

/**
 * Handles incoming support requests and dispatches them to the business logic
 */
public class SupportHandler {

    /**
     * Internal method to send a text/plain response
     * @param response The response object to use
     * @param responseText The text to send as response
     * @throws IOException On an error sending the response
     */
    private void SendOk(HttpServletResponse response, String responseText) throws IOException {
        response.setContentType("text/plain");
        response.setStatus(HttpServletResponse.SC_OK);
        response.getWriter().println(responseText);
    }

    /**
     * Helper method which tries to parse a string to an integer
     * @param text the text to parse
     * @return the integer representation of text, or null if not able to convert text to an integer
     */
    private Integer TryParseInt(String text) {
        try {
           return Integer.parseInt(text);
        } catch (NumberFormatException ex) {
           return null;
        }
     }

    /**
     * Handles Expose_Key call
     * @param request The client's request
     * @param response The response which will be sent to the client
     * @throws Exception On an error during exposing a key
     */
    public void ExposeKey(HttpServletRequest request, HttpServletResponse response)
        throws Exception {

        String email = Core.getStringParameter(request,
                                               "email",
                                               true /*mandatory*/);
        int cid = Core.getIntParameter(request,
                                       "cid",
                                       true /*mandatory*/);

        //Exposing the key
        URI exposedKeyUri = new DeletedKeyExposer().exposeKey(email, cid);

        //Sending response
        SendOk(response, exposedKeyUri.toString());
    }

    /**
     * Handles reset_password call
     * @param request The client's request
     * @param response The response which will be sent to the client
     * @throws Exception On an error resetting the password
     */
    public void ResetPassword(HttpServletRequest request, HttpServletResponse response)
        throws Exception {

        //TODO/FIXME: Re-factoring
        //By now the Auth class handles all the reset functionality
        //and by passing in a request and response object it controls
        //of what to send to the client and error handling. It also reads
        //allot of parameters from the request object
        //TODO: Re-factoring:
        //Splitting the functionality: Parsing parameters in this class
        //and move basic reset functionality to com.openexchange.guard.support.PassworReseter

        //The user's email in order to get the key
        String guardEmail = Core.getStringParameter(request,
                                               "email",
                                               true /* mandatory */);

        //getting key by email
        Access dbAccess = new Access();
        GuardKeys keys = dbAccess.getKeysFromEmail(guardEmail);

        if(keys != null) {
            if (keys.contextid < 0){
                //the user is a guest user
            }
            else {
                //the user is a guard user
                new Auth().resetPass(keys.userid,keys.contextid,request,response,null,true);
            }
        } else {
            throw new IllegalArgumentException(
                String.format("Key not found for email adresse %s",guardEmail));
        }

        //FIXME: Auth does all the responses and error handling
        //        SendOk(response,"OK");
    }

    /**
     * Deletes a user (all related Keys from a user)
     * @param request request
     * @param response response
     * @throws Exception Exception due an error
     */
    public void DeleteUser(HttpServletRequest request, HttpServletResponse response) throws Exception {

        Integer userId = TryParseInt(Core.getStringParameter(request, "user_id"));
        if(userId == null) {
            throw new IllegalArgumentException("parameter user_id missing or not a valid integer");
        }

        Integer contextId = TryParseInt(Core.getStringParameter(request, "cid"));
        if(contextId == null) {
            throw new IllegalArgumentException("parameter cid missing or not a valid integer");
        }

        new UserDeleter().deleteWithBackup(userId, contextId);
        SendOk(response,"OK");
    }

    /**
     * Handles remove_pin calls
     * @param request The client's request
     * @param response The response which will be sent to the client
     */
    public void RemovePin(HttpServletRequest request, HttpServletResponse response)
        throws Exception {

        String email = Core.getStringParameter(request,
                                               "email",
                                               true /*mandatory*/);

        //Refactor
        //removing the pin
        Auth.removePin(email);
        SendOk(response,"OK");
    }

    /**
     * Handles upgrade_guest calls
     * @param request The client's request
     * @param response The response which will be sent to the client
     */
    public void UpgradeGuestAccount(HttpServletRequest request, HttpServletResponse response)
        throws Exception {

        String email = Core.getStringParameter(request,"email",true /* mandatory*/ );

        //TODO: refactor
        //upgrade the guest account to a a real account
        Upgrader.upgrade(email);

        SendOk(response,"OK");
    }
}
