/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.server;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Iterator;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xbill.DNS.DClass;
import org.xbill.DNS.ExtendedResolver;
import org.xbill.DNS.Message;
import org.xbill.DNS.Name;
import org.xbill.DNS.Record;
import org.xbill.DNS.Resolver;
import org.xbill.DNS.ReverseMap;
import org.xbill.DNS.Section;
import org.xbill.DNS.Type;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.GetPubKey;
import com.openexchange.guard.database.RecipKey;
import com.openexchange.guard.exceptions.GuardMissingParameter;
import com.openexchange.guard.logging.LogAction;
import com.openexchange.guard.ox.checkWhitelist;
import com.openexchange.guard.util.Core;

public class PGPServlet extends HttpServlet  {
	private static Logger logger = LoggerFactory.getLogger(PGPServlet.class);

	@Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {

	    LogAction.getPID();
		logger.debug("PGP Request " + request.getQueryString());
		try {
			String operation = Core.getStringParameter(request, "op", true);
			String options = Core.getStringParameter(request, "options");
			String search = Core.getStringParameter(request, "search", true);

			if (search.contains("*")) {  // Wildcard isn't being supported
			    logger.debug("Attempt to use wildcard in remote lookup.  Rejected");
			    LogAction.setFail();
			    response.setStatus(HttpServletResponse.SC_NOT_IMPLEMENTED);
			    return;
			}

			switch (operation) {
			case "index":
				if (search.contains("@")) {
					Access acc = new Access ();
					String ip = Core.getIP(request);
					LogAction.setURL(ip, request.getQueryString(), search);
					ArrayList <PGPPublicKeyRing> keyrings = acc.getPublicKeyRing(normalizeEmail(search)); // zero cid for local search only
					if (keyrings.size() == 0) {
						logger.debug("No Keys found, check whitelist against " + ip);
						if (checkWhitelist.isWhiteListed(ip)) {  // If calling server in whitelist, check if OX member to create
							logger.debug("IP in whitelist, checking against OX backend");
							GetPubKey getter = new GetPubKey();
							// Check and see if we can get public key from the backend if member
							RecipKey key = getter.getPubKey(normalizeEmail(search), 0, 0, Config.passwordFromAddress, true, true, null, Config.passwordFromName, null, null);
							if (key != null) {  // If yes, then get keyring
								keyrings = acc.getPublicKeyRing(normalizeEmail(search));
								logger.debug("Sending list of keys, length = " + keyrings.size());
								LogAction.setSuccess();
								Core.sendOK(response, getMRIndex(keyrings));
								return;
							}
						}
						response.setStatus(HttpServletResponse.SC_NOT_FOUND);
						return;
					}
					logger.debug("Sending list of keys, length = " + keyrings.size());
					LogAction.setSuccess();
					Core.sendOK(response, getMRIndex(keyrings));
					return;
				}
			break;
			case "vindex":
				response.setStatus(HttpServletResponse.SC_NOT_IMPLEMENTED);
				return;

			case "get":
				if (!search.contains("@")) {
					Access acc = new Access ();
					String req = normalizeSearch(search);
                    PGPPublicKeyRing keyring = null;
					if (req != null) {
					    if (req.length() > 8) {
					        Long req_id = Core.getLongPGP(req);
					        keyring = acc.getPublicKeyRingById(req_id, 0, 0);
					    } else {
					        keyring = acc.getPublicKeyRingById(normalizeSearch(search), 0, 0);
					    }
					}
					if (keyring != null) {
					    LogAction.setSuccess();
						logger.debug("Found key, sending");
						response.setContentType("application/pgp-keys");
				        response.setStatus(HttpServletResponse.SC_OK);
				        try {
				            response.getWriter().println(exportPgpPublicKey(keyring));
				        } catch (IOException e) {
				            LogAction.setFail();
				        	logger.error("Error sending PGP Key", e);
				        }
						return;
					} else {
						logger.debug("Key not found");
						response.setStatus(HttpServletResponse.SC_NOT_FOUND);
						return;
					}
				}
			}
		} catch (GuardMissingParameter ex) {
		    LogAction.setFail();
		    logger.info("Fail due to missing parameter");
			Core.sendFail(response, "Missing Parameter " + ex);
			return;
		} catch (Exception e) {
		    Core.sendFail(response, e.getLocalizedMessage());
            return;
        }
	}

	private String normalizeEmail (String email) throws Exception {
	    if (email.contains("%")) {
            throw new Exception ("Bad Email character");
        }
	    return (email);
	}

	/**
	 * Normalize search parameter.  If more than the 16 character ID (Fingerprint), then shorten
	 * @param search
	 * @return
	 */
	private String normalizeSearch (String search) {
	    search = search.replace(" ", "");  // remove padding from a fingerprint
	    search = search.replace("0x", "").toLowerCase();
	    if (search.length() > 16) {  // If more than 16, we'll just look at the last 16
	        search = search.substring(search.length()-16);
	    }
	    return(search);
	}

	private String getMRIndex (ArrayList <PGPPublicKeyRing> keyrings) throws UnsupportedEncodingException {
		StringBuilder resp = new StringBuilder();
		resp.append("info:1:" + keyrings.size() + "\r\n");
		for (int i = 0; i < keyrings.size(); i++) {
			Iterator keys = keyrings.get(i).getPublicKeys();
			while (keys.hasNext()) {
		    	PGPPublicKey key = (PGPPublicKey) keys.next();
		    	String id = Core.PGPKeyLongId(key.getKeyID());
		    	String algo = Integer.toString(key.getAlgorithm());
		    	String keylen = Integer.toString(key.getBitStrength());
		    	long cr = key.getCreationTime().getTime()/1000;
		    	String creation = Long.toString(cr);
		    	String exp = key.getValidDays() == 0 ? "" : Long.toString((key.getValidDays() * 24 * 60 * 60) + cr);
		    	String pub = "pub:" + id + ":" + algo + ":" + keylen + ":" + creation + ":" + exp + ":";
		    	resp.append (pub + "\r\n");
		    	// create primary key line
		    	Iterator it = key.getUserIDs();
		    	while (it.hasNext()) {
		    		resp.append("uid:\"" + (String) it.next() + "\":" + creation + ":" + exp + ":\r\n");
		    		// get userids
		    	}
			}
		}
		return (resp.toString());
	}

    public String exportPgpPublicKey(PGPPublicKeyRing pubring) throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        ArmoredOutputStream arm = new ArmoredOutputStream(out);
        // PGPPublicKeyRing pbkeyring = keyring.get
        // arm.write(pbkeyring.getEncoded());
        Iterator<PGPPublicKey> keys = pubring.getPublicKeys();
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        while (keys.hasNext()) {
            PGPPublicKey k = keys.next();
            k.encode(bout);
        }
        bout.close();
        arm.write(bout.toByteArray());
        arm.close();
        out.close();
        return (new String(out.toByteArray()));
    }


    public static String rDNS (String IP) throws IOException {
        Resolver res = new ExtendedResolver();
        Name name = ReverseMap.fromAddress(IP);
        Record rec = Record.newRecord(name, Type.PTR, DClass.IN);
        Message query = Message.newQuery(rec);
        Message response = res.send(query);
        Record [] answers = response.getSectionArray(Section.ANSWER);
        if (answers.length == 0) {
            return (null);
        }
        return (answers[0].rdataToString());

    }

}
