/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.server;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.Enumeration;
import java.util.Properties;
import java.util.TimeZone;
import javax.mail.Address;
import javax.mail.Header;
import javax.mail.Message.RecipientType;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Session;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;
import com.openexchange.guard.database.Keymanager;
import com.openexchange.guard.database.Sharding;
import com.openexchange.guard.database.ogEmail.OGEmail;
import com.openexchange.guard.database.ogEmail.RestDbOGEmailStorage;
import com.openexchange.guard.database.ogKeyTable.OGKeyTableStorage;
import com.openexchange.guard.database.ogKeyTable.RestDbOGKeyTableStorage;
import com.openexchange.guard.encr.Crypto;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.EncryptedObj;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.exceptions.BadPasswordException;
import com.openexchange.guard.exceptions.BadRSAException;
import com.openexchange.guard.exceptions.GuardMissingParameter;
import com.openexchange.guard.mailcreator.Mail;
import com.openexchange.guard.pgp.PGPResult;
import com.openexchange.guard.pgp.PGPUtils;
import com.openexchange.guard.pgp.PgpHandler;
import com.openexchange.guard.translate.GetText;
import com.openexchange.guard.util.Core;
import com.openexchange.guard.validator.Normalizer;


public class Guest {

	private static Logger logger = LoggerFactory.getLogger(Guest.class);


	/**
	 * Changing Guest password
	 * @param request
	 * @param response
	 * @param cookie
	 */
    public void ChangePass(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) {
        Auth auth = new Auth();
        cookie.JSESSIONID = cookie.OxReaderID; // Clear the ox cookie for just reader.
        try {
            if (auth.changepass(request, response, cookie) != null) {
                response.setStatus(HttpServletResponse.SC_OK);
            } else {
                com.openexchange.guard.util.Core.sendFail(response, "Bad password");
            }
        } catch (Exception ex) {
            logger.error("Error changing password", ex);
        }
    }

    /**
     * For initial decoding of PGP Guest email
     * @param request
     * @param response
     * @param cookie
     * @throws GuardMissingParameter
     * @throws BadRSAException
     */
    public void GetEmail(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws GuardMissingParameter, BadRSAException {
    	decodeGuestPgp (request, response, cookie, "mail");
    }

    /**
     * Decodes Guest PGP email.  Opens attachment if action open, returns email json if action mail
     * @param request
     * @param response
     * @param cookie
     * @param action
     * @throws GuardMissingParameter
     * @throws BadRSAException
     */
    private void decodeGuestPgp(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie, String action) throws GuardMissingParameter, BadRSAException {
    	JsonObject jdata = Core.getJSON(request);
    	String itemid = "";
    	if (jdata != null) {
    		itemid = Core.getStringFromJson(jdata, "item", true);
    	} else {
    		itemid = Core.getStringParameter(request, "email", true);
    	}
    	int userid = Core.getIntParameter(request, "userid");
    	int cid = Core.getIntParameter(request, "cid");
    	if (itemid.startsWith("pgp")) {
    		String auth = Core.getStringParameter(request, "auth", true);
            cookie.JSESSIONID = cookie.OxReaderID; // In case person uses both reader and OX main, we need to clear out the previous cookie
            UserData data = new UserData(auth, cookie);
            if (Core.badAuth(data, userid, cid)) {
                response.setStatus(HttpServletResponse.SC_BAD_REQUEST);
                return;
            }
            try {
                InputStream in = com.openexchange.guard.storage.Storage.readObj(data.userid, data.cid, itemid);
                if (in != null) {
                	 PGPResult result = null;
                     try {
                    	 result = PGPUtils.decryptFile(in, data.userid, data.cid, data.encr_password);

                     } catch (BadPasswordException ex) {
                    	 in.close();
                     	Core.sendFail(response, "Bad password");
                     	return;
                     }
                     in.close();
                     if (result.error != null) {
                     	Core.sendFail(response, result.error);
                     } else {
                    	 switch (action) {
                     		case "mail":
                     			Core.sendOK(response, createReturnJson(result, data.encr_password, cookie));
                     		break;
                     		case "open":
                     			String filename = Core.getStringParameter(request, "attname", true);
                     			boolean download = (Core.getStringParameter(request, "download") == null ? false : (Core.getStringParameter(request, "download").toLowerCase().equals("true")));
                     			PgpHandler.getAttach (response, result, filename, download);
                     		break;
                     	}
                     }
                } else {
                	Core.sendFail(response, "not found");
                	logger.info("Guest item not found: " + itemid);
                }
            } catch (Exception ex) {
            	logger.error("Problemd decoding Guest PGP ", ex);

            }
    	} else {  // Proprietary Guard
    		GetOldEmail (jdata, request, response, cookie);
    	}
    }

    /**
     * Create resturn Json for display in Guest browser
     * @param result
     * @param json
     * @param password
     * @param cookie
     * @return
     * @throws MessagingException
     * @throws IOException
     */
    public static String createReturnJson(PGPResult result, String password, com.openexchange.guard.server.OxCookie cookie) throws MessagingException, IOException {
        Session s = Session.getDefaultInstance(new Properties());
        JsonObject data = new JsonObject();
        InputStream in = new ByteArrayInputStream(result.decoded);
        MimeMessage message = new MimeMessage(s, in);
        Multipart mp = (Multipart) message.getContent();
        JsonArray attachments = new JsonArray();
        PgpHandler.ParseMultipart (mp, attachments);
        JsonObject Results = new JsonObject();
        Results.add("signature", new JsonPrimitive(result.signature));
        Results.add("verified", new JsonPrimitive(result.verified));
        if (password != null) {
        	Access acc = new Access();
            String token = acc.getToken(cookie.JSESSIONID);
        	EncrLib encr = new EncrLib();
        	Results.addProperty("epassword", encr.encryptAES(password, token));
        }
        if (result.error != null) {
            Results.add("error", new JsonPrimitive(result.error));
        }
        data.add("attachments", attachments);
        data.add("results", Results);
        Address[] fromarray = message.getFrom();
        JsonArray fromJson = new JsonArray();
        if (fromarray != null) {
            for (int i = 0; i < fromarray.length; i++) {
                JsonArray recip = new JsonArray();
                InternetAddress fromAddr = (InternetAddress) fromarray[i];
                recip.add(new JsonPrimitive (fromAddr.getPersonal() == null ? "" : fromAddr.getPersonal()));
                recip.add(new JsonPrimitive(fromAddr.getAddress() == null ? "" : fromAddr.getAddress()));
                fromJson.add(recip);
            }
        }
        data.add("from", fromJson);
        Address[] toarray = message.getRecipients(RecipientType.TO);
        JsonArray toJson = new JsonArray();
        if (toarray != null) {
            for (int i = 0; i < toarray.length; i++) {
                JsonArray recip = new JsonArray();
                InternetAddress to = (InternetAddress) toarray[i];
                recip.add(new JsonPrimitive (to.getPersonal() == null ? "" : to.getPersonal()));
                recip.add(new JsonPrimitive(to.getAddress() == null ? "" : to.getAddress()));
                toJson.add(recip);
            }
        }
        data.add("to", toJson);
        Address[] ccarray = message.getRecipients(RecipientType.CC);
        JsonArray ccJson = new JsonArray();
        if (ccarray != null) {
            for (int i = 0; i < ccarray.length; i++) {
                JsonArray recip = new JsonArray();
                InternetAddress cc = (InternetAddress) ccarray[i];
                recip.add(new JsonPrimitive (cc.getPersonal() == null ? "" : cc.getPersonal()));
                recip.add(new JsonPrimitive(cc.getAddress() == null ? "" : cc.getAddress()));
                ccJson.add(recip);
            }
        }
        data.add("cc", ccJson);
        Enumeration<Header> headers = message.getAllHeaders();
        JsonObject headersJson = new JsonObject();
        while (headers.hasMoreElements()) {
            Header h = headers.nextElement();
            headersJson.addProperty(h.getName(), h.getValue());
        }
        data.add("headers", headersJson);
        data.addProperty("subject", message.getSubject());
        final String ISO_FORMAT = "yyyy-MM-dd'T'HH:mm:ss.SSS'+00:00'";
        final SimpleDateFormat sdf = new SimpleDateFormat(ISO_FORMAT);
        final TimeZone utc = TimeZone.getTimeZone("UTC");
        sdf.setTimeZone(utc);
        data.addProperty("sent", sdf.format(message.getSentDate()));
        in.close();
        return (data.toString());
    }



    /**
     * Function to return encrypted attachment (non email)
     *
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void getEncrAttach(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
        String emailid = Core.getStringParameter(request, "email", true);
        if (emailid.startsWith("pgp")) {
        	cookie.JSESSIONID = cookie.OxReaderID;
        	decodeGuestPgp(request, response, cookie, "open");
        	return;
        } else {
        	getEncrAttachOld(request, response, cookie);
        }
    }



    /**
     * Get password reset question
     * @param request
     * @param response
     * @param cookie
     * @throws IOException
     * @throws GuardMissingParameter
     */
    public void getQuestion(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception, GuardMissingParameter {
        String email = Core.getStringParameter(request, "email", true);
        RestDbOGEmailStorage ogEmailStorage = new RestDbOGEmailStorage();
        OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();

        //Getting the mail mapping
        String question = "nf";
        OGEmail emailKeyMapping = ogEmailStorage.getByEmail(email);
        if(emailKeyMapping != null) {
            //Getting the guest's key
            GuardKeys key = ogKeyTableStorage.getKeyForEmailAndContext(email, emailKeyMapping.getContextId());
            if(key != null) {
                question = key.question;
            }
        }
        else {
            logger.error(String.format("No email mapping found for email %s",email));
        }
        Core.sendOK(response, question);
    }

    /**
     * Perform password reset after question answered
     * @param request
     * @param response
     * @param cookie
     * @throws IOException
     * @throws GuardMissingParameter
     */
    public void resetPassword(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws IOException, GuardMissingParameter {
        String email = Core.getStringParameter(request, "email", true);
        String answer = Core.getStringParameter(request, "answer", true);
        String lang = Core.getStringParameter(request, "language");
        int templId = Core.getIntParameter(request, "templid");
        String question = "";

        int cid = 0;
        int userid = 0;
        try {
            if (CheckBad.isBad("reset-" + email, Config.bad_password_count)) {
                Core.sendOK(response, "lockout");
                return;
            }

            String command = "SELECT id, cid, answer, question FROM og_KeyTable WHERE email = ?;";
            DbCommand com = new DbCommand(command);
            com.addVariables(email);
            DbQuery db = new DbQuery();
            db.readFromShard(com, Sharding.getShard(email));
            String prevanswer = null;
            if (db.next()) {
                prevanswer = db.rs.getString("answer");
                userid = db.rs.getInt("id");
                cid = db.rs.getInt("cid");
                question = db.rs.getString("question");
            }
            if (question != null) {  // PIN is not used for answer
            	if (question.equals("PIN")) {
                    prevanswer = null;
                }
            }
            db.close();
            if (prevanswer != null) {
	            EncrLib encr = new EncrLib();
	            if (question.equals("")) {
                    question = "default";
                }
	            prevanswer = encr.decryptAES(prevanswer, Config.rpass, question, Config.AESKeyLen);
	            if (prevanswer == null) {
	                Core.sendFail(response, "Error decoding answer");
	                return;
	            }
	            prevanswer = prevanswer.toLowerCase().replace(" ", "");
	            answer = answer.toLowerCase().replace(" ", "");
	            if (!prevanswer.equals(answer)) {
	                Core.sendOK(response, "No Match");
	                CheckBad.addBad("reset-" + email);
	                return;
	            }
            }
            Keymanager keyman = new Keymanager();
            String newpass = keyman.resetPassword(email);
            if (newpass == null) {
                Core.sendFail(response, "Failed to change");
                return;
            }
            if (newpass.equals("NR")) {  // No recovery available
                Core.sendFail(response, "Failed to change");
                return;
            }
            Sender sender = new Sender();
            com.openexchange.guard.mailcreator.Mail mf = new com.openexchange.guard.mailcreator.Mail();
            if (!sender.send(mf.getResetEmail(email, Mail.getFromAddress(email, email, userid, cid), newpass, lang, templId), null)) {
                Core.sendOK(response, "FailNotify");
            } else {
                Core.sendOK(response, "OK");
            }

        } catch (Exception ex) {
            logger.error("Error resetting password for email account " + email, ex);
            Core.sendFail(response, "Failed to change");
        }
    }

    //////////////////////////////////////////////////////////////////////////////////////////////////
    ///////////////  DEPRECIATED.  FOR OLD GUARD SUPPORT


    /**
     * Depreciated Guard proprietary format
     * @param jdata
     * @param request
     * @param response
     * @param cookie
     * @throws GuardMissingParameter
     */
    public void GetOldEmail(JsonObject jdata, HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws GuardMissingParameter {


        String auth = Core.getStringParameter(request, "auth", true);
        String itemid = Core.getStringFromJson(jdata, "item", true);
        String extraPass = Core.getStringFromJson(jdata, "extrapass");

        int cid = Core.getIntFromJson(jdata, "cid", true);
        cookie.JSESSIONID = cookie.OxReaderID; // In case person uses both reader and OX main, we need to clear out the previous cookie
        UserData data = new UserData(auth, cookie);

        if (data.encr_password == null) {
            response.setStatus(HttpServletResponse.SC_BAD_REQUEST);
            return;
        }
        try {
            EncryptedObj emaildata = com.openexchange.guard.storage.Storage.readEncrObjOld(data.userid, data.cid, itemid);
            if (emaildata.ItemID.equals("not found")) {
                com.openexchange.guard.util.Core.sendFail(response, "not found");
                return;
            }
            Crypto crypto = new Crypto();
            String email = crypto.decodeEncrObj(data.userid, cid, emaildata, data.encr_password);

            if (extraPass != null) { // If we provided extrapass, we need to encrypt to return for attachments
                if (!extraPass.equals("")) {
                    if (extraPass.length() > 2) {
                        String extrap = "";
                        String epassword = "";
                        EncrLib encr = new EncrLib();
                        Access acc = new Access();
                        String token = acc.getToken(cookie.JSESSIONID);
                        email = crypto.decodeExtraPass(email, extraPass, emaildata.ItemID, emaildata.keyLength);
                        if (email.equals("")) {
                            email = "badextra";
                            com.openexchange.guard.util.Core.sendFail(response, email);
                            return;
                        }
                        extrap = encr.encryptAES(extraPass, token);
                        email = "{" + "\"extrapass\":\"" + extrap + "\", \"epass\":\"" + epassword + "\"," + email.substring(1); // add
                                                                                                                                 // extrapass
                                                                                                                                 // info
                                                                                                                                 // to
                                                                                                                                 // email
                                                                                                                                 // json

                    }
                }

            }
            email = Normalizer.normalize(email);
            logger.debug("Guest email decoded OK");
            response.addHeader("Access-Control-Allow-Origin", "*");
            response.setContentType("application/json; charset=UTF-8");
            response.setStatus(HttpServletResponse.SC_OK);
            response.getWriter().write(email);
        } catch (Exception ex) {
            logger.error("Error decoding guest email for itemID " + itemid, ex);
            response.setStatus(HttpServletResponse.SC_BAD_REQUEST);
        }

    }


    public void getEncrAttachOld(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
    	String emailid = Core.getStringParameter(request, "email", true);
        String filename = Core.getStringParameter(request, "attname", true);
        int cid = Core.getIntParameter(request, "cid", true);
        String auth = Core.getStringParameter(request, "auth", true);
        String extrapass = Core.getStringParameter(request, "extrapass");
        String encrextrapass = Core.getStringParameter(request, "encrextrapass");
        String lang = Core.getStringParameter(request, "language");
        cookie.JSESSIONID = cookie.OxReaderID; // In case person uses both reader and OX main, we need to clear out the previous cookie
        UserData data;
        data = new UserData(auth, cookie);
        if (data.userid == -1) {
            response.getWriter().write(GetText.gettext("Unable to decode attachment, bad authentication", lang));
            response.setStatus(HttpServletResponse.SC_BAD_REQUEST);
            return;
        }
        Crypto crypto = new Crypto();
        try {
            EncryptedObj attdata = com.openexchange.guard.storage.Storage.readEncrObjOld(data.userid, data.cid, emailid + "-" + filename);
            String attach = crypto.decodeEncrObj(data.userid, cid, attdata, data.encr_password);
            if (attach != null) {// Data was stored base64, so we need to decode when returned
                if (extrapass == null) {// If plaintext extrapass is null, then look at encryptedpass
                    if (encrextrapass != null) {
                        if (!encrextrapass.equals("")) {
                            EncrLib encr = new EncrLib();
                            Access acc = new Access();
                            String token = acc.getToken(cookie.JSESSIONID);
                            extrapass = encr.decryptAES(encrextrapass, token);
                            attach = crypto.decodeExtraPass(attach, extrapass, attdata.ItemID, attdata.keyLength);
                        }
                    }
                } else {
                    if (!extrapass.equals("")) {
                        EncrLib encr = new EncrLib();
                        attach = encr.decryptAES(attach, extrapass, attdata.ItemID, attdata.keyLength);
                    }
                }
                if (attach == "exp") { // If our email has expired, then just send that
                    com.openexchange.guard.util.Core.sendOK(response, "Attachment Expired");
                    return;
                }
                // util.Core.sendOK(response, attdata.content_type, Base64.decode(attach));
                if (attach.equals("")) {
                    com.openexchange.guard.util.Core.sendFail(response, GetText.gettext("Unable to decode", lang));
                } else {
                    sendToSaveAttach(
                        response,
                        attdata.content_type,
                        Base64.decode(attach),
                        filename.replace(".grd2", "").replace(".grd", ""));
                }
                return;
            }
        } catch (InvalidKeySpecException | NoSuchAlgorithmException | SQLException | PGPException e) {
        	logger.error("Error returning encrypted attachments for email " + emailid, e);
        }
        com.openexchange.guard.util.Core.sendFail(response, GetText.gettext("Unable to decode", lang));

    }


    /**
     * Send attachment contents to browser with flags to save to disk
     *
     * @param response
     * @param type
     * @param answer
     * @param filename
     */
    private void sendToSaveAttach(HttpServletResponse response, String type, byte[] answer, String filename) {
        response.setContentType(type);
        response.setHeader("Content-Disposition", "attachment; filename=\"" + filename + "\"");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        try {
            OutputStream out = response.getOutputStream();
            out.write(answer);
            out.close();
        } catch (IOException e) {
        	logger.error("Error writing output stream for file " + filename, e);
        }
    }

}
