/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgp;

import java.security.SignatureException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import javax.mail.internet.InternetAddress;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.DbQuery;
import com.openexchange.guard.database.ogPGPKeys.OGPGPKey;
import com.openexchange.guard.database.ogPGPKeys.OGPGPKeysStorage;
import com.openexchange.guard.database.ogPGPKeys.RestDbOGPGPKeysStorage;
import com.openexchange.guard.database.pgpKeys.PGPKeys;
import com.openexchange.guard.database.pgpKeys.PGPKeysStorage;
import com.openexchange.guard.database.pgpKeys.RestDbPGPKeysStorage;
import com.openexchange.guard.encr.Crypto;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.logging.LogAction;
import com.openexchange.guard.mailcreator.Attachment;
import com.openexchange.guard.ox.Api;
import com.openexchange.guard.pgp.parser.AsciiArmoredKeyParser;
import com.openexchange.guard.server.OxCookie;
import com.openexchange.guard.util.Core;
import com.openexchange.guard.util.ListUtil;
import com.openexchange.guard.validator.EmailValidator;

public class PGPPublicHandler {

    public static final int MAX_KEY_FILE_UPLOAD_SIZE =  1000000; /*1 MB*/

    private static Logger logger = LoggerFactory.getLogger(PGPPublicHandler.class);

	public void incomingPGPForm(HttpServletRequest request, HttpServletResponse response, OxCookie cookie) throws Exception {
        FileItemFactory fif = new DiskFileItemFactory();
        ServletFileUpload sfu = new ServletFileUpload(fif);
        int userid = Core.getIntParameter(request, "userid", true);
        int cid = Core.getIntParameter(request, "cid", true);
        LogAction.setUser(userid, cid);
        List<FileItem> files = sfu.parseRequest(request);
        String emailRestriction = Core.getStringParameter(request, "emailaddress");  // Optional uid restriction
        //Check authentication
        Api ap = new Api(cookie, request);
        if (!ap.verifyLogin()) {
            LogAction.setFail();
            logger.info("User not logged into UI");
            Core.sendFail(response, "Must be logged in to UI");
            return;
        }

        //Iterate through attachments
        Iterator<FileItem> iterator = files.iterator();
        while (iterator.hasNext()) {
            FileItem fi = iterator.next();

            //Check the file size
            if (fi.getSize() > MAX_KEY_FILE_UPLOAD_SIZE) {
                Core.sendFail(response, "Key size exceeded");
                return;
            }

            //Parsing the input stream for public keys rings and using the factory to create them
            List<PGPPublicKeyRing> publicKeys =
                PGPPublicKeyRingFactory.create(new AsciiArmoredKeyParser().parse(fi.getInputStream()));

            if(publicKeys.size() > 0) {
                //If the client provided an email filter we only consider parsed public keys which userids are in the filter
                String [] emails = emailRestriction == null ? null : emailRestriction.split(",");
                if(emails != null) {
                    publicKeys = getKeysForEmail(publicKeys,Arrays.asList(emails));
                    if(publicKeys.size() == 0) {
                        logger.debug("User email address not found in the uploaded public keys" );
                        Core.sendFail(response, "Bad UID");
                        return;
                    }
                }

                // Start response array
                StringBuilder responseSB = new StringBuilder();
                responseSB.append("[");

                //Saving all the parsed key-rings for the user
                int count = 0;
                for(PGPPublicKeyRing publicKey : publicKeys) {
                    //Saving each key
                    if(saveKeyData(publicKey, cid, userid)) {
                        //Convert the key to JSON and append to the response
                        JsonArray keydata = PgpHandler.getKeyInfo(publicKey);
                        responseSB.append(keydata.toString());
                        //Comma separated, if more than one key
                        if (count != publicKeys.size() - 1){
                            responseSB.append(",");
                        }
                        count++;
                    } else {
                        LogAction.setFail();
                        logger.warn("Unable to save uploaded public key data");
                        Core.sendFail(response, "Problem importing key, possible bad data");
                        return;
                    }
                }
                responseSB.append("]");  // Close the array
                Core.sendOK(response, responseSB.toString());
                LogAction.setSuccess();
            }
            else {
                LogAction.setFail();
                logger.info("Bad data");
                Core.sendFail(response, "Bad data");
                return;
            }
        }
	}

	/**
	 * Filters a list of public keys for uids
	 * @param publicKeys the list of keys to filter
	 * @param emails the filter emails
	 * @return those keys from publicKeys which uids are in emails
	 */
	private List<PGPPublicKeyRing> getKeysForEmail(List<PGPPublicKeyRing> publicKeys, List<String> emails) {
	    ArrayList<PGPPublicKeyRing> ret = new ArrayList<PGPPublicKeyRing>();

	    for(PGPPublicKeyRing key : publicKeys) {
	        Iterator<String> uidIterator = key.getPublicKey().getUserIDs();

	        while(uidIterator.hasNext()) {
	            String uid = uidIterator.next().toLowerCase();
	            for (String email : emails) {
                    if (email.contains("@")) { // If email address, check if in userid
                        if (uid.contains(email.toLowerCase())) {
                            ret.add(key);
                        }
                    }
                }
	        }
	    }
	    return ret;
    }

    private boolean saveKeyData (PGPPublicKeyRing ring, int cid, int userid) throws Exception {
		if (!ring.getPublicKeys().hasNext())
         {
            return (false);  // If no public keys, return false
        }
		JsonArray keydata = PgpHandler.getKeyInfo(ring);
		JsonArray ids = new JsonArray();
		List<String> keyhexids = new ArrayList<String>();
		for (int i = 0; i < keydata.size(); i++) {
			JsonObject key = keydata.get(i).getAsJsonObject();
			JsonArray keyids = key.get("ids").getAsJsonArray();
			keyhexids.add(key.get("Key").getAsString());
			if (keyids.size() > 0) {
                ids = keyids;
            }
		}
		if(keyhexids.size() == 0) {
		    return false;
		}
		boolean added = false;
		for (int j = 0; j < ids.size(); j++) {
			InternetAddress addr = Core.parseEmailAdress(ids.get(j).getAsString());
			try {
				addr.validate();
			} catch (Exception e) {
				continue;
			}
			String email = addr.getAddress();

			//TODO: Use a DI-Container to resolve this
			OGPGPKeysStorage ogpgpKeysStorage = new RestDbOGPGPKeysStorage();
			//Adding the new uploaded key to the storage
			String asciiData = PGPPublicKeyRingExporter.export(ring);
			ogpgpKeysStorage.InsertOrUpdate(userid, cid, email, keyhexids, asciiData);

			added = true;
		}
		return (added);
	}

	public void saveKeyAttachment(HttpServletRequest request, HttpServletResponse response, OxCookie ck) throws Exception {
		int userid = Core.getIntParameter(request, "userid", true);
		int cid = Core.getIntParameter(request, "cid", true);
		LogAction.setUser(userid, cid);
		String folder = Core.getStringParameter(request, "folder", true);
        String  emailid = Core.getStringParameter(request, "emailid", true);
        String attach = Core.getStringParameter(request, "attach", true);
        Api ap = new Api(ck, request);
        if (!ap.verifyLogin()) {
            LogAction.setFail();
            logger.info("User not logged into UI");
        	Core.sendFail(response, "Must be logged in to UI");
        	return;
        }
        PGPPublicKeyRing ring = null;
        String epass = Core.getStringParameter(request, "epass");
        String data = null;
        if (epass == null) {  // If not encrypted, retrieve plaintext
            Attachment att = ap.getPlainAttachment(emailid, attach, folder);
            data = new String (att.content);

        } else {
            // Handle encrypted public key
            String filename = Core.getStringParameter(request, "filename", true);
            boolean inline = Core.getBooleanParameter(request, "inline");
            String password = Crypto.decodeEPass(epass, ck);
            if (password == null) {
                Core.sendFail(response, "bad password");
                return;
            }
            PgpHandler pgh = new PgpHandler();
            byte[] bytedata = null;
            if (inline) {  // pgp inline attachment
                bytedata = pgh.decodeInlineAttach(emailid, folder, attach, filename, userid, cid, password, request, ck);
            } else {  // pgp mime attachment
                bytedata = pgh.getAttach(request, ck, emailid, folder, userid, cid, filename, password);

            }
            if (bytedata == null || bytedata.length < 2) {
                logger.warn("Unable to decode inline pgp public key attachment");
                Core.sendFail(response, "unable to decode");
                return;
            }
            data = new String(bytedata, "UTF-8");
        }
        ring = PGPPublicKeyRingFactory.create(data);
        if (ring == null) {
            LogAction.setFail();
            logger.info("Problem reading key data");
        	Core.sendFail(response, "Problem reading key data");
        	return;
        }
		if (saveKeyData (ring, cid, userid)) {
			LogAction.setSuccess();
			Core.sendOK(response, PgpHandler.getKeyInfo(ring).toString());
		} else {
		    LogAction.setFail();
		    logger.info("Problem saving key data");
			Core.sendFail(response, "Problem saving");
		}

	}


	public void getPubKeyDetail (HttpServletRequest request, HttpServletResponse response) throws Exception {
		int userid = Core.getIntParameter(request, "userid", true);
		int cid = Core.getIntParameter(request, "cid", true);
		LogAction.setUser(userid, cid);
		String ids = Core.getStringParameter(request, "ids", true).trim();
        String[] idList = ids.trim().split("\\s+");
		boolean guard = Core.getBooleanParameter(request, "guard");
		if (guard) {  // If we are to check the Guard tables, do it here
		    JsonObject guardJson = getGuardPubKeyDetail(ids);
		    if (guardJson != null) {
		        guardJson.addProperty("guard", true);
		        Core.sendOK(response, guardJson.toString());
                return;
		    }
		}
		// Otherwise check uploaded
		//TODO: Use a DI-Container to resolve this
		RestDbOGPGPKeysStorage ogpgpKeysStorage = new RestDbOGPGPKeysStorage();
		List<OGPGPKey> ogPGPKeys = ogpgpKeysStorage.getForUserByIds(userid, cid, Arrays.asList(idList));
		OGPGPKey ogPGPKey = null;
		//Getting the first key
		if(ogPGPKeys.size() > 0) {
		    ogPGPKey = ogPGPKeys.get(0);
		}

        if (ogPGPKey != null) {
            PGPPublicKeyRing keyring = PGPPublicKeyRingFactory.create(ogPGPKey.getPublicPGPAscData());
            JsonArray keydata = PgpHandler.getKeyInfo(keyring);
            JsonObject retJson = new JsonObject();
            retJson.add("data", keydata);
            retJson.add("share", new JsonPrimitive(ogPGPKey.getShareLevel()));
            retJson.add("inline", new JsonPrimitive (ogPGPKey.isInline()));
            if (keydata.size() > 0) {
                LogAction.setSuccess();
                Core.sendOK(response, retJson.toString());
                return;
            }
        }

		LogAction.setFail();
		Core.sendFail(response, "Fail");
	}

	private JsonObject getGuardPubKeyDetail (String id) throws SignatureException, PGPException {
	    Access acc = new Access();
	    String[] ids = id.split(" ");
	    for (String single_id : ids) {
	        PGPPublicKeyRing keyring = acc.getPublicKeyRingById(single_id, 0, 0);
	        if (keyring != null) {
	            JsonArray keydata = PgpHandler.getKeyInfo(keyring);
	            JsonObject retJson = new JsonObject();
	            retJson.add("data", keydata);
	            if (keydata.size() > 0) {
	                return (retJson);
	            }
	        }
	    }
	    return(null);
	}

	public void getPubKeyList (HttpServletRequest request, HttpServletResponse response) throws Exception {
		int userid = Core.getIntParameter(request, "userid", true);
		int cid = Core.getIntParameter(request, "cid", true);
		LogAction.setUser(userid, cid);
		StringBuilder searchlist = new StringBuilder();
		String emails = Core.getStringParameter(request, "emails");
		ArrayList<String> emailArray = new ArrayList<String>();
		if (emails != null) {
		    String [] emailList = emails.split(",");
		    for (String email : emailList) {
		        if (!email.trim().equals("")) {
		            if (EmailValidator.validate(email)) {
		                if (searchlist.length() == 0) {
		                    searchlist.append("'" + Core.dbformat(email) + "'");
		                } else {
                            searchlist.append(", '" + Core.dbformat(email) + "'");
                        }
		                emailArray.add(email);
		            }
		        }
		    }
		}
		LogAction.setSuccess();
		Core.sendOK(response, getPubKeyList(userid, cid, searchlist.toString(), emailArray).toString());
	}

	public JsonArray getPubKeyList (int userid, int cid, String searchlist, ArrayList<String> emailArray) throws Exception {

		//TODO: Use a DI-Container or Service locator to inject this
		OGPGPKeysStorage ogpgpKeysStorage = new RestDbOGPGPKeysStorage();
		List<OGPGPKey> pgpKeys = (emailArray == null || emailArray.isEmpty()) ?
		                         ogpgpKeysStorage.getForUser(userid, cid) :
		                         ogpgpKeysStorage.getForUserByEmail(userid, cid, emailArray,false);


		JsonArray ret = new JsonArray();
		for(OGPGPKey pgpKey : pgpKeys) {
		    JsonObject jsonKey = new JsonObject();
		    jsonKey.addProperty("ids", ListUtil.listToString(pgpKey.getKeyIds(), ListUtil.WHITE_SPACE_SEP));
		    jsonKey.addProperty("email", pgpKey.getEmail());
            jsonKey.addProperty("share", pgpKey.getShareLevel());
            jsonKey.addProperty("shared", (userid == pgpKey.getUserId() ? false : true));  // mark if this is a shared key
            jsonKey.addProperty("inline", pgpKey.isInline());
            ret.add(jsonKey);
		}

		if (emailArray.size() > 0) {
		    Access acc = new Access();
		    for (int i = 0; i < emailArray.size(); i++) {
		        String email = emailArray.get(i);
		        GuardKeys gkey = acc.getKeysFromEmail(email);
		        if (gkey != null) {
		            JsonArray info = PgpHandler.getKeyInfo(gkey.pubring);
		            StringBuilder idBuilder = new StringBuilder();
		            for (int j = 0; j < info.size(); j++) {
		                JsonObject keyinfo = (JsonObject) info.get(j);
		                if (j>0) {
                            idBuilder.append(" ");
                        }
		                idBuilder.append(keyinfo.get("Key").getAsString());
		            }
		            JsonObject key = new JsonObject();
		            key.addProperty("ids", idBuilder.toString());
		            key.addProperty("email", email);
		            key.addProperty("share", false);
		            key.addProperty("guard", true);
		            key.addProperty("inline", gkey.inline);
		            ret.add(key);

		        }
		    }
		}
		return(ret);
	}

	private JsonArray getKeyDataFromDB(DbQuery db, int userid) throws Exception {
	    JsonArray retdata = new JsonArray();
	    while (db.next()) {
            JsonObject key = new JsonObject();
            key.addProperty("ids", db.rs.getString("ids"));
            key.addProperty("email", db.rs.getString("Email"));
            key.addProperty("share", db.rs.getInt("share_level"));
            key.addProperty("shared", (userid == db.rs.getInt("userid") ? false : true));  // mark if this is a shared key
            key.add("inline", new JsonPrimitive(db.rs.getBit("inline")));
            retdata.add(key);
        }
        db.close();
        return(retdata);
	}

	public void deleteKey (HttpServletRequest request, HttpServletResponse response) throws Exception {
		int userid = Core.getIntParameter(request, "userid", true);
		int cid = Core.getIntParameter(request, "cid", true);
		LogAction.setUser(userid, cid);
		String ids = Core.getStringParameter(request, "ids", true);
		try {

		    String[] idList = ids.trim().split("\\s+");
		    //TODO: Using a DI-Container to resolve this
		    OGPGPKeysStorage ogpgpKeysStorage = new RestDbOGPGPKeysStorage();
		    ogpgpKeysStorage.Delete(userid, cid, Arrays.asList(idList));

			LogAction.setSuccess();
			Core.sendOK(response, "OK");
		} catch (Exception ex) {
		    LogAction.setFail();
			logger.error("Problem deleting public pgp key ", ex);
			Core.sendFail(response, "Fail");
		}
	}

	public void shareKey (HttpServletRequest request, HttpServletResponse response) throws Exception {
		int userid = Core.getIntParameter(request, "userid", true);
		int cid = Core.getIntParameter(request, "cid", true);
		String ids = Core.getStringParameter(request, "ids", true);
		String[] idList = ids.trim().split("\\s+");
		LogAction.setUser(userid, cid);
		String share = Core.getStringParameter(request, "share", true);
		int sharelevel = share.toLowerCase().trim().equals("true") ? 1 : 0;
		try {
		    //TODO: Use a DI-Container for resolving this
		    RestDbOGPGPKeysStorage ogpgpKeysStorage = new RestDbOGPGPKeysStorage();
		    ogpgpKeysStorage.UpdateShareLevel(userid, cid, Arrays.asList(idList), sharelevel);

			LogAction.setSuccess();
			Core.sendOK(response, "OK");
		} catch (Exception ex) {
		    LogAction.setFail();
			logger.error("Problem sharing public pgp key ", ex);
			Core.sendFail(response, "Fail");
		}
	}

	public void changeInline (HttpServletRequest request, HttpServletResponse response) throws Exception {
		int userid = Core.getIntParameter(request, "userid", true);
		int cid = Core.getIntParameter(request, "cid", true);
		LogAction.setUser(userid, cid);
		String ids = Core.getStringParameter(request, "ids", true);
	    String[] idList = ids.trim().split("\\s+");
		String inlineStr = Core.getStringParameter(request, "inline", true);
		boolean isInline = inlineStr.toLowerCase().trim().equals("true") ? true : false;
		try {
		    //TODO: Use a DI-Container to resolve this
		    RestDbOGPGPKeysStorage ogpgpKeysStorage = new RestDbOGPGPKeysStorage();
		    ogpgpKeysStorage.UpdateInlineMode(userid, cid, Arrays.asList(idList), isInline);

			LogAction.setSuccess();
			Core.sendOK(response, "OK");
		} catch (Exception ex) {
		    LogAction.setFail();
			logger.error("Problem changing inline status public pgp key ", ex);
			Core.sendFail(response, "Fail");
		}
	}

	public static GuardKeys getKeyForPGP (int id, int cid, long keyid) throws Exception {
		return (getKeyForPGP (id, cid, keyid, false));
	}

	/**
	 * Return the Guard Keys populated with PGP data
	 * @param id
	 * @param cid
	 * @param keyid
	 * @param incoming - true if current private key search only
	 * @return
	 * @throws Exception
	 */
	public static GuardKeys getKeyForPGP (int userid, int cid, long keyid, boolean incoming) throws Exception {
	    //TODO: Use a DI-Container for resolving
	    PGPKeysStorage pgpKeysStorage = new RestDbPGPKeysStorage();
	    PGPKeys keyMapping = pgpKeysStorage.getByKeyIdInContext(cid, keyid);
	    if(keyMapping != null) {
           Access acc = new Access();
           GuardKeys keys = acc.getKeys (userid,
                                         cid,
                                         keyMapping.getKeyId(),
                                         !incoming);  // If incoming, we want all of the keys, not just the current
            return (keys);
	    }
	    if (incoming) {
            return(null);
        }
        return (getUploadedPublic (userid, cid, Core.PGPKeyId(keyid)));
    }

	private static GuardKeys getUploadedPublic (int userid, int cid, String pubid) throws Exception {
	    OGPGPKeysStorage ogpgpKeysStorage = new RestDbOGPGPKeysStorage();
	    List<OGPGPKey> ogPGPKeys = ogpgpKeysStorage.getForUserByIds(userid, cid, Arrays.asList(new String[] {pubid}));
	    for(OGPGPKey key : ogPGPKeys) {
	        GuardKeys ret = new GuardKeys();
	        ret.email = key.getEmail();
	        ret.pubring = PGPPublicKeyRingFactory.create(key.getPublicPGPAscData());
	        ret.contextid = cid;
	        ret.userid = userid;
	        if(ret.userid == userid) {
	            return ret;
	        }
	    }
	    return null;
	}

	public static GuardKeys getPGPKeysFromEmail (String email, int userid, int cid) throws Exception {
        //TODO: Use a service locator to resolve this
        OGPGPKeysStorage ogpgpKeysStorage = new RestDbOGPGPKeysStorage();
        List<OGPGPKey> keys  = ogpgpKeysStorage.getForUserByEmail(userid, cid, Arrays.asList(new String[] {email}),true);
        for(OGPGPKey key : keys) {
            if(key.getUserId() == userid) {
                return GuardKeysFactory.Create(key);
            }
        }
        return null;
	}
}
