package com.openexchange.guard.ox.streaming;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import com.openexchange.guard.database.RecipKey;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.pgp.PGPUtils;

/**
 * Encrypts a stream of data using PGP
 *
 * @author Benjamin Gruedelbach
 */
public class PGPStreamEncryptionHandler implements StreamHandler{

    private final ArrayList<RecipKey> recipients;
    private final GuardKeys privateSigningKey;
	private final String password;
    private final boolean armor;

    /**
     * Constructor for creating an encryption and signing handler
     *
     * @param recipients The recipients to encrypt the stream data for
     * @param armor Whether PGP's ASCII armor should be used or not
     */
    public PGPStreamEncryptionHandler(ArrayList<RecipKey> recipients,
									  GuardKeys privateSigningKey,
									  String password,
									  boolean armor)
		throws Exception {
			this.recipients = recipients;
			this.armor = armor;
			this.privateSigningKey = privateSigningKey;
			this.password = password;
    }

    /**
     * Constructor for creating an encryption handler
     *
     * @param recipients The recipients to encrypt the stream data for
     * @param armor Whether PGP's ASCII armor should be used or not
     */
    public PGPStreamEncryptionHandler(ArrayList<RecipKey> recipients,
                                      boolean armor){
			this.recipients = recipients;
			this.privateSigningKey = null;
			this.password = "";
			this.armor = armor;
    }

    @Override
    public void processStream(InputStream inputStream,
        OutputStream outputStream,
        int suggestedBufferSize) throws IOException {

        try {
			if(privateSigningKey == null){
				//Just ecnrypting the file
				PGPUtils.encrypt(outputStream,
								 inputStream,
								 "",  /* WHAT IS THIS FOR? */
								 recipients,
								 armor,
								 false /* withIntegrityCheck */);
			}
			else{
				//Sign and encrypt the file
				PGPUtils.signEncryptFile(outputStream,
										 inputStream,
										 "",  /* WHAT IS THIS FOR? */
										 recipients,
										 privateSigningKey,
										 password,
										 armor,
										 false /*withIntegrityCheck*/);
			}
        } catch (Exception e) {
            throw new IOException(e);
        }
    }

    @Override
    public String getNewContentType(String oldContentType, String fileName) {
        return "pgp/" + oldContentType.replace("/", "--");
    }

    @Override
    public String getNewName(String oldName) {
        return oldName + ".pgp";
    }
}
