package com.openexchange.guard.mailvelope;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.bouncycastle.openpgp.PGPKeyRing;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.Sharding;
import com.openexchange.guard.database.ogEmail.OGEmail;
import com.openexchange.guard.database.ogEmail.OGEmailStorage;
import com.openexchange.guard.database.ogEmail.RestDbOGEmailStorage;
import com.openexchange.guard.database.ogKeyTable.OGKeyTableStorage;
import com.openexchange.guard.database.ogKeyTable.RestDbOGKeyTableStorage;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.logging.LogAction;
import com.openexchange.guard.ox.Api;
import com.openexchange.guard.ox.Upgrader;
import com.openexchange.guard.pgp.PGPPrivateKeyHandler;
import com.openexchange.guard.pgp.PGPPublicHandler;
import com.openexchange.guard.pgp.PGPPublicKeyRingFactory;
import com.openexchange.guard.pgp.parser.AsciiArmoredKeyParser;
import com.openexchange.guard.server.OxCookie;
import com.openexchange.guard.util.Core;

public class PublicKeyImporter {

    private static Logger logger = LoggerFactory.getLogger(PublicKeyImporter.class);

    /**
     * Import Public key from mailvelope
     * Requires uid to be the primary for the account
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void importPublicKey (HttpServletRequest request, HttpServletResponse response, OxCookie cookie) throws Exception {

        JsonObject json = Core.getJSON(request);
        // Verify logged in
        Api ap = new Api(cookie, request);
        if (!ap.verifyLogin()) {
            Core.sendFail(response, "Must be logged in");
            return;
        }
        int cid = ap.getContextId();
        if (cid > 0)
            Sharding.checkTables(cid, true);
        ArrayList<String> emails = ap.getUserEmails();
        String data = null;
        int userid = 0;
        String email = null;
        try {
        email = json.get("email").getAsString();
        if (! emails.contains(email)) {
            Core.sendFail(response, "Email not in account");
            return;
        }
        /////   Restrict to primary email address here ?
        if (!email.toLowerCase().equals(ap.getPrimary().toLowerCase())) {
            Core.sendFail(response, "Email not the primary");
            return;
        }

        if (cid != json.get("cid").getAsInt()) {
            Core.sendFail(response, "Bad login data");
            return;
        }
        userid = json.get("userid").getAsInt();
        data = json.get("keydata").getAsString();
        } catch (Exception e) {
            Core.sendFail(response, "Bad data");
            logger.error("Error getting mailvelope public key json ", e);
            return;
        }
        if (!data.contains("PGP PUBLIC")) {
            Core.sendFail(response, "Bad key data");
            return;
        }
        // Create new key with the data
        GuardKeys newkey = new GuardKeys();
        newkey.contextid = cid;
        newkey.userid = userid;
        newkey.email = email;
        newkey.setPGPKeyRingFromAsc(data);

        // Make sure the key has the email in userids
        if(!hasUserID (newkey.pubring, email)) {
            logger.debug("User email address not found in the uploaded public keys" );
            Core.sendFail(response, "Bad UID");
            return;
        }

        PGPPrivateKeyHandler pk = new PGPPrivateKeyHandler();
        Long dupl;
        // Now, we need to check to make sure we don't already have this key
        boolean duplicate = false;
        if ((dupl = pk.isDuplicate(newkey)) != 0) {  // If public key already registered, possible duplicate
            duplicate = true;
        }
        if (duplicate) {
            //////////// HANDLE MERGE/DUPLICATE public keys
            GuardKeys existing = PGPPublicHandler.getKeyForPGP(userid, cid, dupl, true);
            if (newkey.getPGPSecretKey() != null) {
                Core.sendFail(response, "Private key exists, merge not supported through mailvelope");
                return;
            }
            if ((newkey.pubring = pk.handleDuplicatePublic(newkey, existing, dupl)) == null) {
                LogAction.setFail();
                logger.error("Problem handling duplicate public key");
                Core.sendFail(response, "duplicate error");
                return;
            }
            OGKeyTableStorage ogKeyTableStorage = new RestDbOGKeyTableStorage();
            ogKeyTableStorage.updatePublicKey(newkey);

        } else {

            OGEmailStorage oge = new RestDbOGEmailStorage();
            OGEmail lookup = oge.getByEmail(email);
            // Check and see if previously sent a guest email to this user, and just now signing up as mailvelope
            if (lookup != null) {
                if (lookup.getContextId() < 0) {
                    Upgrader.upgrade(lookup.getUserId(), lookup.getContextId(), userid, cid); // try to upgrade the account
                }
            }
            else {
                oge.insertOrUpdate(email, cid, userid, 0);
            }
            Access acc = new Access();
            acc.addNewKey(newkey);
            Core.sendOK(response, "OK");
        }
    }

    /**
     * Verify has the email in userid in the ring
     * @param ring
     * @param email
     * @return
     */
    private boolean hasUserID (PGPKeyRing ring, String email) {
        Iterator<String> uidIterator = ring.getPublicKey().getUserIDs();

        while(uidIterator.hasNext()) {
            String uid = uidIterator.next().toLowerCase();
            if (uid.contains(email)) return true;
        }

        return (false);
    }

}
