/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.guard.mail;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.SignatureException;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import javax.mail.BodyPart;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Session;
import javax.mail.internet.InternetHeaders;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.lang.StringUtils;
import org.bouncycastle.openpgp.PGPException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.encr.Crypto;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.exceptions.BadPasswordException;
import com.openexchange.guard.mailcreator.Mail;
import com.openexchange.guard.pgp.PGPUtils;
import com.openexchange.guard.util.Core;


public class SignMime {

    private static Logger logger = LoggerFactory.getLogger(SignMime.class);
    
    public void processIncomingMime (HttpServletRequest request, HttpServletResponse response) throws Exception {
        FileItemFactory fif = new DiskFileItemFactory();
        ServletFileUpload sfu = new ServletFileUpload(fif);
        List files = sfu.parseRequest(request);
        Iterator iterator = files.iterator(); 
        String data = "";
        while (iterator.hasNext()) {
            FileItem fi = (FileItem) iterator.next();
            data = fi.getString("UTF-8");
        }
        int userid = Core.getIntParameter(request, "user");
        int cid = Core.getIntParameter(request, "context");
        Access acc = new Access();
        GuardKeys key = acc.getKeys(userid, cid);
        Core.sendOK(response, signMime(data, key));
        
    }
    
    /**
     * Takes a string mime message and returns signed with new attachment
     * @param mimemessage
     * @param key
     * @param password
     * @return
     * @throws MessagingException
     * @throws IOException
     * @throws SignatureException
     * @throws PGPException
     * @throws BadPasswordException 
     */
    public static String signMime (String mimemessage, GuardKeys key) throws MessagingException, IOException, SignatureException, PGPException, BadPasswordException {
        MimeParser mime = new MimeParser(mimemessage.getBytes("UTF-8"));
        MimeMessage message = mime.getMessage();
        String auth = message.getHeader("X-OX-GUARD-AUTH", "");
        message.removeHeader("X-OX-GUARD-AUTH");
        message.removeHeader("X-OX-Guard-Marker");
        boolean isInline = false;
        String inline = message.getHeader("X-Oxguard-PGPInline", "");
        if (inline != null) {
            if (inline.toLowerCase().equals("true")) isInline = true;
        }
        String password = Core.decodeRSApassword(auth);
        if (password == null) {
            logger.info("Bad password for signing");
            throw (new BadPasswordException ("Bad password"));
        }
        if (isInline) {
            SignMime sm = new SignMime();
            String signed = sm.SignInline(message, key, password);
            if (signed != null) return (signed);
            return (mimemessage);
        }
        MimeMultipart mp = mime.getFirstMultipart();

        // OK.  We need to normalize the text, remove whitespaces, etc
        String tosign;
        if (mp != null) {
            tosign = normalize(getPartContent(mp));
        } else {  // If not multipart email, get the content
            tosign = normalize(getMessageContentForSigning(message));
        }
        // Now sign
        ByteArrayInputStream in = new ByteArrayInputStream(tosign.getBytes("UTF-8"));
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        PGPUtils util = new PGPUtils();
        util.createDetachedSignature(in, out, key, password, true);
        out.close();
        // Signed wrapper
        MimeMultipart newcontent = new MimeMultipart("signed; micalg=pgp-sha1; protocol=\"application/pgp-signature\"");
        // Add our normalized text back as a body part
        MimeBodyPart bp = new MimeBodyPart(new ByteArrayInputStream(tosign.getBytes("UTF-8")));
        newcontent.addBodyPart(bp);
        // Create signature attachment
        String signature = new String(out.toByteArray());
        BodyPart sig = new MimeBodyPart();
        sig.setContent(signature, "application/pgp-signature");
        sig.setDisposition("attachment");
        sig.setFileName("signature.asc");
        sig.setHeader("Content-Transfer-Encoding", "7bit");
        sig.setHeader("Content-Type", "application/pgp-signature");
        newcontent.addBodyPart(sig);
        // Update the original mime message

        message.setContent(newcontent);
        message.saveChanges();
        // Return the new string
        return(getMessageContent(message));
        
    }
    
    /**
     * Sign a message inline format, should be plaintext
     * @param mime
     * @param key
     * @param password
     * @return
     * @throws IOException
     * @throws MessagingException
     */
    private String SignInline (MimeMessage mime, GuardKeys key, String password) throws IOException, MessagingException {
        Object content = mime.getContent();
        if (content instanceof Multipart || content instanceof MimeMultipart)  {
            // This is experimental and not working well.  HTML failing, and known limitation of Inline
            MimeMultipart newcontent = signParts ((MimeMultipart) content, key, password);
            mime.setContent(newcontent);
            mime.saveChanges();
            String data = getMessageContent(mime);
            return(data);
        } else {
            // Sign plaintext email
            if (content instanceof String) {
                String msg = (String) content;
                String signed = signMessage(msg, key, password, mime.getContentType() == null ? false : mime.getContentType().contains("html"));
                mime.setContent(signed, mime.getContentType());
                mime.saveChanges();
                String data = getMessageContent(mime);
                return(data);
            }
        }
        return (null);
        
    }

    
    /**
     * Expiremental.  Not yet working or supported.  Signing multipart alternative.  HTML failing
     * @param mp
     * @param key
     * @param password
     * @return
     * @throws MessagingException
     * @throws IOException
     */
    private MimeMultipart signParts (MimeMultipart mp, GuardKeys key, String password) throws MessagingException, IOException { 
        int count = mp.getCount();
        String mtype = mp.getContentType();
        if (mtype != null) {
            String[] typeData = mtype.split(";");
            mtype = typeData[0];
            if (mtype.contains("/")) {
                mtype = mtype.substring(mtype.indexOf("/") + 1);
            }
        } else mtype = "alternative";
        MimeMultipart newMpart = new MimeMultipart(mtype);
        for (int i = 0; i < count; i++) {
            MimeBodyPart bp = (MimeBodyPart) mp.getBodyPart(i);
            Object content = bp.getContent();
            if (content instanceof Multipart || content instanceof MimeMultipart)  {
                content = signParts ((MimeMultipart) content, key, password);
            }
            if (content instanceof String) {
                if (bp.getContentType().contains("text")) {
                    String type = bp.getContentType();
                    boolean html = type.contains("html");
                    String [] ctypes = type.split(";");
                    String encoding = "UTF-8";
                    for (String ctype : ctypes) {
                        if (ctype.contains("charset=")) {
                            int cindex = ctype.indexOf("=");
                            encoding = ctype.substring(cindex+1);
                        }
                    }
                    bp.setText(signMessage((String) content, key, password, html), encoding); 
                    bp.setHeader("Content-Type", type);
                }

            }
            newMpart.addBodyPart(bp);
        }
        return(newMpart);
    }
    
    // Sign the message and return wrapped text
    private String signMessage (String message, GuardKeys key, String password, boolean html) {
        try {
            if (html) message = Mail.getPlainText(message);
            message = normalize((String) message);
            StringBuilder sig = new StringBuilder();
            if (html) sig.append("<pre>\r\n");
            sig.append("-----BEGIN PGP SIGNED MESSAGE-----\r\nHash: SHA256\r\n\r\n");
            sig.append(message);
            sig.append("\r\n");
            String signature = signString (message, key, password);
            sig.append(signature);
            if (html) sig.append("</pre>");
        return (sig.toString());
        } catch (Exception ex) {
            logger.error("Error signing inline message", ex);
            return(message);
        }
    }
    
    private String signString (String message, GuardKeys key, String password) throws SignatureException, IOException, PGPException {
        PGPUtils util = new PGPUtils();
        ByteArrayInputStream in = new ByteArrayInputStream (message.getBytes("UTF-8"));
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        util.createDetachedSignature(in, out, key, password, true);
        String signature = new String(out.toByteArray(), "UTF-8");
        out.close();
        in.close();
        return (signature);
    }
    
    /**
     * Get the content of the multipart for signing
     * @param mp
     * @return
     * @throws IOException
     * @throws MessagingException
     */
    private static String getPartContent (MimeMultipart mp) throws IOException, MessagingException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        // we need to wrap in message to get full writeout of the mimepart contents witht he content-type header
        MimeMessage msg = new MimeMessage(Session.getInstance(new Properties()));
        msg.setContent(mp);
        msg.writeTo(out, new String []{"Message-ID", "MIME-Version"});
        out.close();
        return (new String(out.toByteArray(), "UTF-8"));
    }
    
    /**
     * Remove any whitespaces and normalize with \r\n after each line
     * @param content
     * @return
     */
    public static String normalize(String content) {
        StringBuilder sb = new StringBuilder();
        String[] lines = content.split("\n");
        for (String line : lines) {
            sb.append(StringUtils.stripEnd(line, null) + "\r\n");
        }
        return (sb.toString());
    }
    
    private static String getMessageContent (MimeMessage message) throws IOException, MessagingException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        message.writeTo(out);
        out.close();
        return(out.toString());
    }
    
    private static String getMessageContentForSigning (MimeMessage message) throws IOException, MessagingException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        message.writeTo(out, new String []{"Message-ID", "MIME-Version"});
        out.close();
        return(out.toString());
    }
    
}
