define('oxguard/tour/main', [
    'io.ox/core/tk/wizard',
    'oxguard/oxguard_core',
    'oxguard/util',
    'io.ox/backbone/mini-views/help',
    'gettext!oxguard/tour'
], function (Tour, core, util, HelpView, gt) {
    'use strict';

    function writeMail() {
        ox.registry.call('mail-compose', 'compose');
    }

    function openFiles() {
        ox.launch('io.ox/files/main');
    }

    function openSettings() {
        ox.launch('io.ox/settings/main', { id: 'oxguard/pgp' }).then(function () {
            this.setSettingsPane({ id: 'oxguard/pgp' });
        });
    }
    function setEncryptDefault() {
        ox.launch('io.ox/settings/main', { id: 'oxguard/pgp' }).then(function () {
            this.setSettingsPane({ id: 'oxguard/pgp' });
            //FIXME: global settings?
            if (window.oxguarddata.settings && !window.oxguarddata.settings.pgpdefault) {
                $('#pgpdefault').click();
            }
        });
    }

    Tour.registry.add({
        id: 'default/oxguard',
        priority: 1
    }, function () {
        if (!util.isGuardConfigured()) {
            Tour.registry.run('default/oxguard/createKeys');
            return;
        }

        var tour = new Tour(),
            composeApp;

        tour.step()
            .title(gt.pgettext('tour', 'The Guard module'))
            .content(gt.pgettext('tour', 'Security and privacy matters to everyone. That\'s why you can now send and receive encrypted emails by simply clicking on the lock icon.'))
            .hotspot('#statuslock')
            .waitFor('.og-ready')
            .on('wait', function () {
                if (!composeApp) {
                    //compose not started, yet
                    ox.once('mail:compose:ready', function (mode, app) {
                        composeApp = app;
                        //HACK: can not detect that mail compose window is rendered
                        composeApp.view.$el.addClass('og-ready');
                    });
                    ox.registry.call('mail-compose', 'compose');
                } else {
                    ox.ui.App.reuse(composeApp.cid);
                }
            })
            .on('next', function () {
                $('#statuslock.icon-unlock').click();
            })
            .end();
        tour.once('stop', function () {
            if (composeApp) {
                //prevent app from asking about changed content
                composeApp.model.dirty(false);
                composeApp.quit();
            }
        });

        tour
        .step()
            .title(gt.pgettext('tour', 'Guard Security Settings'))
            .content(gt.pgettext('tour', 'You already set up your Guard account. But you can always change your encryption password, or request a temporary password if you forget it.'))
            .on('wait', function () {
                ox.launch('io.ox/settings/main', { id: 'oxguard' }).then(function () {
                    this.setSettingsPane({ id: 'oxguard' });
                });
            })
            .waitFor('.oxguard_settings.password')
            .spotlight('.oxguard_settings.password')
            .end()
        .step()
            .title(gt.pgettext('tour', 'PGP Encryption Settings'))
            .content(gt.pgettext('tour', 'From here you can update your encryption settings. For example you can choose to send encrypted emails by default.'))
            .on('wait', function () {
                ox.launch('io.ox/settings/main', { id: 'oxguard/pgp' }).then(function () {
                    this.setSettingsPane({ id: 'oxguard/pgp' });
                });
            })
            .waitFor('.oxguard_settings.defaults')
            .spotlight('.oxguard_settings.defaults')
            .hotspot('.oxguard_settings.defaults #pgpdefault')
            .end()
        .step()
            .title(gt.pgettext('tour', 'Signature'))
            .content(gt.pgettext('tour', 'In addition to encrypting your emails you can also sign them. This means that email recipients can be sure that the email really came from you. By ticking this checkbox your signature will be added to outgoing emails by default.'))
            .on('wait', function () {
                ox.launch('io.ox/settings/main', { id: 'oxguard/pgp' }).then(function () {
                    this.setSettingsPane({ id: 'oxguard/pgp' });
                });
            })
            .waitFor('.oxguard_settings.defaults')
            .spotlight('.oxguard_settings.defaults')
            .hotspot('.oxguard_settings.defaults #pgpsign')
            .end()
        .step()
            .title(gt.pgettext('tour', 'Alternative: PGP Inline'))
            .content(gt.pgettext('tour', 'By default Guard sends encrypted emails using PGP/Mime format. To communicate with some systems you may need to use PGP Inline.'))
            .on('wait', function () {
                ox.launch('io.ox/settings/main', { id: 'oxguard/pgp' }).then(function () {
                    this.setSettingsPane({ id: 'oxguard/pgp' });
                });
            })
            .waitFor('.oxguard_settings.defaults')
            .spotlight('.oxguard_settings.defaults')
            .hotspot('.oxguard_settings.defaults #inlinedefault')
            .end()
        .step()
            .title(gt.pgettext('tour', 'Advanced Settings'))
            .content(gt.pgettext('tour', 'Advanced users can manage their key lists from the PGP Encryption Settings.'))
            .on('wait', function () {
                ox.launch('io.ox/settings/main', { id: 'oxguard/pgp' }).then(function () {
                    this.setSettingsPane({ id: 'oxguard/pgp' });
                });
            })
            .waitFor('.oxguard_settings.keys')
            .spotlight('.oxguard_settings.keys')
            .end()
        .step()
            .title(gt.pgettext('tour', 'Remove encryption'))
            .content($('<div>').append(
                $('<p>').text(gt.pgettext('tour', 'You are now able to send secure emails.')),
                $('<p>').text(gt.pgettext('tour', 'Removing the encryption is easy, too: Simply click on the lock icon again.'))
            ))
            .on('wait', function () {
                ox.ui.App.reuse(composeApp.cid);
            })
            .hotspot(['#statuslock'])
            .waitFor('.og-ready')
            .on('next', function () {
                $('#statuslock.icon-lock').click();
            })
            .end()
        .step()
            .title(gt.pgettext('tour', 'Encrypt files'))
            .content(gt.pgettext('tour', 'You can also protect your files. Just click on Encrypt and the selected file will be stored securely.'))
            .navigateTo('io.ox/files/main')
            .waitFor('.classic-toolbar')
            .hotspot('.classic-toolbar a[data-action=encrypt]')
            .on('wait', function () {
                $('.file-list-view > .list-item.selectable:not(.file-type-folder):not(.file-type-guard):first').click();
                $('.classic-toolbar [data-action=more] ~ ul').css('display', 'block');
            })
            .on('hide', function () {
                $('.classic-toolbar [data-action=more] ~ ul').css('display', '');
            })
            .end()
        .step()
            .title(gt.pgettext('tour', 'Restart Guided Tour'))
            .content(gt.pgettext('tour', '<em>Hint:</em> you can start guided tours, any time you need them, from the system menu.'))
            .hotspot('[data-action="guard-tour"]')
            .referTo('#topbar-settings-dropdown')
            .waitFor('#topbar-settings-dropdown')
            .on('wait', function () {
                $('#topbar-settings-dropdown').css('display', 'block');
            })
            .on('hide', function () {
                $('#topbar-settings-dropdown').css('display', '');
            })
            .end()
        .step()
            .title(gt.pgettext('tour', 'Guided Tour completed'))
            .content($('<div>').append(
                $('<p>').text(gt.pgettext('tour', 'You successfully learned how to protect your emails and files.')),
                $('<p>').text(gt.pgettext('tour', 'What you can do now:')),
                $('<p>').append($('<a href="#">').click(writeMail).text(gt.pgettext('tour', 'Write an encrypted email'))),
                $('<p>').append($('<a href="#">').click(openFiles).text(gt.pgettext('tour', 'Encrypt your sensitive files in Drive'))),
                $('<hr>'),
                $('<p>').append($('<a href="#">').click(setEncryptDefault).text(gt.pgettext('tour', 'Activate encryption for all new emails'))),
                $('<p>').append($('<a href="#">').click(openSettings).text(gt.pgettext('tour', 'Review your encryption settings'))),
                $('<p class="help">').append(new HelpView({
                    base: 'help-guard',
                    content: gt.pgettext('tour', 'Learn more')
                }).render().$el)
            ))
            .end()
        .start();
    });

    Tour.registry.add({
        id: 'default/oxguard/createKeys',
        priority: 1
    }, function () {
        var def = $.Deferred();

        if (util.isGuardConfigured()) return def.reject('configured');

        if (_.device('small') || _.device('smartphone')) {
            return core.createKeys();
        }

        var tour = new Tour(),
            model = new core.CreateKeysViewModel({
                initialSetup: true
            }),
            view = new core.CreateKeysView({
                model: model
            });

        tour.once('stop', function () {
            if (model.get('sent') === false) {
                def.reject('cancel');
            } else {
                def.resolve('OK');
            }

            view.remove();
            model.off();
            view = model = null;
        });
        //Guard is not configured, yet. Abort after this step in order to allow the user to start the configuration.
        tour.step()
            .title(gt.pgettext('tour', 'Create Your Guard Account'))
            .content($('<div>').append(
                $('<p>').text(gt.pgettext('tour', 'You now need to choose a password which will be used to encrypt or decrypt items. You only need to set this up once. Your password will then remain valid until you change it.'))
            ))
            .end()
        .step()
            .title(gt('Create Guard Security Keys'))
            .content(view.render().$el)
            .beforeShow(function () {
                var self = this;
                this.toggleNext(false);
                model.on('change', function (model) {
                    self.toggleNext(model.isValid());
                });
                model.once('send:ok', function () {
                    //HACK: no way to pause the wizard until some event
                    $('body').addClass('og-ready');
                });
            })
            .on('next', function () {
                model.send();
            })
            .end()
        .step()
            .title(gt.pgettext('tour', 'Guard set up completed'))
            .content($('<div>').append(
                $('<p>').text(gt.pgettext('tour', 'You successfully set up secure communication. Your private key has been created.')),
                $('<p>').text(gt.pgettext('tour', 'What you can do now:')),
                $('<p>').append($('<a href="#">').click(function () {
                    $('.wizard-step .btn-primary').click();
                    Tour.registry.run('default/oxguard');
                }).text(gt('Guided tour for Guard'))),
                $('<p>').append(new HelpView({
                    base: 'help-guard',
                    content: gt.pgettext('tour', 'Learn more')
                }).render().$el)
            ))
            .mandatory()
            .beforeShow(function () {
                //no going back, from here
                this.toggleBack(false);
            })
            .on('ready', function () {
                //HACK: no way to pause the wizard until some event
                $('body').removeClass('og-ready');
            })
            .waitFor('.og-ready')
            .end()
        .start();

        return def;
    });

    Tour.registry.add({
        id: 'default/oxguard/choice',
        priority: 1
    }, function () {
        return require(['settings!oxguard', 'oxguard/tour/choice', 'mailvelope/main', 'io.ox/core/capabilities'])
        .then(function (settings, Choice, api, capabilities) {

            var def = $.Deferred();

            if (settings.get('cryptoProvider') !== undefined) {
                def.resolve(settings.get('cryptoProvider'));
                return def;
            } else if (api.isMailvelopeSupported() && (capabilities.has('forceMailvelope') || !capabilities.has('guard-mail'))) {
                settings.set('cryptoProvider', 'mailvelope').save();
                def.resolve('mailvelope');
                return (def);
            }

            var model = settings.createModel(Backbone.Model);
            new Tour().step()
                .title(gt('Choose your security solution'))
                .content(new Choice.View({
                    model: model
                }).render().$el)
                .beforeShow(function () {
                    var tour = this;
                    this.$el.addClass('choice_wizard');

                    this.$('[data-action="back"]').hide();
                    this.$('[data-action="next"]').hide();
                    this.$('[data-action="done"]').hide();

                    model.on('change:cryptoProvider', function (model, provider) {
                        settings.save();
                        def.resolve(provider);
                        tour.trigger('done');
                    });
                })
                .end()
            .on('stop', function () {
                if (def.state() === 'pending' && settings.get('cryptoProvider') === undefined) def.reject('incomplete');
            })
            .start();

            return def;
        }).then(function (provider) {
            //FIXME: unify calls via Tour.registry API. if's are bad
            if (provider === 'guard') {
                return Tour.registry.get('default/oxguard/createKeys').get('run')();
            }
            if (provider === 'mailvelope') {
                return require(['mailvelope/tour']).then(function (runTour) {
                    return runTour();
                });
            }

            return provider;
        });
    });

    return Tour.registry;
});
