/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2011
 */

define('oxguard/mail/options_new', [
    'oxguard/mail/oxguard_mail_compose_core',
    'oxguard/mail/util',
    'gettext!oxguard'
], function (core, util, gt) {
    'use strict';

    var lockclass = 'icon-lock icon-2x fa fa-lock fa-2x oxguard_lock';

    function createlock(baton) {
        if (window.oxguarddata.settings.oxguard)  // Only people with oxguard permissions can change
            baton.view.listenTo(baton.view.model, 'change:encrypt', function () {
                setLock(baton);
            });
        setLock(baton);
    }

    function setLock (baton) {
        if (baton.model.get('encrypt')) {
            core.lock(baton);
            if (window.oxguarddata.settings.pgpsign) {
                baton.model.set('PGPSignature', true);
            }
        } else {
            core.unlock(baton);
        }
    }

    function warnHTML (baton) {
        var format = baton.model.get('PGPFormat');
        if (format === 'inline' || format === 'mime') {  // If one is selected, turn on encryption
            core.lock(baton);
        }
        if (format !== 'inline') return;
        if (baton.model.get('editorMode') === 'text') return;
        require (['io.ox/core/tk/dialogs'], function (dialogs) {
            var dialog = new dialogs.CreateDialog({ width: 450, height: 300, center: true });
            dialog.header($('<h4>').text(gt('HTML Formatted Email')));
            var body = $('<p>');
            body.append(gt('HTML formatted emails may cause issues with PGP inline.  Click OK to change to Plain Text formatted Emails'));
            dialog.getBody().append(body);
            dialog
            .addPrimaryButton('ok', gt('OK'), 'okx')
            .addButton('cancel', gt('Cancel'), 'cancel')
            .on('ok', function () {
                baton.model.set('editorMode', 'text');
            })
            .show();
        });
    }

    /// Disable the recipient inputs and color the background
    function lockReply (baton) {
        var view = baton.view.$el;
        view.find('.token-input').attr('disabled', 'disabled');
        view.find('.token-input').addClass('og_disabled').click(notice);
        view.find('.tokenfield').addClass('og_disabled').click(notice);
    }

    function notice () {
        require (['io.ox/core/notifications'], function (notifications) {
            notifications.yell('error', gt('Recipients cannot be added to this encrypted reply'));
        });
    }

    function getType(current) {
        if (current === null || current === undefined) {  // if no decoded data, then return default
            if (window.oxguarddata.settings.inline === true) {
                return ('inline');
            } else return ('mime');
        }
        if (current.pgpFormat === 'pgpmime') return ('mime');
        return ('inline');
    }

    function createOptions(baton, count) {
        if (window.oxguarddata.settings.oxguard === undefined) {  // If we don't yet have response from the Guard backend, check every second then handle
            if (count === undefined) count = 0;
            if (count > 10) return;  // limit our loop to 10 seconds
            window.setTimeout(function () {
                count++;
                createOptions(baton, count);
            }, 1000);
        }
        // Lets see if we should allow user to see OxGuard compose
        if ((window.oxguarddata.settings.oxguard === false) || (window.oxguarddata.settings.oxguard === undefined)) {
            if (window.oxguarddata.action !== 'secureReply') {// OK if secure reply, otherwise
                return;
            }
        }
        if (window.oxguarddata.action === 'sendfile') {// If it is going to be sending already encrypted file, then don't add encryption options
            window.oxguarddata.action = null;
            return;
        }
        if (window.oxguarddata.sentfolder === undefined) {// Get sent folder data for sending
            require(['settings!io.ox/mail']).done(function (settings) {
                window.oxguarddata.sentfolder = settings.get('folder/sent');
                window.oxguarddata.draftfolder = settings.get('folder/drafts');
            });
        }
        window.setTimeout (function () {
            createlock(baton);
            // If this is a secure reply, then disable all of the options to make it plain text
            if ((window.oxguarddata.action === 'secureReply') || (window.oxguarddata.action === 'fileAttach') || (util.hasEncryptedAttachment(baton) > 0)) {
                if (window.oxguarddata.replyInline === true) {
                    baton.model.set('PGPFormat', 'inline');
                    window.oxguarddata.replyInline = null;
                }
                if (baton.model.get('PGPFormat') === undefined) { // If not previously defined
                    baton.model.set('PGPFormat', getType(window.oxguarddata.current));
                }
                baton.model.set('encrypt', true);
                if (window.oxguarddata.settings.oxguard === false || window.oxguarddata.settings.secureReply === true) {
                    baton.view.$el.find('#statuslock')
                    .attr('class', lockclass)
                    .click(false)
                    .attr('title', gt('Reply must be encrypted'));
                    baton.view.$el.find('[data-name="encrypt"]').click(false);
                }
                if ((window.oxguarddata.action === 'secureReply') && (window.oxguarddata.settings.oxguard === false)) {
                    // If secure reply, and does not have OG, only allow reply to orig
                    window.setTimeout (function () {
                        lockReply (baton);
                    }, 500);
                }
                window.oxguarddata.action = null;
            }
        }, 200);

        if (baton.model.get('PGPSignature') === undefined) {
            if (window.oxguarddata.settings.pgpsign) {
                baton.model.set('PGPSignature', true);
            }
        }

        if (baton.model.get('newmail') === true) {  // New email doesn't have recips yet.  Check to see if there are defaults
            // If default is to use guard, go ahead and lock
            if (window.oxguarddata.settings.pgpdefault) {
                baton.model.set('encrypt', true);
                core.lock(baton);
            } else baton.model.set('encrypt', false);
            if (window.oxguarddata.settings.inline === true) {
                if (window.oxguarddata.settings.pgpdefault) {
                    baton.model.set('PGPFormat', 'inline');
                    warnHTML(baton);
                } else {
                    baton.model.set('PGPFormat', 'i');  // deactivated inline status
                }
            } else baton.model.set('PGPFormat', 'mime');
            baton.model.unset('newmail');
        }
        baton.model.on('change:PGPFormat', function () {
            warnHTML(baton);
        });
        return;
    }

    return {
        createOptions: createOptions,
        warnHTML: warnHTML
    };
});
