/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2011
 */
define('oxguard/mail/mail_extensions_compose', [
    'io.ox/core/extensions',
    'oxguard/mail/prep_reply',
    'io.ox/mail/util',
    'settings!io.ox/mail',
    'io.ox/core/api/account',
    'gettext!oxguard'
], function (ext, prep, util, settings, account, gt) {
    'use strict';

    var isDraftFolder = function (folder_id) {
        return _.contains(account.getFoldersByType('drafts'), folder_id);
    }, isDraftMail = function (mail) {
        return isDraftFolder(mail.folder_id) || ((mail.flags & 4) > 0);
    };

    ext.point('io.ox/mail/actions/reply').extend({
        id: 'reply_guard',
        index: 1,
        requires: function (e) {
            // must be top-level
            if (!e.collection.has('toplevel', 'some')) return;
            // multiple selection
            if (e.baton.selection && e.baton.selection.length > 1) return;
            // multiple and not a thread?
            if (!e.collection.has('one') && !e.baton.isThread) return;
            // get first mail
            var data = e.baton.first();
            // has sender? and not a draft mail
            return util.hasFrom(data) && !isDraftMail(data);
        },
        action: function (baton) {
            getCurrent(baton);
            if (window.oxguarddata.current === undefined) {
                return;
            }
            try {
                if (window.oxguarddata.current.og || window.oxguarddata.current.PGP) {
                    doReply(false);
                    baton.preventDefault();
                    //baton.disable('io.ox/mail/actions/reply', 'default');
                }
            } catch (e) {
                console.log(e);
            }
        }
    });

    ext.point('io.ox/mail/actions/reply-all').extend({
        id: 'reply_all_guard',
        index: 1,
        requires: function (e) {
            // must be top-level
            if (!e.collection.has('toplevel', 'some')) return;
            // multiple selection
            if (e.baton.selection && e.baton.selection.length > 1) return;
            // multiple and not a thread?
            if (!e.collection.has('one') && !e.baton.isThread) return;
            // get first mail
            var data = e.baton.first();
            // other recipients that me? and not a draft mail
            return util.hasOtherRecipients(data) && !isDraftMail(data);
        },
        action: function (baton) {
            getCurrent(baton);
            if (window.oxguarddata.current !== null) {
                var data = window.oxguarddata.current;
                if (data === undefined) return;
                if (data.PGPDecoded) {  // If we have PGP decode data, then use it to compose new email
                    window.oxguarddata.action = 'secureReply';
                    doReply (data, true);
                    baton.preventDefault();
                    //baton.disable('io.ox/mail/actions/reply-all', 'reply-all');
                    return;
                }
            }
        }
    });

    ext.point('io.ox/mail/actions/forward').extend({
        id: 'forward_guard',
        index: 1,
        requires: function (e) {
            return e.collection.has('toplevel', 'some');
        },
        action: function (baton) {
            getCurrent(baton);
            if (window.oxguarddata.current && window.oxguarddata.current.PGPDecoded) {
                var data = window.oxguarddata.current;
                window.oxguarddata.action = 'secureReply';
                doForward (data);
                baton.preventDefault();
                //baton.disable('io.ox/mail/actions/forward', 'forward');
                return;
            }
        }
    });

    ext.point('io.ox/mail/actions/edit').extend({
        id: 'edit_gaurd',
        index: 1,
        requires: function (e) {
            // must be top-level
            if (!e.collection.has('toplevel')) return;
            // multiple selection
            if (e.baton.selection && e.baton.selection.length > 1) return;
            // multiple and not a thread?
            if (!e.collection.has('one') && !e.baton.isThread) return;
            // get first mail
            var data = e.baton.first();
            // must be draft folder
            return data && isDraftMail(data);
        },
        action: function (baton) {
            var data = baton.first(),
                check = false;
            _.each(ox.ui.apps.models, function (app) {
                if (app.refId === data.id) {
                    check = true;
                    app.launch();
                }
            });
            if (check === true) return;
            if (data.PGPDecoded) {  // If we have PGP decode data, then mark as encrypted so that key is locked
                getCurrent(baton);
                window.oxguarddata.action = 'fileAttach';
                doDraft (window.oxguarddata.current);
                baton.preventDefault();
                //baton.disable('io.ox/mail/actions/edit', 'edit');
                return;
            }
        }
    });

    // Update guard current with the active email, be it selected or top of thread
    function getCurrent (baton) {
        if (baton.data instanceof Array) {
            // if just decoded, current will be udated, otherwise, load from baton
            if ((window.oxguarddata.current !== null) && (window.oxguarddata.current !== undefined)) return;
            window.oxguarddata.current = baton.data[0].Guard;
            return;
        }
        window.oxguarddata.current = baton.data.Guard;
    }

    ext.point('io.ox/mail/actions/compose').replace({
        id: 'default',
        action: function (baton) {
            ox.registry.call('mail-compose', 'compose', { folder_id: baton.app.folder.get(), newmail: true });
        }
    });

    // Do the setup and actions of reply
    function doReply (all) {
        var data = JSON.parse(JSON.stringify(window.oxguarddata.current));
        data.attachments = stripAttach(data);
        data = prepContent(data, 'reply');  //  Add information to forward body
        if (all) {
            var recips = addTo();
            data.to = recips.to;
            data.cc = recips.cc;
        } else {
            data.to = [];
            data.cc = [];
        }
        try {
            var replyto = window.oxguarddata.current.headers['Reply-To'];
            if (replyto !== undefined && replyto.indexOf('<') > -1) {
                var email = replyto.substring(replyto.indexOf('<') + 1);
                email = email.replace('>', '');
                var sender = [replyto.substring(0, replyto.indexOf('<')), email];
                data.to.push(sender);
            } else {
                data.to.push(window.oxguarddata.current.from[0]);
            }
        } catch (e) {
            console.log(e);
            data.to.push(window.oxguarddata.current.from[0]);
        }
        window.oxguarddata.action = 'secureReply';
        window.oxguarddata.replyInline = isInline(data);
        data.headers['In-Reply-To'] = data.headers['Message-ID'];
        data.msgref = data.folder_id + '/' + data.id;
        if (data.headers.References === undefined) {
            data.headers.References = data.headers['Message-ID'];
        } else {
            data.headers.References = data.headers.References + ' ' + data.headers['Message-ID'];
        }
        data.subject = gt ('Re: %1$s', data.subject);
        openEditor (data);
    }

    function openEditor (data) {
        if (getFormatType(data) === 'text') {
            ox.registry.call('mail-compose', 'compose', { to: data.to, cc: data.cc, subject: data.subject, attachments: data.attachments, editorMode: 'text', preferredEditorMode: 'text', content_type: 'text' })
            .done(function (e) {
                e.app.view.editor.focus();
                checkSaved(data, e.app.model);
            });
        } else {
            ox.registry.call('mail-compose', 'compose', { to: data.to, cc: data.cc, subject: data.subject, attachments: data.attachments, editorMode: 'html' })
            .done(function (e) {
                e.app.view.editor.focus();
                checkSaved(data, e.app.model);
            });
        }
    }

    // Check header data to see if Signing/inline has been saved for a draft email
    function checkSaved(data, model) {
        try {
            if (data.msgref !== undefined) {
                if (data.headers['In-Reply-To'] !== undefined) {
                    var msgref = {
                        'In-Reply-To': data.headers['In-Reply-To'],
                        'References': data.headers.References
                    };
                    model.set('headers', msgref);
                } else {
                    var msgref = {
                        'X-OxGuard-Draft': data.msgref
                    };
                    model.set('headers', msgref);
                }
            }
            if (data.headers === undefined) return;
            var headers = data.headers;
            if (headers['X-OxGuard-Sign'] === 'true') model.set('PGPSignature', true);
            if (headers['X-OxGuard-Format'] !== undefined) {
                model.set('PGPFormat', headers['X-OxGuard-Format']);
            }
        } catch (e) {
            console.log(e);
        }
    }

    // Get format of compose.  If reply is text, then open as text.  Also text if mandates inline
    function getFormatType (data) {
        if ((data.replytype === 'text/plain')) {
            return ('text');
        }
        return ('html');
    }

    // Strip all attachments that are not text and inline for replies
    function stripAttach (data) {
        var newattach = [];
        for (var i = 0; i < data.attachments.length; i++) {
            var att = data.attachments[i];
            if (att.content_type.indexOf('text') > -1) {
                if (att.disp === 'inline') newattach.push(att);
            }
        }
        return (newattach);
    }

    function isInline (data) {
        if ((data.attachments === undefined) || (data.attachments.length === 0)) return (false);
        return (data.attachments[0].pgpFormat === 'pgpinline');
    }

    // For replyAll, we need to add back all the recipients from decrypted email
    function addTo () {
        var to = window.oxguarddata.current.to;
        var newto = [];
        //   var user = baton.app.getView().form.find('.sender-dropdown').val();// Get current users email
        var user = settings.get('defaultSendAddress', '');
        for (var i = 0; i < to.length; i++) {
            if (user.indexOf(to[i][1]) < 0) { // dont add the user to recip list
                newto.push(to[i]);
            }
        }
        var cc = window.oxguarddata.current.cc;
        var newcc = [];
        for (var i = 0; i < cc.length; i++) {
            if (user.indexOf(cc[i][1]) < 0) { // dont add the user to recip list
                newcc.push(cc[i]);
            }
        }
        return {
            to: newto,
            cc: newcc
        };
    }

    // Add decoded data with prepared text to body and set content_type
    function prepContent (data, type) {
        if (data.attachments[0] === undefined) {
            var blank = {
                content: '<br>',
                content_type: 'text/html',
                disp: 'inline',
                id: 1,
                size: 1
            };
            data.attachments.push(blank);
        }
        if (data.replytype === 'text/plain') {
            data.attachments[0].content = prep.prepareTextReply($.extend({}, data), type);
            data.attachments[0].content_type = 'text/plain';
        } else {
            data.attachments[0].content = prep.prepareReply($.extend({}, data), type);
            data.attachments[0].content_type = 'text/html';
        }
        return (data);
    }

    function doForward (data) {
        require (['oxguard/mail/prep_reply'], function (prep) {
            var text = (getFormatType (data) === 'text');
            var attachment = {  // Prepare main body
                content: text ?
                    prep.prepareTextReply($.extend({}, data), 'forward') :
                    prep.prepareReply($.extend({}, data), 'forward'),
                content_type: text ? 'text/plain' : 'text/html',
                disp: 'inline'
            };
            var attachments = [attachment];
            var msgref = data.folder_id + '/' + data.id;
            for (var i = 0; i < data.attachments.length; i++) {  // Add all the attachments
                if (data.attachments[i].disp !== 'inline') {
                    data.attachments[i].ogAttach = msgref;
                    attachments.push(data.attachments[i]);
                }
            }
            data.attachments = attachments;
            data.subject = gt ('Fwd: %1$s', data.subject);
            openEditor (data);
        });
    }

    function doDraft (data) {
        var attachment = {  // Prepare main body, need cleanup of components, can't just push
            content: data.attachments[0].content,
            content_type: data.attachments[0].content_type,
            disp: 'inline'
        };
        var attachments = [attachment];
        var msgref = data.folder_id + '/' + data.id;
        for (var i = 0; i < data.attachments.length; i++) {  // Add all the attachments
            if (data.attachments[i].disp !== 'inline') {
                data.attachments[i].ogAttach = msgref;
                attachments.push(data.attachments[i]);
            }
        }
        data.attachments = attachments;
        data.msgref = msgref;
        openEditor (data);
    }
});
