define('oxguard/files/api', [
    'oxguard/core/og_http',
    'io.ox/files/api',
    'gettext!oxguard'
], function (og_http, filesAPI, gt) {
    'use strict';

    var api = {};

    api.encrypt = function encrypt(list) {
        list = [].concat(list);
        //remove all encrypted files from list
        list = list.filter(function (file) {
            return !file.isEncrypted();
        });
        if (list.length === 0) return;

        var def = $.Deferred();

        //queue the encryption requests (encrypt one file at a time)
        list.reduce(function (acc, file) {
            return acc.then(function () {
                return upload(file).then(function uploaded(response) {
                    file.set('id', response.data);
                    file.set('filename', file.get('filename') + '.pgp');
                    filesAPI.propagate('change:file', {
                        id: file.get('id'),
                        folder_id: file.get('folder_id')
                    });
                }, function uploadFailed(e) {
                    def.reject(e);
                })
                .always(function () {
                    def.notify(file);
                });
            });
        }, $.when()).then(def.resolve, def.reject);

        return def;
    };

    function upload(file) {

        var params = '&id=' + file.get('id') +
            '&folder=' + file.get('folder_id') +
            '&session=' + ox.session +
            '&userid=' + ox.user_id +
            '&cid=' + ox.context_id +
            (file.get('version') ? ('&version=' + file.get('version')) : '') +
            '&language=' + ox.language +
            '&filename=' + encodeURIComponent(file.get('filename'));

        // Send for encryption
        return og_http.post(ox.apiRoot + '/oxguard/file?action=encrypt', params).then(function (res) {
            try {
                return JSON.parse(res);
            } catch (e) {
                console.warn('Error parsing response:', e);
                return undefined;
            }
        }, function (e) {
            return (e);
        });
    }

    api.removeEncryption = function removeEncryption(list, auth) {
        var def = $.Deferred();
        list = [].concat(list);
        list.forEach(function (file) {
            var params = '&id=' + file.get('id') +
                '&folder=' + file.get('folder_id') +
                '&session=' + ox.session +
                '&auth=' + encodeURIComponent(auth.auth) +
                '&userid=' + ox.user_id +
                '&cid=' + ox.context_id +
                (file.get('version') ? ('&version=' + file.get('version')) : '') +
                '&download=' + false +
                '&language=' + ox.language +
                '&encrextra=' + encodeURIComponent(auth.encrExtra || '') +
                '&filename=' + (file.get('origName') === undefined ? encodeURIComponent(file.get('filename')) : encodeURIComponent(file.get('origName')));

            og_http.get(ox.apiRoot + '/oxguard/file?action=remove', params).then(function (response) {
                try {
                    response = JSON.parse(response);
                } catch (e) {
                    console.warn('Error parsing response:', e);
                }
                file.set('id', response.data);
                file.set('filename', file.get('filename').replace(/\.(grd|grd2|pgp)$/, ''));

                filesAPI.propagate('change:file', {
                    id: file.get('id'),
                    folder_id: file.get('folder_id')
                });
                def.resolve(response);
                return response;
            })
            .fail(function (data) {
                require (['io.ox/core/notifications'], function (notifications) {
                    if (data.responseText.indexOf('Bad Password') > -1) {
                        //#. Bad, incorrect password
                        notifications.yell('error', gt('Bad Password'));
                        def.reject('Bad password');
                        return;
                    }
                    if (data.responseText.indexOf('Unable to save') > -1) {
                        notifications.yell('error', gt('Unable to save decrypted file'));
                        def.reject('Unable');
                        return;
                    }
                    notifications.yell('error', gt('Unable to decrypt') + '\r\n' + data.responseText);
                    def.reject('error');
                });
            });
        });
        return (def);
    };

    api.upload = function (options) {
        options.module = 'oxguard/files';
        options.params = {
            action: 'new',
            filename: options.filename,
            userid: ox.user_id,
            cid: ox.context_id
        };
        return filesAPI.upload(options);
    };

    api.versions = {
        upload: function (options) {
            options.module = 'oxguard/files';
            options.params = {
                action: 'update',
                filename: options.filename,
                userid: ox.user_id,
                cid: ox.context_id
            };
            return filesAPI.versions.upload(options);
        }
    };

    //dummy event hub. Guard API does not need to trigger any events, yet.
    api.trigger = _.noop;

    return api;
});
