/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.setup;

import java.util.ArrayList;
import java.util.Iterator;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;
import com.openexchange.guard.database.DbStrings;
import com.openexchange.guard.database.Keymanager;
import com.openexchange.guard.database.OxResultSet;
import com.openexchange.guard.database.globalSettings.GlobalSettingsStorage;
import com.openexchange.guard.database.globalSettings.RestDbGlobalSettingsStorage;
import com.openexchange.guard.database.paging.ConfigDbPagedDbQuery;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.util.Core;


public class Upgrader {

	private class Schema {
		public String name;
		public int cid;
		public int db;

		public Schema (String name, int cid, int db) {
			this.name = name;
			this.cid = cid;
			this.db = db;
		}
	}

	private static Logger logger = LoggerFactory.getLogger(Upgrader.class);

    final com.openexchange.guard.dbpool.Manager umgr = new com.openexchange.guard.dbpool.Manager();

    int previousVersion = 0;

    final String updatePGPcommand = "ALTER TABLE `og_KeyTable`" +
            " MODIFY COLUMN `PGPSecret`  text CHARACTER SET utf8 COLLATE utf8_unicode_ci NULL AFTER `id`, " +
			" MODIFY COLUMN `PGPPublic`  text CHARACTER SET utf8 COLLATE utf8_unicode_ci NULL DEFAULT NULL AFTER `PGPSecret`," +
			" ADD COLUMN `keyid`  bigint NULL DEFAULT '0' AFTER `settings`," +
			" ADD COLUMN `version`  int(10) NOT NULL DEFAULT 0 AFTER `keyid`," +
			" ADD COLUMN `current`  bit NOT NULL DEFAULT b'1' AFTER `settings`, " +
			" ADD COLUMN `inline`  bit NOT NULL DEFAULT b'0' AFTER `version`, " +
			" DROP PRIMARY KEY, " +
			" ADD PRIMARY KEY (`id`, `cid`, `keyid`); ";

    final String alterFileCache = "ALTER TABLE `FileCache` ADD PRIMARY KEY (`itemID`);";
    final String alterSharding = "ALTER TABLE `sharding` MODIFY COLUMN `current_shard`  int(11) NOT NULL FIRST , ADD PRIMARY KEY (`current_shard`);";

    public boolean checkUpgrade () throws Exception {

        //TODO: Use a Service locator / DI-Container for resolving
        GlobalSettingsStorage globalSettingsStorage = new RestDbGlobalSettingsStorage();

        int previousVersion = Integer.parseInt(globalSettingsStorage.getByKey("version"));
        return previousVersion < Config.version;
    }
    /**
     * For future database upgrade action.
     *
     * @return
     * @throws Exception
     */
    public boolean doUpgrade() throws Exception {

        if (!com.openexchange.guard.util.CheckOx.oxAlive()) {
            logger.error("OX Not responding.  Will wait 15 seconds and try again");
            Thread.sleep(15000);
            if (!com.openexchange.guard.util.CheckOx.oxAlive()) {
                logger.error("OX Not responding.  Will wait 15 seconds and try again");
                Thread.sleep(15000);
                if (!com.openexchange.guard.util.CheckOx.oxAlive()) {
                    logger.error("FAILED TO CONNECT TO OX BACKEND.  EXITING");
                    System.out.println("Unable to connect to OX backend.  Verify open-xchange-rest is installed and this server has access to port " + Config.ox_backend_port);
                    Runtime.getRuntime().halt(1);
                }
            }
        }

        if (previousVersion < 7) {
        	Keymanager keyman = new Keymanager();
            GuardKeys keys = keyman.createMasterKeysNC(-2, 0, "dont@remove.com", "master", Config.rpass, "", false,true);
            if (keys == null) {
            	logger.error("Unable to create master keys for client encryption");
                return (false);
            }

        }


        try {
            return (setCurrent(Config.version));
        } catch (Exception ex) {
            logger.error("Error setting new version to " + Config.version, ex);
            return (false);
        }

    }

    private boolean setCurrent(int version) {
        try {
            //TODO: Use a Service locator / DI-Container for resolving
            GlobalSettingsStorage restDbGlobalSettingsStorage = new RestDbGlobalSettingsStorage();
            restDbGlobalSettingsStorage.Update("version", Integer.toString(version));
            return true;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }

    private void addPGPSchema (Schema schema) throws Exception {
    	String test = "SELECT id FROM og_KeyTable LIMIT 1";
    	DbCommand testcom = new DbCommand (test);
    	DbQuery testdb = new DbQuery();
    	if (schema.cid > 0) {
    	    try {
    	        testdb.readFromDB(testcom, schema.cid);
    	    } catch (Exception e) {
                if (e.getMessage().contains("doesn't exist")) {
                    logger.info("No tables for database " + schema.name);
                    return;
                }
                logger.error("Problem reading from database", e);
                return;
            }
    	} else {
    		if (schema.cid < 0) {
    			try {
    				testdb.readFromShard(testcom, schema.db);
    			} catch (Exception e) {
    				if (e.getMessage().contains("doesn't exist")) {
    					logger.info("No guest tables for guest database " + schema.name);
    					return;
    				}
    				logger.error("Problem reading from guest database", e);
    				return;
    			}
    		} else {
    			if (schema.cid == 0) {
    				testdb.readOG(testcom);
    			}
    		}
    	}
    	testdb.next();
    	boolean updated = testdb.rs == null ? false : testdb.rs.has("keyid");
    	testdb.close();
    	if (!updated) {
    		System.out.println("Updating " + schema.name);
	    	DbCommand com = new DbCommand (updatePGPcommand);
	    	DbQuery db = new DbQuery ();
	    	if (schema.cid > 0) {
    			db.writeToDB(com, schema.cid);
    		} else {
    			if (schema.cid < 0) {
    				try {
    					db.writeToShard(com, schema.db);
    				} catch (Exception e) {
    					logger.error("Problem updating guest database", e);
    				}
    			} else {
    				if (schema.cid == 0) {
    					db.writeOxGuard(com);
    				}
    			}
    		}
	    	db.close();
	    	String command = "UPDATE og_KeyTable SET version = -1";
	    	com = new DbCommand(command);
	    	db = new DbQuery ();
	    	if (schema.cid > 0) {
    			db.writeToDB(com, schema.cid);
    		} else {
    			if (schema.cid < 0) {
    				try {
    					db.writeToShard(com, schema.db);
    				} catch (Exception e) {
    					logger.error("Problem upgrading guest database", e);
    				}
    			} else {
    				if (schema.cid == 0) {
    					db.writeOxGuard(com);
    				}
    			}
    		}
	    	db.close();
	    	System.out.println("Done Upgrading " + schema.name);
    	} else {
    		System.out.println (schema.name + " OK");
    	}
    	DbCommand com = new DbCommand (DbStrings.createPGPKeysTable);
    	DbQuery crDb = new DbQuery();
		if (schema.cid > 0) {  // Only createing PGP Keys table for local users
	    	try {

	    			crDb.writeToDB(com, schema.cid);

	    	} catch (Exception e) {
	    		if (crDb.getError().contains("exists")) {
	    			System.out.println("PGPTable exists");
	    		} else {
	    			logger.error("Problem creating PGPKeyTable ", e);
	    		}
	    	}
		} else {
		    if (schema.cid < 0) {
		        try {
		            crDb.writeToShard(com, schema.db);
		        } catch (Exception e) {
		            if (crDb.getError().contains("exists")) {
	                    System.out.println("PGPTable exists");
	                } else {
	                    logger.error("Problem creating PGPKeyTable ", e);
	                }
		        }
		    }
		}
    }

    public void doPGPUpdateAll () throws Exception {
        addMissingDBKeys();
    	updateAllSchemas();
    	addPGPLookup();
    	PGPTables();
    	System.out.println();
    	System.out.println("Done");
    }


    public void updateAllSchemas() throws Exception {
        String command = "SELECT db_schema, cid from context_server2db_pool GROUP BY db_schema ORDER BY db_schema,cid";  // This is only legal in MySql.  We don't care what cid is returned
        ConfigDbPagedDbQuery query = new ConfigDbPagedDbQuery(command);
        query.readFromConfigDb();
        OxResultSet row = null;
        while( (row = query.next()) != null) {
            Schema scheme = new Schema(row.getString("db_schema"), row.getInt("cid"), 0);
            addPGPSchema(scheme);
        }

        command = "SELECT current_shard FROM sharding";
        DbCommand com = new DbCommand(command);
        DbQuery db = new DbQuery();
        db.readOG(com);
        if (db.next()) {
            int shardmax = db.rs.getInt("current_shard");
            for (int i = 1; i <= shardmax; i++ ) {
                Schema scheme = new Schema("Guest shard " + i, -1, i);
                addPGPSchema(scheme);
            }
        }

        Schema scheme = new Schema("Master Key Table", 0, 0);
        addPGPSchema(scheme);
    }

    public void addPGPLookup () throws Exception {
    	try {
    		DbCommand com = new DbCommand(DbStrings.createPGPKeysLookupTable);
    		DbQuery db = new DbQuery();
    		db.writeOxGuard(com);
    		db.close();
    	} catch (Exception ex) {
    		logger.error("Problem adding PGP Keys Lookup Table ", ex);
    	}
    	try {
    	    DbCommand com = new DbCommand(DbStrings.createRemoteKeyCache);
    	    DbQuery db = new DbQuery();
    	    db.writeOxGuard(com);
    	    db.close();
    	} catch (Exception ex) {
            logger.error("Problem adding Remote Key Cache Lookup Table ", ex);
        }
        try {
            DbCommand com = new DbCommand(DbStrings.createRemoteKeys);
            DbQuery db = new DbQuery();
            db.writeOxGuard(com);
            db.close();
        } catch (Exception ex) {
            logger.error("Problem adding Remote Keys Lookup Table ", ex);
        }
        try {
            DbCommand com = new DbCommand(DbStrings.createDeletedKeys);
            DbQuery db = new DbQuery();
            db.writeOxGuard(com);
            db.close();
        } catch (Exception ex) {
            logger.error("Problem adding deleted keys Table ", ex);
        }

    }

    private void addMissingDBKeys() {
        System.out.print("Adding database indexes for Guard tables...");
        try{
            DbCommand com = new DbCommand(alterFileCache);
            DbQuery db = new DbQuery();
            db.writeOxGuard(com);
            db.close();
        } catch (Exception e) {
            logger.error("Problem adding primary key to File Cache database: " + e.getMessage());
            System.out.println("unable to add database index to FileCache: " + e.getMessage());
        }

        try{
            DbCommand com = new DbCommand(alterSharding);
            DbQuery db = new DbQuery();
            db.writeOxGuard(com);
            db.close();
        } catch (Exception e) {
            logger.error("Problem adding primary key to Sharding database: " + e.getMessage());
            System.out.println("unable to add database index to Sharding: " + e.getMessage());
        }
        System.out.println("done adding keys");


    }

    public void listSchema() throws Exception {
    	ArrayList <Schema> schemas = getSchemas();
    	for (int i = 0; i < schemas.size(); i++) {
    		System.out.println(schemas.get(i).name + ": " + schemas.get(i).cid);
    	}
    }

    private ArrayList <Schema> getSchemas () throws Exception {
    	String command = "SELECT db_schema, cid from context_server2db_pool GROUP BY db_schema;";  // This is only legal in MySql.  We don't care what cid is returned
    	DbCommand com = new DbCommand (command);
    	DbQuery db = new DbQuery ();
    	db.readFromConfigDb(com);
    	ArrayList <Schema> schemas = new ArrayList <Schema> ();
    	while (db.next()) {
    		Schema scheme = new Schema(db.rs.getString("db_schema"), db.rs.getInt("cid"), 0);
    		schemas.add(scheme);
    	}
    	db.close();
    	command = "SELECT current_shard FROM sharding";
    	com = new DbCommand(command);
    	db = new DbQuery();
    	db.readOG(com);
    	if (db.next()) {
    		int shardmax = db.rs.getInt("current_shard");
    		for (int i = 1; i <= shardmax; i++ ) {
    			Schema scheme = new Schema("Guest shard " + i, -1, i);
    			schemas.add(scheme);
    		}
    	}
    	Schema scheme = new Schema("Master Key Table", 0, 0);
		schemas.add(scheme);
    	return (schemas);
    }

    public static boolean PGPTables () {
    	try {
	    	String command = "SELECT email, id, cid FROM og_email";
	    	DbCommand com = new DbCommand(command);
	    	DbQuery db = new DbQuery();
	    	db.readOG(com);
	    	Access acc = new Access();
	    	int total = db.getRowCount();
	    	int current = 0;
	    	while (db.next()) {
	    		System.out.print("Updating " + ++current + "/" + total +"      \r");
	    		String email = db.rs.getString("email");
	    		int id = db.rs.getInt("id");
	    		int cid = db.rs.getInt("cid");
		    	//	if (cid > 0) {
		    		try {
		    			GuardKeys key = acc.getKeys(id, cid);
		    			PGPPublicKeyRing pubring = key.pubring;
		    			Iterator<PGPPublicKey> it = pubring.getPublicKeys();
		    			long keyid = 0;
		    			command = "INSERT INTO PGPKeys VALUES (?, ?, ?, ?, 1, ?) ON DUPLICATE KEY UPDATE id = id";
		    			while (it.hasNext()) {
		    				PGPPublicKey pubkey = it.next();
		    				DbCommand com2 = new DbCommand(command);
		    				com2.addVariables(pubkey.getKeyID());
			    			com2.addVariables(email);
			    			if (keyid == 0) {
                                keyid = pubkey.getKeyID();
                            }
			    			String hex = Core.PGPKeyId(pubkey.getKeyID());
			    			com2.addVariables(hex);
			    			com2.addVariables(cid);
			    			com2.addVariables(keyid);
			    			DbQuery db2 = new DbQuery();
			    			db2.writeOxGuard(com2);

			    			db2.close();
		    			}
		    			String keycommand = "UPDATE og_KeyTable SET keyid = ? WHERE id = ? and cid = ?";
		    			DbCommand comkey = new DbCommand(keycommand);
		    			comkey.addVariables(keyid);
		    			comkey.addVariables(id);
		    			comkey.addVariables(cid);
		    			DbQuery dbkey = new DbQuery();
		    			dbkey.write(comkey, id, cid);
		    			dbkey.close();

		    		} catch (Exception ex) {
		    			logger.error("Unable to get PGP key for user " + email);
		    		}
	    	//	}
	    	}
	    	return (true);
    	} catch (Exception ex) {
    		logger.error("Unable to update PGP key table ", ex);
    		return (false);
    	}
    }

}
